# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import re

from django.db import models
from django.db.models import fields, TextField
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from common.utils.fields import RegExpField, TrimmedCharField
from common.utils.text import split_string

log = logging.getLogger(__name__)


class CoachSchema(models.Model):
    SEAT_SELECTION_ALGO_CHOICES = (
        ('old', _(u'Старая версия')),
        ('two-level', _(u'Версия от 02.2015 с выбором двух нижних'))
    )

    name = TrimmedCharField(_(u'Название'), max_length=100, null=False, blank=False, default=None)
    image = models.ImageField(
        _(u'Изображение'),
        upload_to='data/coachinfo/image', null=True, blank=True,
        height_field='height',
        width_field='width',
    )
    width = models.IntegerField(_(u'Ширина'), editable=False, null=True)
    height = models.IntegerField(_(u'Высота'), editable=False, null=True)
    place_selection_algorithm = models.CharField(_(u'Алгоритм выбора мест'), max_length=20, default='old',
                                                 choices=SEAT_SELECTION_ALGO_CHOICES, db_column='seat_selection_algo')
    _schema = models.TextField(_(u'Схема'), db_column='schema')
    _place_groups = models.TextField(_(u'Группы мест'), db_column='group_places')
    details = models.TextField(_(u'Дополнительная информация о вагоне и местах'), default='', null=False, blank=True)
    svg_schema = models.TextField(_(u'Размеченая схема вагона (содержание svg файла)'),
                                  default='', null=False, blank=True)
    two_storey = models.BooleanField(_(u'Двухэтажный вагон'), default=False)

    @cached_property
    def places(self):
        groups_strings = self._place_groups.split('\n')
        groups = [parse_integer_list(group_string) for group_string in groups_strings if group_string]
        group_numbers_by_places = {}
        for group_number, group in enumerate(groups, 1):
            for place_number in group:
                group_numbers_by_places[str(place_number)] = group_number

        schema_strings = self._schema.split('\n')

        places = []
        for place_string in schema_strings:
            place_components = parse_integer_list(place_string)
            if not place_components:
                continue
            place_number, left, top = place_components[:3]
            geometry = {'left': left, 'top': top}
            places.append(CoachPlace(place_number, geometry, group_numbers_by_places.get(str(place_number))))

        return places

    @cached_property
    def details_dict(self):
        details_json = self.details and self.details.strip()
        if not details_json:
            return {}

        try:
            details_dict = json.loads(details_json)
            if not isinstance(details_dict, dict):
                log.error('JSON CoachSchema.details не являятся словарем, id={}.'.format(self.id))
                return {}
            return details_dict
        except ValueError:
            log.error('Не удалось распарсить JSON CoachSchema.details, id={}.'.format(self.id))
            return {}

    @property
    def place_flags(self):
        return self.details_dict.get('placeFlags', {})

    @property
    def hide_place_numbers(self):
        return self.details_dict.get('hidePlaceNumbers', False)

    def __unicode__(self):
        return self.name

    class Meta:
        app_label = 'order'
        db_table = 'order_coachinfo'
        verbose_name = _(u'схема вагона')
        verbose_name_plural = _(u'схемы вагонов')


class CoachSchemaBinding(models.Model):
    """Привязка схемы к вагону"""
    CLASS_CHOICES = [
        (u'compartment', _(u'купе')),
        (u'suite', _(u'СВ')),
        (u'sitting', _(u'сидячий')),
        (u'platzkarte', _(u'плацкарт')),
        (u'soft', _(u'мягкий')),
        (u'common', _(u'общий')),
    ]

    klass = fields.CharField(_(u'Категория вагона'), choices=CLASS_CHOICES, blank=True, default='', max_length=100)
    service_class = RegExpField(_(u'Класс обслуживания'), blank=True, default='', max_length=100)
    international_service_class = RegExpField(_('Международный класс обслуживания'), default='', blank=True,
                                              max_length=100)
    coach_subtype_code = TrimmedCharField(_(u'Подтип вагона'), max_length=100, default='', blank=True)
    road = RegExpField(_(u'Дорога принадлежности вагона'), max_length=255, default='', blank=True)
    train_number = TextField(_(u'Номер поезда'), blank=True, default='', )
    coach_number = RegExpField(_(u'Номер вагона'), blank=True, default='', max_length=255)
    priority = fields.IntegerField(_(u'Приоритет'), null=False, blank=False, default=10)
    schema = models.ForeignKey(CoachSchema, verbose_name=_(u'схема'), db_column='info_id', null=False, blank=False)
    direction_confirmed = models.BooleanField(_(u'Направление движения подтверждено'), default=False)
    im_car_scheme_id = RegExpField(_('RailwayCarSchemeId из ИМ'), default='', blank=True, max_length=255)

    @cached_property
    def coach_subtype_codes(self):
        return split_string(self.coach_subtype_code)

    @cached_property
    def service_class_pattern(self):
        return u'^{}$'.format(re.sub(u'\*', u'.*', self.service_class))

    def __unicode__(self):
        return "%s %s %s %s" % (self.klass, self.train_number, self.coach_number, self.priority)

    class Meta:
        app_label = 'order'
        db_table = 'order_coachinfobinding'
        verbose_name = _(u'привязка схемы вагона')
        verbose_name_plural = _(u'привязки схем вагонов')


class CoachPlace(object):
    def __init__(self, number, geometry, group_number):
        super(CoachPlace, self).__init__()

        self.number = number
        self.geometry = geometry
        self.group_number = group_number


def parse_integer_list(integers_string):
    integers_string = integers_string.strip()
    if not integers_string:
        return []

    return [int(number) for number in integers_string.split(',')]
