# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re
from operator import attrgetter

from django.utils.encoding import force_text
from django.utils.functional import cached_property

from common.dynamic_settings.default import conf
from common.models.currency import Price
from common.utils.exceptions import SimpleUnicodeException
from common.utils.text import split_string
from travel.rasp.train_api.tariffs.train.base.coach_category_traits import CoachCategoryTraits
from travel.rasp.train_api.train_partners.base.train_details.lazy_bandit_cost_cache import LazyBanditCostCache
from travel.rasp.train_api.train_partners.base.train_details.coach_schemas import BestSchemaFinder
from travel.rasp.train_api.train_partners.base.train_details.price_rules import get_rules
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.adult_tariffs import ensure_adult_tariffs
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.child_tariffs import set_child_tariffs
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.fee import (
    set_tariffs_with_fee, set_tariffs_lazy, post_set_tariffs
)
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.non_refundable import set_non_refundable_tariffs
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.tariff_calculator import (
    CHILD_COEFF, get_base_tariff)
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost

log = logging.getLogger(__name__)


PLACE_RE = re.compile(r'^(А)?(\d+)([МЖСЦИ])?$', re.U)
PLACE_TYPE_CODES = {
    'М': 'male',
    'Ж': 'female',
    'С': 'mixed',
    'Ц': 'single',
}


class CurrentClassDetailsError(SimpleUnicodeException):
    pass


class BaseTrainDetails(object):
    station_from = None
    station_to = None
    departure = None
    arrival = None
    raw_train_name = None
    is_firm = None
    start_number = None
    schemas = []
    yandex_uid = None

    contract = None
    lazy_cost_cache = None

    def add_schemas(self, coaches):
        already_added_schema_ids = set()
        best_schema_finder = BestSchemaFinder(self.start_number)

        for coach in coaches:
            if not coach.need_schema():
                continue

            place_numbers = [place.number for place in coach.places]

            schema, schema_binding = best_schema_finder.find_best_schema(
                coach_type=coach.type,
                service_class_code=coach.service_class_code,
                coach_subtype_code=coach.coach_subtype_code,
                road=coach.road,
                coach_number=coach.number,
                two_storey=coach.two_storey,
                place_counts=coach.place_counts,
                place_numbers=place_numbers,
                international_service_class_code=getattr(coach, 'international_service_class_code', None),
                im_car_scheme_id=coach.im_scheme_id,
            )

            if schema:
                coach.set_schema(schema, schema_binding)
                if schema.id not in already_added_schema_ids:
                    already_added_schema_ids.add(schema.id)
                    self.schemas.append(schema)

    def add_tariffs(self, coaches, icookie, bandit_type, in_suburban_search, original_query):
        coaches_to_tariff = filter(lambda c: c.need_tariffs(), coaches)
        for coach in coaches_to_tariff:
            coach.set_tariffs_without_fee()
            coach.reservation_variants = coach.get_reservation_variants()

        if conf.TRAIN_PURCHASE_BANDIT_CHARGING and conf.TRAIN_PURCHASE_LAZY_BANDIT_CHARGING:
            self.lazy_cost_cache = LazyBanditCostCache(icookie, self.yandex_uid, self, bandit_type, in_suburban_search, original_query)
            for coach in coaches_to_tariff:
                set_tariffs_lazy(coach, self.lazy_cost_cache)
            self.lazy_cost_cache.materialize_ticket_costs()
            for coach in coaches_to_tariff:
                post_set_tariffs(coach, self.lazy_cost_cache)
        else:
            for coach in coaches_to_tariff:
                coach.set_tariffs(yandex_uid=self.yandex_uid)


class BaseClassDetails(object):
    def has_saleable_coaches(self):
        return not self.is_for_children

    def get_tariffs_info(self):
        raise NotImplementedError

    def _parse_category_traits(self):
        category_traits = CoachCategoryTraits.loads(self.raw_category_traits or '')
        self.can_choose_male_female_mix_places = category_traits.can_choose_male_female_mix_places
        self.places_without_numbers = category_traits.places_without_numbers
        self.is_firm = self.train_details.is_firm and not category_traits.is_not_firm
        self.pet_in_coach = category_traits.pet_in_coach
        self.pet_places_only = category_traits.pet_places_only
        self.is_for_children = category_traits.is_for_children

    @cached_property
    def service_tariff(self):
        raise NotImplementedError


class BaseCoachDetails(object):
    type = None
    current_class_details = None
    can_choose_bedding = False
    owner = None
    places = []
    price_without_places = None
    place_counts = {}
    price_rules = []
    child_coeff = CHILD_COEFF
    base_tariff = None
    schema_id = None
    direction_confirmed = False
    reservation_variants = None
    errors = set()
    has_non_refundable_tariff = False
    im_scheme_id = None

    def set_adult_tariffs_without_fee(self):
        raise NotImplementedError

    def get_place_counts(self):
        """
        :return: {
            'total': int,
            'lower_coupe': int,
            'upper_coupe': int,
            'lower_side': int,
            'upper_side': int
        }
        """
        raise NotImplementedError

    def set_tariffs_without_fee(self):
        self.set_adult_tariffs_without_fee()

        can_select_places = not self.current_class_details.places_without_numbers and self.schema_id

        if can_select_places:
            self.price_rules = get_rules(train_number=self.current_class_details.train_details.start_number,
                                         coach=self, query=self.current_class_details.train_details.query)
            self.base_tariff = get_base_tariff(self)

        ensure_adult_tariffs(self)
        set_child_tariffs(self)
        if self.has_non_refundable_tariff and conf.TRAIN_PURCHASE_ENABLE_NON_REFUNDABLE:
            set_non_refundable_tariffs(self)

    def iter_place_prices(self):
        for place in self.places:
            yield place
        if self.price_without_places:
            yield self.price_without_places

    def set_tariffs(self, yandex_uid=None):
        # TODO (syukhno): вынести вычисления из парсера
        if not self.places and not self.price_without_places:
            return
        set_tariffs_with_fee(
            contract=self.current_class_details.train_details.contract,
            coach=self, yandex_uid=yandex_uid,
            func_calculate_ticket_cost=calculate_ticket_cost,
        )
        self.adult_tariff = max([place.adult_tariff for place in self.iter_place_prices()])
        self.bedding_tariff_with_fee = max([place.bedding_tariff_with_fee for place in self.iter_place_prices()])
        self.set_reservation_variants_amount(yandex_uid)

    def set_schema(self, schema, schema_binding):
        self.schema_id = schema.id
        self.direction_confirmed = schema_binding.ConfirmedDirection if conf.TRAIN_BACKEND_USE_PROTOBUFS['schemas'] else \
                                   schema_binding.direction_confirmed

    def need_schema(self):
        return not self.current_class_details.places_without_numbers and self.places

    def need_tariffs(self):
        return not self.current_class_details.places_without_numbers and (
            self.places or self.place_counts.get('total'))

    def get_reservation_variants(self):
        return None

    def set_reservation_variants_amount(self, yandex_uid):
        return None

    @cached_property
    def place_numbers_set(self):
        return set(place.number for place in self.places)


class PlaceNumberError(SimpleUnicodeException):
    pass


class PlaceDetails(object):
    def __init__(self, number, gender=None,
                 original_adult_tariff=None,
                 original_child_tariff=None,
                 adult_tariff=None,
                 child_tariff=None,
                 bedding_tariff_with_fee=None,
                 yandex_fee_percent=None,
                 original_child_non_refundable_tariff=None,
                 original_adult_non_refundable_tariff=None,
                 adult_non_refundable_tariff=None,
                 child_non_refundable_tariff=None,
                 is_bandit_fee_applied=False,
                 bandit_type=None,
                 bandit_version=None):
        self.number = number
        self.gender = gender
        self.original_adult_tariff = original_adult_tariff
        self.original_child_tariff = original_child_tariff
        self.adult_tariff = adult_tariff
        self.child_tariff = child_tariff
        self.bedding_tariff_with_fee = bedding_tariff_with_fee
        self.yandex_fee_percent = yandex_fee_percent
        self.original_child_non_refundable_tariff = original_child_non_refundable_tariff
        self.original_adult_non_refundable_tariff = original_adult_non_refundable_tariff
        self.adult_non_refundable_tariff = adult_non_refundable_tariff
        self.child_non_refundable_tariff = child_non_refundable_tariff
        self.is_bandit_fee_applied = is_bandit_fee_applied
        self.bandit_type = bandit_type
        self.bandit_version = bandit_version

    def ensure_adult_tariff(self, original_adult_tariff):
        if not self.original_adult_tariff:
            self.original_adult_tariff = original_adult_tariff

    def ensure_child_tariff(self, original_child_tariff):
        if not self.original_child_tariff:
            self.original_child_tariff = original_child_tariff

    def set_non_refundable_tariffs(self, original_adult_non_refundable_tariff, original_child_non_refundable_tariff):
        self.original_adult_non_refundable_tariff = original_adult_non_refundable_tariff
        self.original_child_non_refundable_tariff = original_child_non_refundable_tariff

    def set_tariffs_with_fee(
        self, adult_tariff_with_fee, child_tariff_with_fee,
        bedding_tariff_with_fee, yandex_fee_percent,
        is_bandit_fee_applied=False,
        bandit_type=None, bandit_version=None,
        adult_non_refundable_tariff=None, child_non_refundable_tariff=None,
    ):
        if adult_non_refundable_tariff is not None:
            adult_non_refundable_tariff = Price(adult_non_refundable_tariff)
        if child_non_refundable_tariff is not None:
            child_non_refundable_tariff = Price(child_non_refundable_tariff)
        self.adult_tariff = Price(adult_tariff_with_fee)
        self.child_tariff = Price(child_tariff_with_fee)
        self.bedding_tariff_with_fee = Price(bedding_tariff_with_fee)
        self.yandex_fee_percent = yandex_fee_percent
        self.is_bandit_fee_applied = is_bandit_fee_applied
        self.bandit_type = bandit_type
        self.bandit_version = bandit_version
        self.adult_non_refundable_tariff = adult_non_refundable_tariff
        self.child_non_refundable_tariff = child_non_refundable_tariff


def init_places(numbers_string, coach_type):
    places = {}
    for raw_place in split_string(numbers_string):
        try:
            number, gender = parse_place_number(raw_place, coach_type=coach_type)
        except PlaceNumberError as error:
            log.error(force_text(error))
        else:
            places[number] = PlaceDetails(number, gender)
    return sorted(places.values(), key=attrgetter('number'))


def parse_place_number(raw_place, coach_type):
    match = PLACE_RE.match(raw_place)
    if match is None:
        raise PlaceNumberError(u'Неизвестный формат номера места {!r}'.format(raw_place))
    extra, number, type_code = match.groups()
    if type_code == u'И':
        raise PlaceNumberError(u'Места для инвалидов мы продавать [пока] не можем.')
    if extra and coach_type != 'common':
        raise PlaceNumberError(u'Дополнительное место не в общем вагоне')
    gender = PLACE_TYPE_CODES.get(type_code)
    return int(number), gender
