# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from operator import attrgetter

from django.db.models import Q

from common.apps.train.models import PlacePriceRules
from common.models.geo import Country
from travel.rasp.library.python.common23.date import environment


def is_base_rule(rule):
    """
    :type rule: PlacePriceRules
    """
    return rule.price_percent == 100


def get_rules(train_number, coach, query):
    station_from = query.station_from
    station_to = query.station_to
    when = query.when

    if (
        coach.owner == 'ФПК' and
        (station_from.country_id != Country.RUSSIA_ID or station_to.country_id != Country.RUSSIA_ID)
    ):
        # Костыль. Нужно занести его в правила тарификации.
        return []

    now = environment.now_aware().astimezone(station_from.pytz).replace(tzinfo=None)
    days_before_departure = (when - now).days

    query = (Q(coach_type=coach.type) &
             (Q(departure_period_begin=None) | Q(departure_period_begin__lte=when)) &
             (Q(departure_period_end=None) | Q(departure_period_end__gte=when)) &
             (Q(sale_period_begin=None) | Q(sale_period_begin__gte=days_before_departure)) &
             (Q(sale_period_end=None) | Q(sale_period_end__lte=days_before_departure)))

    # Сначла выбираем все правила актуальные на дату отправления, отфильтрованные по типу вагона.
    rules = list(PlacePriceRules.objects.filter(query))

    train_number_rules = [rule for rule in rules if rule.check_train_number(train_number)]
    if train_number_rules:
        # Выбираем правила, подходящие по номеру поезда
        rules = train_number_rules

        coach_number_rules = [rule for rule in rules if rule.check_coach_number(coach.number)]
        if coach_number_rules:
            # Выбираем правила, подходящие по номеру вагона
            rules = coach_number_rules
        else:
            # Выеркиваем правила со специфичными номерами вагонов
            rules = [rule for rule in rules if not rule.coach_numbers]
    else:
        # Вычеркиваем правила со специфичными номерами поездов
        rules = [rule for rule in rules if not rule.train_number]

        owner_rules = [rule for rule in rules if rule.check_owner(coach.owner)]
        if owner_rules:
            # Выбираем правила, подходящие по владельцу вагона
            rules = owner_rules
        else:
            # Вычеркиваем правила со специфичными владельцами вагонов
            rules = [rule for rule in rules if not rule.owners]

    if rules:
        rules = ensure_base_rule(rules, coach.type, coach.place_numbers_set)

    rules = [rule for rule in rules if rule.place_numbers_set & coach.place_numbers_set]
    return rules


def ensure_base_rule(rules, coach_type, place_numbers):
    can_apply_for_child = rules[0].can_apply_for_child if len(rules) > 0 else True

    not_base_rules = [rule for rule in rules if not is_base_rule(rule)]
    not_base_places_numbers = set.union(*([rule.place_numbers_set for rule in not_base_rules] or [set()]))

    base_rule_place_numbers = place_numbers - not_base_places_numbers
    base_rule = PlacePriceRules(coach_type=coach_type,
                                place_numbers=','.join(str(place_number) for place_number in base_rule_place_numbers),
                                price_percent=100,
                                can_apply_for_child=can_apply_for_child)

    return sorted(not_base_rules + [base_rule], key=attrgetter('price_percent'), reverse=True)
