# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from marshmallow import Schema, fields, ValidationError
from marshmallow.decorators import post_load
from marshmallow_enum import EnumField

from common.models.geo import Station
from common.serialization.common_schemas import PriceSchema
from common.serialization.fields import DatetimeAwareField
from travel.rasp.train_api.serialization.experiment import ExperimentQuerySchema
from travel.rasp.train_api.serialization.fields import StationField, validate_station_express_code, CallableStringField
from travel.rasp.train_api.tariffs.train.base.models import PlaceReservationType
from travel.rasp.train_api.train_partners.base.train_details import TrainDetailsQuery
from travel.rasp.train_api.train_purchase.core.models import TrainPartner

log = logging.getLogger(__name__)


class TrainDetailsQuerySchema(ExperimentQuerySchema):
    partner = EnumField(TrainPartner, by_value=True,
                        required=True, missing=TrainPartner.UFS)
    station_from = StationField(required=True, load_from='stationFrom', validate=validate_station_express_code)
    station_to = StationField(required=True, load_from='stationTo', validate=validate_station_express_code)
    when = fields.DateTime()
    number = fields.String(required=True)
    include_price_fee = fields.Boolean(load_from='includePriceFee', missing=False)
    price_exp_id = fields.String(load_from='priceExpId', missing=None)
    yandex_uid = fields.String(missing=None)
    provider = fields.String(missing=None)
    mock_im_path = fields.String(load_from='mockImPath', missing=None)
    mock_im_auto = fields.Boolean(load_from='mockImAuto', missing=False)

    @post_load
    def build_query(self, data):
        from travel.rasp.train_api.train_partners.ufs.train_details.receiver import UfsTrainDetailsQuery

        if data['partner'] == TrainPartner.UFS:
            return UfsTrainDetailsQuery(
                data['partner'], data['station_from'], data['station_to'],
                when=data['when'], number=data['number'], include_price_fee=data['include_price_fee'],
            )
        else:
            return TrainDetailsQuery(
                data['partner'], data['station_from'], data['station_to'], when=data['when'], number=data['number'],
                include_price_fee=data['include_price_fee'], provider=data['provider'],
                yandex_uid=data['yandex_uid'] if data['price_exp_id'] else None,
                raw_query_data=data, mock_im_path=data['mock_im_path'], mock_im_auto=data['mock_im_auto']
            )


class InternalTrainDetailsQuerySchema(ExperimentQuerySchema):
    station_from = fields.Method(deserialize='load_station_by_express_code', required=True, load_from='stationFrom')
    station_to = fields.Method(deserialize='load_station_by_express_code', required=True, load_from='stationTo')
    when = fields.DateTime()
    number = fields.String(required=True)

    @post_load
    def build_query(self, data):
        return TrainDetailsQuery(
            TrainPartner.IM, data['station_from'], data['station_to'], when=data['when'], number=data['number'],
            include_price_fee=True, provider='P1', yandex_uid=None, raw_query_data=data,
        )

    def load_station_by_express_code(self, value):
        try:
            station = Station.get_by_code('express', value)
            return station
        except Exception:
            raise ValidationError('No station with express code')


class PlacePriceSchema(Schema):
    adult_tariff = fields.Nested(PriceSchema, dump_to='adultTariff')
    child_tariff = fields.Nested(PriceSchema, dump_to='childTariff')
    adult_non_refundable_tariff = fields.Nested(PriceSchema, dump_to='adultNonRefundableTariff')
    child_non_refundable_tariff = fields.Nested(PriceSchema, dump_to='childNonRefundableTariff')
    bedding_tariff_with_fee = fields.Nested(PriceSchema, dump_to='beddingTariffWithFee')


class PlaceDetailsSchema(PlacePriceSchema):
    number = fields.Integer()
    gender = fields.String()


class PlaceCountsSchema(Schema):
    total = fields.Integer()
    lower_coupe = fields.Integer(dump_to='lowerCoupe')
    upper_coupe = fields.Integer(dump_to='upperCoupe')
    lower_side = fields.Integer(dump_to='lowerSide')
    upper_side = fields.Integer(dump_to='upperSide')


class TariffInfoSchema(Schema):
    code = fields.String()
    title = CallableStringField(attribute='L_title')
    min_age = fields.Integer(dump_to='minAge')
    min_age_includes_birthday = fields.Boolean(dump_to='minAgeIncludesBirthday')
    max_age = fields.Integer(dump_to='maxAge')
    max_age_includes_birthday = fields.Boolean(dump_to='maxAgeIncludesBirthday')
    need_document = fields.Boolean(dump_to='needDocument')
    without_place = fields.Boolean(dump_to='withoutPlace')


class TariffTypeSchema(Schema):
    code = fields.String()
    provider_code = fields.String(dump_to='providerCode')
    discount = fields.Float()
    validators = fields.String()


class ServiceClassSchema(Schema):
    key = fields.String()
    code = fields.String()
    title = fields.String()
    description = fields.String()
    international_code = fields.String(dump_to='internationalCode')


class ReservationVariantSchema(Schema):
    amount = fields.Float()
    non_refundable_amount = fields.Float(dump_to='nonRefundableAmount')
    places_count = fields.Integer(dump_to='placesCount')
    passengers = fields.Dict()
    give_child_without_place = fields.Boolean(dump_to='giveChildWithoutPlace')


class CoachDetailsSchema(Schema):
    # train_start_number - номер поезда, с которым он отправляется с начальной станции маршрута
    train_start_number = fields.String(dump_to='trainStartNumber')
    number = fields.String()
    type = fields.String()
    has_dynamic_pricing = fields.Boolean(dump_to='hasDynamicPricing')
    service_class = fields.Nested(ServiceClassSchema, dump_to='serviceClass')
    electronic_ticket = fields.Boolean(dump_to='electronicTicket')
    two_storey = fields.Boolean(dump_to='twoStorey')
    bedding_tariff_with_fee = fields.Nested(PriceSchema, dump_to='beddingTariff')
    can_choose_bedding = fields.Boolean(dump_to='canChooseBedding')
    facilities = fields.List(fields.String, many=True)
    places = fields.Nested(PlaceDetailsSchema, many=True)
    price_without_places = fields.Nested(PlacePriceSchema, dump_to='priceWithoutPlaces')
    schema_id = fields.Integer(dump_to='schemaId')
    coach_subtype_code = fields.String(dump_to='coachSubtypeCode')
    place_counts = fields.Nested(PlaceCountsSchema, dump_to='placeCounts')
    places_by_type = fields.Dict(dump_to='placesByType')
    places_by_compartment = fields.Dict(dump_to='placesByCompartment')
    adult_tariff = fields.Nested(PriceSchema, dump_to='adultTariff')
    owner = fields.String()
    pet_in_coach = fields.Boolean(dump_to='petInCoach')
    pets_allowed = fields.Boolean(dump_to='petsAllowed')
    pets_segregated = fields.Boolean(dump_to='petsSegregated')
    is_firm = fields.Boolean(dump_to='isFirm')
    loyalty_cards = fields.List(fields.String(), dump_to='loyaltyCards')
    tariffs_info = fields.Nested(TariffInfoSchema, many=True, dump_to='tariffsInfo')
    tariff_types = fields.Nested(TariffTypeSchema, many=True, dump_to='tariffTypes')
    through_arrival = DatetimeAwareField(dump_to='throughArrival')
    arrival = DatetimeAwareField(dump_to='arrival')
    direction_confirmed = fields.Boolean(dump_to='directionConfirmed')
    place_reservation_type = EnumField(PlaceReservationType, by_value=True, dump_to='placeReservationType')
    reservation_variants = fields.Nested(ReservationVariantSchema, many=True, dump_to='reservationVariants')
    is_special_sale_mode = fields.Boolean(dump_to='isSpecialSaleMode')
    has_non_refundable_tariff = fields.Boolean(dump_to='hasNonRefundableTariff')
    place_requirements_available = fields.Boolean(dump_to='placeRequirementsAvailable')
    fee_calculation_token = fields.String(dump_to='feeCalculationToken')


class CoachPlaceGeometrySchema(Schema):
    left = fields.Integer()
    top = fields.Integer()


class CoachPlaceSchema(Schema):
    number = fields.Integer()
    geometry = fields.Nested(CoachPlaceGeometrySchema)
    group_number = fields.Integer(dump_to='groupNumber')


class CoachSchemaSchema(Schema):
    id = fields.Integer()
    url = fields.String(attribute='image.url')
    width = fields.Integer()
    height = fields.Integer()
    place_selection_algorithm = fields.String(dump_to='placeSelectionAlgorithm')
    places = fields.Nested(CoachPlaceSchema, many=True)
    place_flags = fields.Dict(dump_to='placeFlags')
    hide_place_numbers = fields.Boolean(dump_to='hidePlaceNumbers')
    svg_schema = fields.String(dump_to='svg')


class SettlementSchema(Schema):
    title = fields.String()
    title_locative = fields.String(dump_to='titleLocative')
    preposition = fields.String()


class StationSchema(Schema):
    id = fields.Integer()
    express_code = fields.String(dump_to='expressCode')
    title = fields.String()
    settlement = fields.Nested(SettlementSchema)


class TrainDetailsSchema(Schema):
    raw_train_category = fields.String(dump_to='rawTrainCategory')
    raw_train_name = fields.String(dump_to='rawTrainName')
    electronic_ticket = fields.Boolean(dump_to='electronicTicket')
    # ticket_number - номер поезда, с которым он отправляется со станции отправления пассажира
    # (этот номер прописывается в бланке билета)
    ticket_number = fields.String(dump_to='ticketNumber')
    # start_number - номер поезда, с которым он отправляется с начальной станции маршрута
    start_number = fields.String(dump_to='startNumber')
    coaches = fields.Nested(CoachDetailsSchema, many=True)
    schemas = fields.Nested(CoachSchemaSchema, many=True)
    station_from = fields.Method('build_station_from', dump_to='stationFrom')
    station_to = fields.Method('build_station_to', dump_to='stationTo')
    is_firm = fields.Boolean(dump_to='isFirm')
    allowed_document_types = fields.List(fields.String(), dump_to='allowedDocumentTypes')
    departure = DatetimeAwareField()
    arrival = DatetimeAwareField()
    is_suburban = fields.Boolean(dump_to='isSuburban')
    im_initial_station_name = fields.String(dump_to='imInitialStationName')
    im_final_station_name = fields.String(dump_to='imFinalStationName')
    route_policy = fields.String(dump_to='routePolicy')
    is_cppk = fields.Boolean(dump_to='isCppk')
    is_covid_certificate_required = fields.Boolean(dump_to='isCovidCertificateRequired')
    provider = fields.String()
    tariff_categories = fields.Nested(TariffInfoSchema, many=True, dump_to='tariffCategories')

    def _build_station(self, station, express_code):
        station_data = {
            'id': station.id,
            'title': station.L_title(),
            'express_code': express_code
        }
        if station.settlement:
            station_data['settlement'] = {
                'title': station.settlement.L_title(),
                'title_locative': station.settlement.L_title(case='locative'),
                'preposition': station.settlement.L_title(case='preposition_v_vo_na'),
            }
        return StationSchema().dump(station_data).data

    def build_station_from(self, train_details):
        return self._build_station(train_details.station_from, train_details.express_from)

    def build_station_to(self, train_details):
        return self._build_station(train_details.station_to, train_details.express_to)
