# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple
from django.conf import settings
from django.db.models import Q
from six import ensure_text

from common.apps.train.models import ServiceClass
from common.dynamic_settings.default import conf
from common.settings.utils import define_setting, bool_converter
from common.utils.text import split_string
from travel.proto.dicts.trains.coach_type_pb2 import TCoachType
from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER
from travel.rasp.train_api.train_partners.base.train_details.utils import check_train_number

log = logging.getLogger(__name__)

define_setting('TRAIN_SERVICE_CLASS_FROM_PARTNER', default=False, converter=bool_converter)

ServiceClassLight = namedtuple(
    'ServiceClass',
    ['key', 'code', 'title', 'description', 'international_code']
)


def filter_classes(classes, split_condition, match_condition):
    filtered_classes = [service_class for service_class in classes
                        if split_condition(service_class) and match_condition(service_class)]

    if filtered_classes:
        return filtered_classes

    return [service_class for service_class in classes if not split_condition(service_class)]


def get_service_class(service_class_details, two_storey):
    get_method = _get_service_class_with_protobufs if conf.TRAIN_BACKEND_USE_PROTOBUFS['service_class'] else _get_service_class
    return get_method(service_class_details, two_storey)


def _get_service_class(service_class_details, two_storey):
    train_details = service_class_details.train_details
    when = train_details.when
    code = service_class_details.service_class_code
    partner_class = ServiceClassLight(
        key=code,
        code=code,
        title=None,
        description=service_class_details.service_class_description,
        international_code=service_class_details.international_service_class_code,
    )

    query = (Q(coach_category=service_class_details.coach_type) &
             (Q(start_date=None) | Q(start_date__lte=when)) &
             (Q(end_date=None) | Q(end_date__gte=when)))
    classes = list(ServiceClass.objects.filter(query))

    classes = filter_classes(classes,
                             lambda klass: bool(klass.codes),
                             lambda klass: code in klass.codes)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.coach_owners),
                             lambda klass: service_class_details.owner.upper() in klass.coach_owners)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.train_number),
                             lambda klass: check_train_number(klass.train_number, train_details.start_number))

    classes = filter_classes(classes,
                             lambda klass: bool(klass.brand_title),
                             lambda klass: klass.brand_title.upper() == train_details.brand)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.two_storey is not None),
                             lambda klass: klass.two_storey == two_storey)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.is_firm_coach is not None),
                             lambda klass: klass.is_firm_coach == service_class_details.is_firm)

    if not classes:
        log.warning(
            'Не найден класс обслуживания для поезда {} №{}, код класса {}, {} вагон на дату {}.'
            .format(service_class_details.owner,
                    train_details.start_number,
                    service_class_details.service_class_code,
                    'фирменный' if service_class_details.is_firm else 'нефирменный',
                    when))
        return partner_class

    if len(classes) > 1:
        log.error(
            'Не определен однозначно класс обслуживания для поезда {} №{}, код класса {}, {} вагон на дату {}. '
            'Классы-кандидаты: {}'.format(
                service_class_details.owner,
                train_details.start_number,
                service_class_details.service_class_code,
                'фирменный' if service_class_details.is_firm else 'нефирменный',
                when,
                [service_class.id for service_class in classes])
        )
        return None

    service_class = classes[0]
    description = service_class_details.service_class_description
    if service_class.L_description() and not settings.TRAIN_SERVICE_CLASS_FROM_PARTNER:
        description = service_class.L_description()

    return ServiceClassLight(
        key=', '.join(service_class.codes),
        code=code,
        title=service_class.L_title(),
        description=description,
        international_code=service_class_details.international_service_class_code,
    )


def _get_service_class_with_protobufs(service_class_details, two_storey):
    train_details = service_class_details.train_details
    when = train_details.when
    code = service_class_details.service_class_code
    partner_class = ServiceClassLight(
        key=code,
        code=code,
        title=None,
        description=service_class_details.service_class_description,
        international_code=service_class_details.international_service_class_code,
    )

    query = lambda sc: (TCoachType.EType.Name(sc.CoachCategory) == service_class_details.coach_type.upper()
                        and (not sc.HasField('StartDate') or sc.StartDate.ToDatetime() <= when)
                        and (not sc.HasField('EndDate') or sc.EndDate.ToDatetime() >= when))
    classes = filter(query, PROTOBUF_DATA_PROVIDER.service_class_repo.values())

    classes = filter_classes(classes,
                             lambda klass: bool(klass.Code.strip()),
                             lambda klass: code in split_string(ensure_text(klass.Code)))

    classes = filter_classes(classes,
                             lambda klass: bool(klass.CoachOwner.strip()),
                             lambda klass: service_class_details.owner.upper() in split_string(ensure_text(klass.CoachOwner).upper()))

    classes = filter_classes(classes,
                             lambda klass: bool(klass.TrainNumber.strip()),
                             lambda klass: check_train_number(ensure_text(klass.TrainNumber), train_details.start_number))

    classes = filter_classes(classes,
                             lambda klass: bool(klass.BrandTitle.strip()),
                             lambda klass: ensure_text(klass.BrandTitle).upper() == train_details.brand)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.HasField('TwoStorey')),
                             lambda klass: klass.TwoStorey.value == two_storey)

    classes = filter_classes(classes,
                             lambda klass: bool(klass.HasField('IsFirmCoach')),
                             lambda klass: klass.IsFirmCoach.value == service_class_details.is_firm)

    if not classes:
        log.warning(
            'Не найден класс обслуживания для поезда {} №{}, код класса {}, {} вагон на дату {}.'
            .format(service_class_details.owner,
                    train_details.start_number,
                    service_class_details.service_class_code,
                    'фирменный' if service_class_details.is_firm else 'нефирменный',
                    when))
        return partner_class

    if len(classes) > 1:
        log.error(
            'Не определен однозначно класс обслуживания для поезда {} №{}, код класса {}, {} вагон на дату {}. '
            'Классы-кандидаты: {}'.format(
                service_class_details.owner,
                train_details.start_number,
                service_class_details.service_class_code,
                'фирменный' if service_class_details.is_firm else 'нефирменный',
                when,
                [ensure_text(service_class.Name) for service_class in classes])
        )
        return None

    service_class = classes[0]
    description = service_class_details.service_class_description
    if service_class.Description and not settings.TRAIN_SERVICE_CLASS_FROM_PARTNER:
        description = ensure_text(service_class.Description)

    return ServiceClassLight(
        key=ensure_text(service_class.Code),
        code=code,
        title=ensure_text(service_class.Title),
        description=description,
        international_code=service_class_details.international_service_class_code,
    )
