# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from decimal import Decimal

from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.tariff_calculator import calc_tariff_by_coeff


def ensure_adult_tariffs(coach):
    if _check_adult_tariffs_already_set(coach.iter_place_prices()):
        return
    if not coach.price_rules or coach.min_tariff == coach.max_tariff:
        for place in coach.iter_place_prices():
            place.ensure_adult_tariff(coach.max_tariff)
        return
    _set_adult_tariffs_by_rules(coach)


def _check_adult_tariffs_already_set(places):
    return all([place.original_adult_tariff for place in places])


def _set_adult_tariffs_by_rules(coach):
    place_numbers = coach.place_numbers_set
    max_tariff_used = False

    for rule in coach.price_rules:
        # place_numbers_of_current_rule - номера мест, соответствующих текущему правилу тарификации
        place_numbers_of_current_rule = place_numbers & rule.place_numbers_set
        if not place_numbers_of_current_rule:
            continue

        rule_places = [place for place in coach.iter_place_prices() if place.number in place_numbers_of_current_rule]

        # other_place_numbers - номера мест, соответствующих другим правилам тарификации
        other_place_numbers = place_numbers - place_numbers_of_current_rule

        if not max_tariff_used:
            # Текущее правило тарификации соответствует самым дорогим свободным местам в вагоне -
            # берем максимальный тариф
            adult_tariff = coach.max_tariff
            max_tariff_used = True
        elif not other_place_numbers:
            # Текущее правило тарификации соответствует самым дешевым свободным местам в вагоне -
            # берем минимальный тариф
            adult_tariff = coach.min_tariff
        else:
            # Места с одним из средних тарифов - вычисляем тариф, используя базовый тариф и процент.
            adult_tariff = _calc_tariff(rule, coach.base_tariff, coach.service_tariff)

        for place in rule_places:
            place.ensure_adult_tariff(adult_tariff)

        # К следующему правилу тарификации переходим, используя оставшиеся места
        place_numbers = other_place_numbers
        if not place_numbers:
            break


def _calc_tariff(rule, base_tariff, service_tariff):
    """
    Вычисляет тариф со скидкой по базовому (100%) тарифу и правилу тарификации.
    """
    coeff = Decimal(rule.price_percent / 100)
    return calc_tariff_by_coeff(coeff, base_tariff, service_tariff)
