# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import base64

from common.models.currency import Price
from travel.rasp.train_api.tariffs.train.base.models import PlaceReservationType
from travel.rasp.train_api.train_bandit_api.client import BanditClient
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost
from travel.rasp.train_bandit_api.proto import api_pb2

log = logging.getLogger(__name__)

SOFT_RESERVATION_TYPES = (PlaceReservationType.TWO_PLACES_AT_ONCE, PlaceReservationType.FOUR_PLACES_AT_ONCE)


def set_tariffs_with_fee(
    contract, coach, yandex_uid=None,
    func_calculate_ticket_cost=calculate_ticket_cost
):
    for place in coach.iter_place_prices():
        adult_tariff = (
            coach.min_tariff if coach.place_reservation_type in SOFT_RESERVATION_TYPES else place.original_adult_tariff
        )
        adult_ticket_cost = func_calculate_ticket_cost(
            contract, coach.type, adult_tariff, coach.service_tariff,
            yandex_uid=yandex_uid,
        )
        child_ticket_cost = func_calculate_ticket_cost(
            contract, coach.type, place.original_child_tariff, coach.service_tariff,
            yandex_uid=yandex_uid,
        )
        adult_non_refundable_tariff = child_non_refundable_tariff = None
        if place.original_adult_non_refundable_tariff:
            adult_non_refundable_tariff = func_calculate_ticket_cost(
                contract, coach.type, place.original_adult_non_refundable_tariff, coach.service_tariff,
                yandex_uid=yandex_uid,
            )
        if place.original_child_non_refundable_tariff:
            child_non_refundable_tariff = func_calculate_ticket_cost(
                contract, coach.type, place.original_child_non_refundable_tariff, coach.service_tariff,
                yandex_uid=yandex_uid,
            )
        place.set_tariffs_with_fee(
            adult_tariff_with_fee=_result_tariff(adult_ticket_cost, coach.can_choose_bedding),
            child_tariff_with_fee=_result_tariff(child_ticket_cost, coach.can_choose_bedding),
            bedding_tariff_with_fee=adult_ticket_cost.full_bedding_amount,
            yandex_fee_percent=adult_ticket_cost.yandex_fee_percent,
            is_bandit_fee_applied=adult_ticket_cost.is_bandit_fee_applied,
            bandit_type=adult_ticket_cost.bandit_type,
            bandit_version=adult_ticket_cost.bandit_version,
            adult_non_refundable_tariff=_result_tariff(adult_non_refundable_tariff, coach.can_choose_bedding),
            child_non_refundable_tariff=_result_tariff(child_non_refundable_tariff, coach.can_choose_bedding),
        )


def _result_tariff(ticket_cost, can_choose_bedding):
    if ticket_cost is None:
        return None
    return ticket_cost.full_amount - ticket_cost.full_bedding_amount if can_choose_bedding else ticket_cost.full_amount


def _create_adult_ticket_cost_consumer(coach, place):
    def consumer(adult_ticket_cost):
        place.adult_tariff = Price(_result_tariff(adult_ticket_cost, coach.can_choose_bedding))
        place.bedding_tariff_with_fee = Price(adult_ticket_cost.full_bedding_amount)
        place.yandex_fee_percent = adult_ticket_cost.yandex_fee_percent
        place.is_bandit_fee_applied = adult_ticket_cost.is_bandit_fee_applied
        place.bandit_type = adult_ticket_cost.bandit_type
        place.bandit_version = adult_ticket_cost.bandit_version
    return consumer


def _create_coach_adult_tariff_consumer(coach, lazy_cost_cache):
    def consumer(adult_ticket_cost):
        coach.adult_tariff = Price(_result_tariff(adult_ticket_cost, coach.can_choose_bedding))
        coach.bedding_tariff_with_fee = Price(adult_ticket_cost.full_bedding_amount)
        context = BanditClient.to_proto_context(lazy_cost_cache.create_context(coach.type))
        fee_calculation_token = api_pb2.TFeeCalculationToken(
            Context=context,
            Permille=int(adult_ticket_cost.yandex_fee_percent * 1000),
            IsBanditFeeApplied=adult_ticket_cost.is_bandit_fee_applied,
            RequestedBanditType=lazy_cost_cache.bandit_type,
            ActualBanditType=adult_ticket_cost.bandit_type,
            ActualBanditVersion=adult_ticket_cost.bandit_version,
        )
        coach.fee_calculation_token = base64.b64encode(fee_calculation_token.SerializeToString())
    return consumer


def _create_child_ticket_cost_consumer(coach, place):
    def consumer(child_ticket_cost):
        place.child_tariff = Price(_result_tariff(child_ticket_cost, coach.can_choose_bedding))
    return consumer


def _create_adult_non_refundable_tariff_consumer(coach, place):
    def consumer(adult_non_refundable_tariff):
        place.adult_non_refundable_tariff = Price(_result_tariff(adult_non_refundable_tariff, coach.can_choose_bedding))
    return consumer


def _create_child_non_refundable_tariff_consumer(coach, place):
    def consumer(child_non_refundable_tariff):
        place.child_non_refundable_tariff = Price(_result_tariff(child_non_refundable_tariff, coach.can_choose_bedding))
    return consumer


def set_tariffs_lazy(coach, lazy_cost_cache):
    log.debug(len(coach.places))
    if not coach.places:
        lazy_cost_cache.calculate_ticket_cost_lazy(
            coach.type, coach.service_class_code,
            coach.min_tariff if coach.place_reservation_type in SOFT_RESERVATION_TYPES else coach.max_tariff,
            coach.service_tariff,
            _create_coach_adult_tariff_consumer(coach, lazy_cost_cache)
        )
    for place in coach.iter_place_prices():
        adult_tariff = (
            coach.min_tariff if coach.place_reservation_type in SOFT_RESERVATION_TYPES else place.original_adult_tariff
        )
        lazy_cost_cache.calculate_ticket_cost_lazy(
            coach.type, coach.service_class_code, adult_tariff, coach.service_tariff,
            _create_adult_ticket_cost_consumer(coach, place)
        )
        lazy_cost_cache.calculate_ticket_cost_lazy(
            coach.type, coach.service_class_code, place.original_child_tariff, coach.service_tariff,
            _create_child_ticket_cost_consumer(coach, place)
        )
        if place.original_adult_non_refundable_tariff:
            lazy_cost_cache.calculate_ticket_cost_lazy(
                coach.type, coach.service_class_code, place.original_adult_non_refundable_tariff, coach.service_tariff,
                _create_adult_non_refundable_tariff_consumer(coach, place)
            )
        if place.original_child_non_refundable_tariff:
            lazy_cost_cache.calculate_ticket_cost_lazy(
                coach.type, coach.service_class_code, place.original_child_non_refundable_tariff, coach.service_tariff,
                _create_child_non_refundable_tariff_consumer(coach, place)
            )
    set_reservation_variants_amount_lazy(coach, lazy_cost_cache)


def post_set_tariffs(coach, lazy_cost_cache):
    can_select_places = not coach.current_class_details.places_without_numbers and coach.schema_id
    places = [p for p in coach.iter_place_prices()]
    if places:
        max_tariff_place = max(places, key=lambda p: p.adult_tariff)
        if not can_select_places:
            coach.adult_tariff = max_tariff_place.adult_tariff
        coach.bedding_tariff_with_fee = max_tariff_place.bedding_tariff_with_fee
        context = BanditClient.to_proto_context(lazy_cost_cache.create_context(coach.type))
        fee_calculation_token = api_pb2.TFeeCalculationToken(
            Context=context,
            Permille=int(max_tariff_place.yandex_fee_percent * 1000),
            IsBanditFeeApplied=max_tariff_place.is_bandit_fee_applied,
            RequestedBanditType=lazy_cost_cache.bandit_type,
            ActualBanditType=max_tariff_place.bandit_type,
            ActualBanditVersion=max_tariff_place.bandit_version,
        )
        coach.fee_calculation_token = base64.b64encode(fee_calculation_token.SerializeToString())


def set_reservation_variants_amount_lazy(coach, lazy_cost_cache):
    if not coach.reservation_variants:
        return
    for variant in coach.reservation_variants:
        lazy_cost_cache.calculate_ticket_cost_lazy(
            coach.type, coach.service_class_code, variant.original_amount, variant.original_service_amount,
            variant.set_calculated_ticket_cost,
        )
        if variant.original_non_refundable_amount:
            lazy_cost_cache.calculate_ticket_cost_lazy(
                coach.type, coach.service_class_code, variant.original_non_refundable_amount,
                variant.original_service_amount, variant.set_non_refundable_amount,
            )
