# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import re

BRAND_LASTOCHKA = 'ЛАСТОЧКА'
BRAND_SAPSAN = 'САПСАН'
OWNER_DOSS = 'ДОСС'


def check_train_number(train_number_re_string, train_number):
    if not train_number_re_string:
        return False

    train_number_re_string = re.compile(train_number_re_string, flags=re.U | re.I)
    return bool(train_number_re_string.match(train_number))


def can_segregate_pets(coach_owner, train_brand, coaches):
    """
    Проверяет, могут ли быть разделены места в вагонах с животными на:
    - места для пассажиров без животных, провоз животных запрещен,
    - места для пассажиров с животными.
    :param coach_owner: Владелец вагона
    :param train_brand: Бренд поезда
    :param coaches: Список вагонов
    :return: Возможность сегрегации мест по признаку наличия животного.
    """
    return (any([coach.pet_places_only for coach in coaches]) or
            coach_owner == OWNER_DOSS and train_brand in (BRAND_LASTOCHKA, BRAND_SAPSAN))


def _set_pets_in_coach(coach, can_segregate):
    if coach.pet_places_only:
        coach.pets_allowed = True
        coach.pets_segregated = True
        return coach

    if coach.pet_in_coach:
        coach.pets_allowed = not can_segregate
        coach.pets_segregated = can_segregate
        return coach

    coach.pets_allowed = False
    coach.pets_segregated = False
    return coach


def set_pets(coach_owner, train_brand, coaches):
    """
    Устанавливает флаги pets_segregated и pets_allowed для каждого вагона.
    :param coach_owner: Владелец вагона
    :param train_brand: Бренд поезда
    :param coaches: Список вагонов
    """
    can_segregate = can_segregate_pets(coach_owner, train_brand, coaches)
    return [_set_pets_in_coach(coach, can_segregate) for coach in coaches]


def is_pets_allowed(train_brand, pet_places_only, pet_in_coach):
    """
    Возвращает допустимость провоза животных.
    :param train_brand: Название поезда: Сапсан, Ласточка.
    :param pet_places_only: Признак категории вагона «места для пассажиров с животными» (* Ж)
    :param pet_in_coach: Признак категории вагона «есть места для пассажиров с животными» (Ж)
    :return: Допустим ли провоз животных в данном вагоне.
    """
    if pet_places_only:
        return True

    if train_brand and train_brand.upper() in (BRAND_LASTOCHKA, BRAND_SAPSAN):
        return False

    return pet_in_coach


PLACES_RE = re.compile(r'(?P<Prefix>[А-Я]*)(?P<StringNumber>[0-9]*)(?P<Suffix>[А-Я]*)', re.U + re.I)


def yt_parse_place(raw_place):
    place_value = {}
    matched = PLACES_RE.match(raw_place)
    for group_name in ('Prefix', 'StringNumber', 'Suffix'):
        value = matched.group(group_name)
        place_value[group_name] = value

    place_value['Number'] = int(matched.group('StringNumber'))
    return place_value


def yt_build_car_type_flags(raw_car_description):
    """
    :param raw_car_description: CarDescription для IM, <R> для UFS
    """
    def get_suffix(_rzd_car_type):
        if _rzd_car_type.startswith('У'):
            return 'U'

        return {
            'НФ': 'NF',
            'Ж': 'ZH',
            '*Ж': 'ZH1',
            '*Д': 'D1',
            'БН': 'BN',
            'И': 'INV',
            'МЖ': 'MZH'}.get(_rzd_car_type)

    car_type_flags = {}
    text = raw_car_description.strip()
    rzd_car_types = text.replace('* ', '*').split(' ')
    for rzd_car_type in rzd_car_types:
        suffix = get_suffix(rzd_car_type)
        if suffix == 'U':
            value = int(rzd_car_type[1:])
        else:
            value = True
        car_type_flags[suffix] = value
    return car_type_flags


def yt_build_train_type_flags(raw_train_type):
    raw_train_type = raw_train_type.strip()
    flags = {}
    for rzd_train_type in raw_train_type.split(' '):
        suffix = {
            'ПАСС': 'PASS',
            'СК': 'SK',
            'СКРСТ': 'SKRST',
            'ФИРМ': 'FIRM'
        }.get(rzd_train_type)
        flags[suffix] = True

    return flags


def prepend_keys(path, source_dict):
    """
    Добавляет path в ключи source_dict
    >>> prepend_keys('b', {'a': 1}) == {'b__a': 1}
    """
    return {'{}__{}'.format(path, k): v for k, v in source_dict.items()}
