# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import decimal
import json
import logging
from datetime import datetime

import requests
import six
from django.conf import settings
from django.utils.text import force_text

from common.dynamic_settings.default import conf
from common.utils.date import MSK_TZ, smart_localize
from common.utils.field_masker import FieldMasker
from common.utils.railway import get_railway_tz_by_point
from common.utils.yasmutil import Metric, MeasurableDecorator
from travel.rasp.train_api.train_partners import config
from travel.rasp.train_api.train_partners.base import PartnerError, RzhdStatus, AbstractPartnerMeasurable
from travel.rasp.train_api.train_purchase.core.enums import DocumentType, TrainPartner, OperationStatus, TrainPartnerCredentialId

log = logging.getLogger(__name__)

IM_DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
IM_BUS_TRANSPORT_TYPE = 'Bus'

PROVIDER_P1 = 'P1'
PROVIDER_P2 = 'P2'


@six.python_2_unicode_compatible
class ImCredentialNotFoundError(Exception):
    def __init__(self, credential_id):
        self.credential_id = credential_id

    def __str__(self):
        return '<ImCredentialNotFoundError: Credentials not found for credential_id="{}">'.format(self.credential_id)


@six.python_2_unicode_compatible
class ImError(PartnerError):
    def __init__(self, code, message, message_params):
        super(ImError, self).__init__(code, message)
        self.message_params = message_params

    def __str__(self):
        return ('<ImError: code="{self.code}" message="{self.message}"'
                ' message_params={self.message_params}>'.format(self=self))

    def is_communication_error(self):
        return self.code in [1, -1, -3]

    def is_retry_allowed(self):
        return self.code in [1, 2, 94]

    def is_refund_not_found_error(self):
        # 180 	Заказ не найден: {0}
        return self.code == 180

    def is_trains_not_found_error(self):
        return (self.code in (301, 310, 312, 313, 314, 315, 316) or
                (self.code == 43 and self.message_params == ['request.DepartureDate']))

    def is_update_from_express_error(self):
        return self.code == 61

    def is_empty_result_error(self):
        return self.code == 311

    def is_non_cacheable_error(self):
        return self.code in conf.TRAIN_PARTNERS_IM_NONCACHEABLE_ERROR_CODES

    def is_invalid_passenger_phone(self):
        return self.code == 1385

    def is_invalid_passenger_email(self):
        return self.code == 1386

    def get_user_message(self):
        try:
            return conf.TRAIN_PARTNERS_IM_ENDUSER_ERROR_MSG[str(self.code)]
        except KeyError:
            pass
        except Exception:
            log.exception('Проверьте дин.настройку TRAIN_PARTNERS_IM_ENDUSER_ERROR_MSG')
        return self.message

    def get_data(self):
        return {
            'code': self.code,
            'message_params': self.message_params,
            'message': self.message
        }

    @classmethod
    def raise_exception(cls, code, message, message_params):
        # некритично, если нет подходящих мест, или некорректный ввод
        raise ImNonCriticalError(code, message, message_params) \
            if code in conf.TRAIN_PARTNERS_IM_NONCRITICAL_ERROR_CODES else ImError(code, message, message_params)

    @classmethod
    def from_get_response_error(cls):
        return cls(code=-1, message="Can't get response", message_params=None)

    @classmethod
    def from_parse_json_error(cls):
        return cls(code=-2, message='Not a json answer', message_params=None)

    @classmethod
    def from_communication_error(cls):
        return cls(code=-3, message='Communication error', message_params=None)

    def __reduce__(self):
        return self.__class__, (self.code, self.message, self.message_params)


@six.python_2_unicode_compatible
class ImNonCriticalError(ImError):
    def __str__(self):
        return ('<ImNonCriticalError: code="{self.code}" message="{self.message}"'
                ' message_params={self.message_params}>'.format(self=self))


def get_im_raw_response(method, params, credential_id, field_masker, timeout=None):
    log.info('Making request to IM: %s with parameters\n %s', method,
             force_text(json.dumps(field_masker.apply(params), sort_keys=True, indent=2, ensure_ascii=False)))

    credentials = config.TRAIN_PARTNERS_CREDENTIALS.get(credential_id)
    if not credentials:
        raise ImCredentialNotFoundError(credential_id)

    response = requests.post(
        '{config.TRAIN_PARTNERS_IM_URL}{method}'.format(config=config, method=method),
        auth=(credentials.login, credentials.password),
        headers={'POS': credentials.pos, 'Content-Type': 'application/json'},
        data=json.dumps(params, sort_keys=True),
        timeout=timeout or settings.DEFAULT_PARTNER_TIMEOUT,
    )

    if 'application/pdf' in response.headers.get('Content-Type', '') or response.content.startswith(b'%PDF'):
        text = '<some pdfdata>'
    else:
        try:
            data = response.json()
            text = force_text(json.dumps(field_masker.apply(data), indent=2, ensure_ascii=False, sort_keys=True))
        except Exception:
            text = response.text

    log.info('Response to {response.request.url}: status code {response.status_code}, data:\n{text}'
             .format(response=response, text=text))

    return response


def get_im_response(method, params, credential_id=TrainPartnerCredentialId.IM,
                    field_masker=FieldMasker(), timeout=None):
    @measurable_im(endpoint_name=im_method_to_metric(method))
    def execute():
        return _get_im_response(method, params, credential_id, field_masker, timeout)
    return execute()


def im_method_to_metric(method):
    return method.replace('/', '_').lower()


def _get_im_response(method, params, credential_id, field_masker, timeout):
    try:
        response = get_im_raw_response(method, params, credential_id, field_masker, timeout)
    except ImCredentialNotFoundError as e:
        log.exception(force_text(e))
        raise
    except Exception:
        log.exception('Ошибка получения ответа от ИМ %s', method)
        raise ImError.from_get_response_error()

    if response.status_code == 502:
        log.exception('Ошибка связи с ИМ %s', method)
        raise ImError.from_communication_error()

    try:
        result = response.json(parse_float=decimal.Decimal)
    except Exception:
        log.exception('Ошибка разбора ответа от ИМ %s', method)
        raise ImError.from_parse_json_error()

    if response.status_code != requests.codes.ok:
        ImError.raise_exception(result['Code'], result['Message'], result['MessageParams'])
    return result


def parse_datetime(text, point=None):
    if text is None:
        return None
    dt = datetime.strptime(text, IM_DATETIME_FORMAT)
    if point is None:
        tz = MSK_TZ
    else:
        tz = get_railway_tz_by_point(point)
    return smart_localize(dt, tz)


def _reverse_dict(source):
    return {v: k for k, v in source.items()}


# следующие не поддерживаются: MilitaryOfficerCard, ReturnToCisCertificate, DiplomaticPassport, ServicePassport
DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE = {
    DocumentType.RUSSIAN_PASSPORT: 'RussianPassport',
    DocumentType.BIRTH_CERTIFICATE: 'BirthCertificate',
    DocumentType.RUSSIAN_INTERNATIONAL_PASSPORT: 'RussianForeignPassport',
    DocumentType.FOREIGN_DOCUMENT: 'ForeignPassport',
    DocumentType.SAILOR_PASSPORT: 'SailorPassport',
    DocumentType.MILITARY_CARD: 'MilitaryCard'
}
DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE = _reverse_dict(DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE)

BLANK_STATUS_TO_RZHD_STATUS = {
    'ElectronicRegistrationAbsent': RzhdStatus.NO_REMOTE_CHECK_IN,
    'ElectronicRegistrationPresent': RzhdStatus.REMOTE_CHECK_IN,
    'NotConfirmed': RzhdStatus.RESERVATION,
    'Voided': RzhdStatus.CANCELLED,
    'Returned': RzhdStatus.REFUNDED,
    'PlacesReturned': RzhdStatus.PLACES_REFUNDED,
    'VoucherIssued': RzhdStatus.STRICT_BOARDING_PASS,
    'TripWasInterrupted': RzhdStatus.INTERRUPTED,
    'TripWasInterruptedAndResumedAfter': RzhdStatus.RESUMED,
}
RZHD_STATUS_TO_BLANK_STATUS = _reverse_dict(BLANK_STATUS_TO_RZHD_STATUS)
IM_OPERATION_STATUS_TO_OPERATION_STATUS = {
    'Succeeded': OperationStatus.OK,
    'InProcess': OperationStatus.IN_PROCESS,
    'Error': OperationStatus.FAILED,
}
OPERATION_STATUS_TO_IM_OPERATION_STATUS = _reverse_dict(IM_OPERATION_STATUS_TO_OPERATION_STATUS)


class measurable(AbstractPartnerMeasurable):
    prefix = TrainPartner.IM.value


class measurable_im(MeasurableDecorator):
    prefix = TrainPartner.IM.value

    def _handle_error(self, exc):
        result = super(measurable_im, self)._handle_error(exc)
        if isinstance(exc, ImError):
            error_type = 'other_errors_cnt'
            if exc.is_communication_error():
                error_type = 'communication_errors_cnt'
            elif isinstance(exc, ImNonCriticalError):
                error_type = 'non_critical_errors_cnt'
            result.extend([
                Metric(self._name('errors_cnt'), 1, 'ammm'),
                Metric(self._name(error_type), 1, 'ammm'),
                Metric('errors_new_cnt', 1, 'ammm'),  # TODO: после удаления старого @measurable вернуть errors_cnt
                Metric(error_type, 1, 'ammm'),
                Metric(self._name('im_response_codes'), [[exc.code, 1]], 'ahhh'),
            ])
        return result
