# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.encoding import force_text

from common.utils.field_masker import FieldMasker
from travel.rasp.train_api.train_partners.base.confirm_ticket import ConfirmResult
from travel.rasp.train_api.train_partners.im.base import measurable, get_im_response, parse_datetime

CONFIRM_TICKET_ENDPOINT = 'Order/V1/Reservation/Confirm'
CANCEL_TICKET_ENDPOINT = 'Order/V1/Reservation/Cancel'

IM_INSURANCE_CONFIRM_TYPE = 'ApiContracts.Insurance.V1.Messages.Travel.InsuranceTravelConfirmResponse, ApiContracts'

field_masker = FieldMasker(mask_fields={
    'Customers': [{'DocumentNumber': 1, 'Birthday': 1, 'BirthDate': 1}]
})


@measurable()
def confirm_order(order, site_fee=0, timeout=None):
    response = get_im_response(CONFIRM_TICKET_ENDPOINT, {
        'OrderId': order.current_partner_data.im_order_id,
        'OrderCustomerIds': None,
        'OrderCustomerDocuments': None,
        'ProviderPaymentForm': 'Card'
    }, credential_id=order.partner_credential_id, field_masker=field_masker, timeout=timeout)
    confirm_result = next(cr for cr in response['ConfirmResults']
                          if str(cr['OrderItemId']) == order.current_partner_data.operation_id)
    insurances_results = {force_text(cr['OrderItemId']): cr['IsSucceeded'] for cr in response['ConfirmResults']
                          if cr.get('$type') == IM_INSURANCE_CONFIRM_TYPE}

    return ConfirmResult(expire_set_er=parse_datetime(confirm_result['ExpirationElectronicRegistrationDateTime']),
                         order_num=str(confirm_result['ReservationNumber']), insurances_results=insurances_results)


@measurable()
def cancel_order(order, partner_data=None, timeout=None):
    get_im_response(CANCEL_TICKET_ENDPOINT, {
        'OrderId': (partner_data or order.current_partner_data).im_order_id,
    }, credential_id=order.partner_credential_id, field_masker=field_masker, timeout=timeout)
