# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import copy
import random
from datetime import datetime, timedelta

import factory

from travel.rasp.library.python.common23.tester.helpers.class_counter import inc_class_counter
from common.utils.date import UTC_TZ
from travel.rasp.library.python.common23.date.environment import now_aware
from travel.rasp.train_api.train_partners.im.base import (
    IM_DATETIME_FORMAT, RZHD_STATUS_TO_BLANK_STATUS, DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE,
    OPERATION_STATUS_TO_IM_OPERATION_STATUS
)
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus


def make_im_customer(passenger, override=None):
    override = override or {}
    return {
        "OrderCustomerId": 66650,
        "FirstName": override.get('first_name', passenger.first_name),
        "MiddleName": override.get('patronymic', passenger.patronymic),
        "LastName": override.get('last_name', passenger.last_name),
        "Sex": "Male",
        "BirthDate": "1976-10-12T00:00:00",
        "DocumentNumber": "4601123450",
        "DocumentValidTill": "9999-12-31T23:59:59",
        "DocumentType": "RussianPassport",
        "CitizenshipCode": "RU"
    }


def make_im_blank(ticket, override=None):
    override = override or {}
    status = RZHD_STATUS_TO_BLANK_STATUS[override.get('rzhd_status', ticket.rzhd_status)]
    return {
        "$type": "ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayOrderItemBlankInfo, ApiContracts",
        "VoucherNumber": None,
        "BaseFare": 0.0,
        "AdditionalPrice": 0.0,
        "ServicePrice": 0.0,
        "VatRateValues": None,
        "TariffType": "Full",
        "BlankStatus": status,
        "IsElectronicRegistrationSet": True,
        "IsMealOptionPossible": False,
        "PendingElectronicRegistration": "NoValue",
        "ElectronicRegistrationSetDateTime": None,
        "SignSequence": None,
        "PlaceQuantity": 1,
        "OrderItemBlankId": int(override.get('blank_id', ticket.blank_id)),
        "PreviousOrderItemBlankId": int(override.get('previous_blank_id', 0)),
        "BlankNumber": "71234567890000",
        "Amount": 9481.7
    }


def make_im_order_item(order, override):
    override = override or {}
    im_status = OPERATION_STATUS_TO_IM_OPERATION_STATUS[override.get('status', OperationStatus.OK)]
    is_refund = override.get('is_refund', False)
    if is_refund:
        blank_params_by_blank_id = override.get('refund_blanks', {})
    else:
        blank_params_by_blank_id = override.get('blanks', {})
    blanks = [
        make_im_blank(t, blank_params_by_blank_id.get(t.blank_id))
        for t in order.iter_tickets()
    ]

    result = {
        "$type": "ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayFullOrderItemInfo, ApiContracts",
        "ServiceType": "Tickets",
        "PlaceQuantity": 1,
        "OriginStationName": "МОСКВА ОКТЯБРЬСКАЯ",
        "DestinationStationName": "САНКТ-ПЕТЕРБУРГ-ГЛАВН.",
        "TrainNumber": "054ЧА",
        "BookingTrainNumber": None,
        "TrainNumberToGetRoute": None,
        "CarNumber": "07",
        "CarType": "Luxury",
        "ElectronicRegistrationExpirationDateTime": "2016-11-01T22:40:00",
        "PlaceReservationType": "Usual",
        "ServiceClass": "1Б",
        "AdditionalInformation": "",
        "CarrierDescription": "ЗАО ТК \"ГСЭ\"",
        "OrderItemCustomers": [
            {
                "$type": "ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayOrderItemCustomerInfo, ApiContracts",
                "OrderItemBlankId": blanks[0]['OrderItemBlankId'],
                "Places": "001",
                "PlaceQuantity": 1,
                "OrderCustomerId": 66650,
                "OrderItemCustomerId": 51445,
                "Amount": 9481.7,
                "ClientFeeCalculation": {
                    "Charge": 100.0,
                    "Profit": 0.0
                }
            }
        ],
        "OrderItemBlanks": blanks,
        "ArrivalDateTime": "2016-11-02T08:36:00",
        "OriginLocationCode": "2006004",
        "OriginLocationName": "МОСКВА",
        "DestinationLocationCode": "2004001",
        "DestinationLocationName": "САНКТ-ПЕТЕРБУРГ",
        "OrderId": 51978,
        "AgentReferenceId": override.get('reference_id'),
        "OrderItemId": int(order.current_partner_data.operation_id),
        "Amount": 9481.7,
        "ReservationNumber": "71234567890000",
        "OperationType": "Purchase",
        "SimpleOperationStatus": im_status,
        "DetailedOperationStatus": "Error",
        "DepartureDateTime": "2016-11-01T23:40:00",
        "CreateDateTime": datetime.strftime(override.get('create_dt', datetime(2016, 10, 27, 18, 50, 4)),
                                            IM_DATETIME_FORMAT),
        "ConfirmTimeLimit": "2016-10-27T19:04:04",
        "ConfirmDateTime": "2016-10-27T19:00:15",
        "ClientFeeCalculation": {
            "Charge": 100.0,
            "Profit": 0.0
        },
        "AgentFeeCalculation": {
            "Charge": 100.0,
            "Profit": 0.0
        },
        "ProviderPaymentForm": None,
        "IsExternallyLoaded": False
    }
    if is_refund:
        result['OperationType'] = 'Return'
    return result


class ImOrderCustomerFactory(factory.DictFactory):
    BirthDate = '1989-03-03T00:00:00'
    CitizenshipCode = 'RU'
    DocumentNumber = '3708123123'
    DocumentType = 'RussianPassport'
    DocumentValidTill = None
    FirstName = 'Петр'
    LastName = 'Петров'
    MiddleName = 'Петрович'
    OrderCustomerId = 0
    Sex = 'Male'

    @classmethod
    def create(cls, **kwargs):
        passenger = kwargs.pop('train_order_passenger', None)
        if passenger:
            kwargs.setdefault('FirstName', passenger.first_name)
            kwargs.setdefault('LastName', passenger.last_name)
            kwargs.setdefault('MiddleName', passenger.patronymic)
            kwargs.setdefault('DocumentType', DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE[passenger.doc_type])
            kwargs.setdefault('OrderCustomerId', passenger.customer_id)
        return super(ImOrderCustomerFactory, cls).create(**kwargs)


class ImOrderItemBlankFactory(factory.DictFactory):
    AdditionalPrice = 500
    BaseFare = 500.11
    Amount = 1000.11
    BlankNumber = factory.LazyAttribute(
        lambda _x: str(inc_class_counter(ImOrderItemBlankFactory, 'BlankNumber', start_value=10000000000000)))
    BlankStatus = 'ElectronicRegistrationPresent'
    ElectronicRegistrationSetDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=3)).strftime('%Y-%m-%dT%H:%M:%S'))
    IsElectronicRegistrationSet = True
    IsMealOptionPossible = False
    OrderItemBlankId = factory.LazyAttribute(
        lambda _x: inc_class_counter(ImOrderItemBlankFactory, 'OrderItemBlankId', start_value=100000))
    PendingElectronicRegistration = 'NoValue'
    PlaceQuantity = 1
    PrepaidMealInfo = None
    PreviousOrderItemBlankId = 0
    ServicePrice = 99
    SignSequence = ''.join('{:02x}'.format(random.randint(0, 255)) for i in range(136)).upper()
    TariffInfo = {
        'TariffName': 'ПОЛНЫЙ',
        'TariffType': 'Full'
    }
    TariffType = 'Full'
    VatRateValues = [
        {
            'Rate': 0,
            'Value': 0,
        },
        {
            'Rate': 18,
            'Value': 15.1,
        }]
    VoucherNumber = None

    @classmethod
    def create(cls, **kwargs):
        if 'train_order_ticket' in kwargs:
            ticket = kwargs.pop('train_order_ticket')
            kwargs.setdefault('OrderItemBlankId', int(ticket.blank_id))
            kwargs.setdefault('BlankStatus', RZHD_STATUS_TO_BLANK_STATUS[ticket.rzhd_status])
        if '$type' not in kwargs:
            kwargs['$type'] = 'ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayOrderItemBlankInfo, ApiContracts'
        if 'VatRateValues' not in kwargs and 'Amount' in kwargs:
            kwargs['VatRateValues'] = [
                {
                    'Rate': 0,
                    'Value': 0,
                },
                {
                    'Rate': 18,
                    'Value': round(kwargs['Amount'] * 0.18 / 1.18, 2),
                }]
        return super(ImOrderItemBlankFactory, cls).create(**kwargs)


class ImOrderItemCustomerFactory(factory.DictFactory):
    Amount = 1000.11
    ClientFeeCalculation = None
    OrderCustomerId = 0
    OrderItemBlankId = 0
    OrderItemCustomerId = factory.LazyAttribute(
        lambda _x: inc_class_counter(ImOrderItemCustomerFactory, 'OrderItemCustomerId', start_value=200000)
    )
    PlaceQuantity = 1
    Places = '017П'

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        if '$type' not in kwargs:
            kwargs['$type'] = 'ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayOrderItemCustomerInfo, ApiContracts'
        return kwargs

    @classmethod
    def create(cls, **kwargs):
        passenger = kwargs.pop('train_order_passenger', None)
        if passenger:
            kwargs.setdefault('OrderCustomerId', passenger.customer_id)
        return super(ImOrderItemCustomerFactory, cls).create(**kwargs)


class ImInsuranceOrderItemCustomerFactory(factory.DictFactory):
    ImType = 'ApiContracts.Order.V1.Info.OrderItem.Insurance.InsuranceOrderItemCustomerInfo, ApiContracts'

    AgentFeeCalculation = {
        "Charge": 0,
        "Profit": 65
    }
    Amount = 100.0
    ClientFeeCalculation = None
    OrderCustomerId = 0
    OrderItemCustomerId = factory.LazyAttribute(
        lambda _x: inc_class_counter(ImOrderItemCustomerFactory, 'OrderItemCustomerId', start_value=200000)
    )

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        if '$type' not in kwargs:
            kwargs['$type'] = cls.ImType
        return kwargs

    @classmethod
    def create(cls, **kwargs):
        passenger = kwargs.pop('train_order_passenger', None)
        if passenger:
            kwargs.setdefault('OrderCustomerId', passenger.customer_id)
        return super(ImInsuranceOrderItemCustomerFactory, cls).create(**kwargs)


class ImBaseOrderItemFactory(factory.DictFactory):
    OrderId = 0
    OrderItemId = factory.LazyAttribute(
        lambda _x: inc_class_counter(ImRailwayOrderItemFactory, 'OrderItemId', start_value=110000))


class ImRailwayOrderItemFactory(ImBaseOrderItemFactory):
    ImType = 'ApiContracts.Order.V1.Info.OrderItem.Railway.RailwayFullOrderItemInfo, ApiContracts'

    AdditionalInformation = 'ПРЕДВАРИТЕЛЬНЫЙ ДОСМОТР НА ВОКЗАЛЕ. ВРЕМЯ ОТПР И ПРИБ МОСКОВСКОЕ'
    AgentFeeCalculation = {
        "Charge": 112.5,
        "Profit": 0
    }
    AgentReferenceId = ''
    Amount = 1000.11
    ArrivalDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(days=14)).strftime('%Y-%m-%dT%H:%M:%S'))
    BookingTrainNumber = '760А'
    CarNumber = '09'
    CarType = 'Sedentary'
    CarrierDescription = 'ДОСС РЖД / ОАО \"РЖД\" (7708503727)'
    ClientFeeCalculation = None
    ConfirmDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=1)).strftime('%Y-%m-%dT%H:%M:%S'))
    ConfirmTimeLimit = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(minutes=12)).strftime('%Y-%m-%dT%H:%M:%S'))
    CreateDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=3)).strftime('%Y-%m-%dT%H:%M:%S'))
    DepartureDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(days=14, hours=-4)).strftime('%Y-%m-%dT%H:%M:%S'))
    DestinationLocationCode = '2004001'
    DestinationLocationName = 'САНКТ-ПЕТЕРБУРГ-ГЛАВН.'
    DestinationStationName = 'С-ПЕТЕР-ГЛ'
    DestinationTimeZoneDifference = 0
    DetailedOperationStatus = 'Succeeded'
    ElectronicRegistrationExpirationDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(days=14, hours=-5)).strftime('%Y-%m-%dT%H:%M:%S'))
    IsExternallyLoaded = False
    IsOnlyFullReturnPossible = False
    OperationType = 'Purchase'
    OrderItemBlanks = factory.List([
        factory.SubFactory(ImOrderItemBlankFactory)
    ])
    OrderItemCustomers = factory.List([])
    OriginLocationCode = '2006004'
    OriginLocationName = 'МОСКВА ОКТЯБРЬСКАЯ'
    OriginStationName = 'МОСКВА ОКТ'
    OriginTimeZoneDifference = 0
    PlaceQuantity = 0
    PlaceReservationType = 'Usual'
    PosSysName = 'yandex_test2'
    ProviderPaymentForm = 'Card'
    ReservationNumber = '0'
    ServiceClass = '2С'
    ServiceType = 'Tickets'
    SimpleOperationStatus = 'Succeeded'
    TrainNumber = '760АА'
    TrainNumberToGetRoute = '760А'

    @classmethod
    def create(cls, **kwargs):
        if 'train_order' in kwargs:
            train_order = kwargs.pop('train_order')
            kwargs.setdefault('OrderItemId', int(train_order.current_partner_data.operation_id))
            blanks = [ImOrderItemBlankFactory(train_order_ticket=ticket) for ticket in train_order.iter_tickets()]
            kwargs.setdefault('OrderItemBlanks', blanks)
        if '$type' not in kwargs:
            kwargs['$type'] = cls.ImType
        return super(ImRailwayOrderItemFactory, cls).create(**kwargs)

    @classmethod
    def _after_postgeneration(cls, obj, create, results=None):
        obj['create'] = create
        obj['results'] = results

        obj['PlaceQuantity'] = sum(blank['PlaceQuantity'] for blank in obj['OrderItemBlanks'])
        if obj['OrderItemBlanks']:
            obj['ReservationNumber'] = obj['OrderItemBlanks'][0]['BlankNumber']
        obj['Amount'] = sum(blank['Amount'] for blank in obj['OrderItemBlanks'])
        return obj


class ImInsuranceOrderItemFactory(ImBaseOrderItemFactory):
    ImType = 'ApiContracts.Order.V1.Info.OrderItem.Insurance.RailwayInsuranceFullOrderItemInfo, ApiContracts'

    AdditionalInformation = 'ПРЕДВАРИТЕЛЬНЫЙ ДОСМОТР НА ВОКЗАЛЕ. ВРЕМЯ ОТПР И ПРИБ МОСКОВСКОЕ'
    AgentFeeCalculation = {
        "Charge": 0,
        "Profit": 65
    }
    AgentReferenceId = ''
    Amount = 100.0
    ClientFeeCalculation = None
    ConfirmDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=1)).strftime('%Y-%m-%dT%H:%M:%S'))
    ConfirmTimeLimit = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(minutes=12)).strftime('%Y-%m-%dT%H:%M:%S'))
    CreateDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=3)).strftime('%Y-%m-%dT%H:%M:%S'))
    DepartureDateTime = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) + timedelta(days=14, hours=-4)).strftime('%Y-%m-%dT%H:%M:%S'))
    DetailedOperationStatus = 'Succeeded'
    IsExternallyLoaded = False
    MainOrderItemId = 0,
    OperationType = 'Purchase'
    OrderItemBlanks = factory.List([])
    OrderItemCustomers = factory.List([])
    Package = 'AccidentWithFloatPremium'
    PosSysName = 'yandex_test2'
    PreviousOrderItemId = 0
    ProviderPaymentForm = 'Card'
    ReservationNumber = '0'
    SimpleOperationStatus = 'Succeeded'
    Supplier = 'Renessans'

    @classmethod
    def create(cls, **kwargs):
        passenger = kwargs.pop('train_order_passenger', None)
        if passenger:
            insurance = passenger.insurance
            kwargs.setdefault('OrderItemId', int(insurance.operation_id))
            customers = [ImInsuranceOrderItemCustomerFactory(train_order_passenger=passenger)]
            kwargs.setdefault('OrderItemCustomers', customers)
            if insurance.operation_status:
                kwargs.setdefault('SimpleOperationStatus',
                                  OPERATION_STATUS_TO_IM_OPERATION_STATUS[insurance.operation_status])
        if '$type' not in kwargs:
            kwargs['$type'] = cls.ImType
        return super(ImInsuranceOrderItemFactory, cls).create(**kwargs)


class ImOrderInfoFactory(factory.DictFactory):
    Amount = 1000.11
    Confirmed = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=1)).strftime('%Y-%m-%dT%H:%M:%S'))
    ContactEmails = ['']
    ContactPhone = ''
    Created = factory.LazyAttribute(
        lambda _x: (now_aware().astimezone(UTC_TZ) - timedelta(minutes=3)).strftime('%Y-%m-%dT%H:%M:%S'))
    OrderCustomers = factory.List([
        factory.SubFactory(ImOrderCustomerFactory)
    ])
    OrderId = factory.LazyAttribute(
        lambda _x: inc_class_counter(ImOrderInfoFactory, 'OrderId', start_value=120000))
    OrderItems = factory.List([
        factory.SubFactory(ImBaseOrderItemFactory)
    ])
    PosSysName = 'yandex_test2'

    @classmethod
    def create(cls, **kwargs):
        if 'train_order' in kwargs:
            train_order = kwargs.pop('train_order')
            customers = [ImOrderCustomerFactory(train_order_passenger=passenger)
                         for passenger in train_order.passengers]
            order_items = [ImRailwayOrderItemFactory(train_order=train_order)]
            for passenger in train_order.passengers:
                if passenger.insurance and passenger.insurance.operation_id:
                    order_items.append(ImInsuranceOrderItemFactory(train_order_passenger=passenger))
            kwargs.setdefault('OrderItems', order_items)
            kwargs.setdefault('OrderCustomers', customers)
        if kwargs.get('OrderItems'):
            first_item = kwargs['OrderItems'][0]
            kwargs.setdefault('Confirmed', first_item['ConfirmDateTime'])
            kwargs.setdefault('Created', first_item['CreateDateTime'])
        return super(ImOrderInfoFactory, cls).create(**kwargs)

    @classmethod
    def _after_postgeneration(cls, obj, create, results=None):
        if obj['OrderItems']:
            obj['Amount'] = sum(item['Amount'] if item['OperationType'] == 'Purchase' else -item['Amount']
                                for item in obj['OrderItems'])

        for order_item in obj['OrderItems']:
            order_item['OrderId'] = obj['OrderId']

        customers_by_blank = {}
        customers_len = len(obj['OrderCustomers'])
        for order_item in obj['OrderItems']:
            if order_item['OperationType'] == 'Purchase' and order_item['$type'] == ImRailwayOrderItemFactory.ImType:
                if not order_item['OrderItemCustomers']:
                    for i, blank in enumerate(order_item['OrderItemBlanks']):
                        customer_id = obj['OrderCustomers'][i % customers_len]['OrderCustomerId']
                        order_item['OrderItemCustomers'].append(ImOrderItemCustomerFactory(
                            OrderItemBlankId=blank['OrderItemBlankId'],
                            Amount=blank['Amount'],
                            OrderCustomerId=customer_id,
                            Places=str(i + 1),
                            PlaceQuantity=1,
                        ))
                customers_by_blank = {oic['OrderItemBlankId']: oic['OrderCustomerId']
                                      for oic in order_item['OrderItemCustomers']}
            else:
                for i, blank in enumerate(order_item['OrderItemBlanks']):
                    customer_id = customers_by_blank[blank['PreviousOrderItemBlankId']]
                    order_item['OrderItemCustomers'].append(ImOrderItemCustomerFactory(
                        OrderItemBlankId=blank['OrderItemBlankId'],
                        Amount=blank['Amount'],
                        OrderCustomerId=customer_id,
                        Places="",
                        PlaceQuantity=0,
                    ))
        return obj

    @classmethod
    def add_refund_item(cls, order, blank_id_to_refund, amount_to_refund=None, **kwargs):
        order = copy.deepcopy(order)
        sale_item = next(
            oi for oi in order['OrderItems']
            if oi['OperationType'] == 'Purchase'
            and oi['$type'] == ImRailwayOrderItemFactory.ImType
        )
        blank_to_refund = next(x for x in sale_item['OrderItemBlanks'] if x['OrderItemBlankId'] == blank_id_to_refund)
        blank_to_refund['BlankStatus'] = 'Returned'
        if amount_to_refund is None:
            amount_to_refund = round(blank_to_refund['Amount'] * 0.8)
        vat_values = copy.deepcopy(blank_to_refund['VatRateValues'])
        vat_values.append({
            'Rate': 18,
            'Value': round((blank_to_refund['Amount'] - amount_to_refund) * 0.18 / 1.18, 2),
        })
        refund_blank = ImOrderItemBlankFactory(
            PreviousOrderItemBlankId=blank_to_refund['OrderItemBlankId'],
            BlankNumber=blank_to_refund['BlankNumber'],
            PlaceQuantity=blank_to_refund['PlaceQuantity'],
            BlankStatus='ElectronicRegistrationAbsent',
            Amount=amount_to_refund,
            VatRateValues=vat_values,
        )
        kwargs.setdefault('OperationType', 'Return')
        kwargs.setdefault('OrderItemBlanks', [refund_blank])
        refund_item = ImRailwayOrderItemFactory(**kwargs)
        order['OrderItems'].append(refund_item)
        ImOrderInfoFactory._after_postgeneration(order, True)
        return order

    @classmethod
    def add_refund_insurance_item(cls, order_info, operation_id_to_refund):
        order_info = copy.deepcopy(order_info)
        sale_item = next(
            oi for oi in order_info['OrderItems']
            if oi['OperationType'] == 'Purchase'
            and oi['$type'] == ImInsuranceOrderItemFactory.ImType
            and oi['OrderItemId'] == operation_id_to_refund
        )
        refund_item = copy.deepcopy(sale_item)

        refund_item['OperationType'] = 'Return'
        refund_item['PreviousOrderItemId'] = operation_id_to_refund
        refund_item['OrderItemId'] = 39863826 + int(operation_id_to_refund),
        order_info['OrderItems'].append(refund_item)
        ImOrderInfoFactory._after_postgeneration(order_info, True)
        return order_info
