# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.encoding import force_text
from django.utils.functional import cached_property
from six.moves import UserDict

from common.utils.field_masker import FieldMasker
from travel.rasp.train_api.train_partners.base.get_order_info import TicketInfo, InsuranceInfo, PassengerInfo, OrderInfoResult
from travel.rasp.train_api.train_partners.im.base import (
    get_im_response, BLANK_STATUS_TO_RZHD_STATUS, IM_OPERATION_STATUS_TO_OPERATION_STATUS, parse_datetime, measurable
)
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus

IM_ORDER_INFO_METHOD = 'Order/V1/Info/OrderInfo'

field_masker = FieldMasker(mask_fields={
    'OrderCustomers': [{'DocumentNumber': 1, 'BirthDate': 1}]
})


@measurable()
def get_order_info(order, timeout=None):
    return _get_order_info({'OrderId': order.current_partner_data.im_order_id},
                           credential_id=order.partner_credential_id, timeout=timeout)


@measurable()
def get_order_info_by_reference_id(order, reference_id):
    return _get_order_info({'AgentReferenceId': reference_id}, credential_id=order.partner_credential_id)


def _get_order_info(params, credential_id, timeout=None):
    response = get_im_response(IM_ORDER_INFO_METHOD, params,
                               credential_id=credential_id, field_masker=field_masker, timeout=timeout)

    reference_id = params.get('AgentReferenceId')
    order_items = [OrderItem(item) for item in response['OrderItems']]
    buy_ticket_item = next(item for item in order_items if item.is_buy and item.is_ticket)
    sorted_refund_ticket_items = get_sorted_refund_ticket_items(order_items)
    insurance_buy_items = [item for item in order_items if item.is_buy and item.is_insurance]
    insurance_refund_items = [item for item in order_items if item.is_refund and item.is_insurance]

    result = OrderInfoResult(
        buy_operation_id=force_text(buy_ticket_item['OrderItemId']),
        expire_set_er=buy_ticket_item.expire_set_er,
        status=get_status(reference_id, order_items, buy_ticket_item, sorted_refund_ticket_items),
        order_num=buy_ticket_item['ReservationNumber'],
        reserved_to=buy_ticket_item.reserved_to,
    )

    doc_id_by_passenger_id = {p['OrderCustomerId']: p['DocumentNumber'] for p in response['OrderCustomers']}
    birth_date_by_passenger_id = {p['OrderCustomerId']: parse_datetime(p['BirthDate'])
                                  for p in response['OrderCustomers']}
    passenger_by_id = {}
    for passenger in buy_ticket_item['OrderItemCustomers']:
        passenger_id = passenger['OrderCustomerId']
        passenger_by_id[passenger_id] = PassengerInfo(
            blank_id=force_text(passenger['OrderItemBlankId']),
            doc_id=doc_id_by_passenger_id[passenger_id],
            birth_date=birth_date_by_passenger_id[passenger_id],
            customer_id=force_text(passenger_id),
        )
    ticket_by_blank_id = {}
    for blank in buy_ticket_item['OrderItemBlanks']:
        blank_id = force_text(blank['OrderItemBlankId'])
        ticket_by_blank_id[blank_id] = TicketInfo(
            blank_id=blank_id,
            rzhd_status=BLANK_STATUS_TO_RZHD_STATUS[blank['BlankStatus']],
            amount=blank['Amount'],
            pending=blank.get('PendingElectronicRegistration') == 'ToCancel'
        )

    for order_item in sorted_refund_ticket_items:
        for passenger in order_item['OrderItemCustomers']:
            passenger_id = passenger['OrderCustomerId']
            passenger_by_id[passenger_id].refund_blank_id = force_text(passenger['OrderItemBlankId'])

        operation_id = force_text(order_item['OrderItemId'])
        for blank in order_item['OrderItemBlanks']:
            blank_id = force_text(blank['PreviousOrderItemBlankId'])
            ticket_by_blank_id[blank_id].refund_blank_id = force_text(blank['OrderItemBlankId'])
            ticket_by_blank_id[blank_id].refund_operation_id = operation_id
            ticket_by_blank_id[blank_id].refund_amount = blank['Amount']

            if (
                IM_OPERATION_STATUS_TO_OPERATION_STATUS[order_item['SimpleOperationStatus']] == OperationStatus.OK and
                order_item['IsExternallyLoaded']
            ):
                ticket_by_blank_id[blank_id].is_external = True

    insurances_by_id = {}
    for order_item in insurance_buy_items:
        item_status = IM_OPERATION_STATUS_TO_OPERATION_STATUS[order_item['SimpleOperationStatus']]
        order_item_id = force_text(order_item['OrderItemId'])
        insurances_by_id[order_item_id] = InsuranceInfo(
            order_item_id,
            order_item['Amount'],
            force_text(order_item['OrderItemCustomers'][0]['OrderCustomerId']),
            item_status,
            order_item.get('AgentReferenceId')
        )
    for order_item in insurance_refund_items:
        item_status = IM_OPERATION_STATUS_TO_OPERATION_STATUS[order_item['SimpleOperationStatus']]
        if item_status == OperationStatus.FAILED:
            pass
        order_item_id = force_text(order_item['PreviousOrderItemId'])
        if order_item_id not in insurances_by_id:
            insurances_by_id[order_item_id] = InsuranceInfo(
                order_item_id,
                order_item['Amount'],
                force_text(order_item['OrderItemCustomers'][0]['OrderCustomerId']),
                OperationStatus.OK,
                None
            )
        insurances_by_id[order_item_id].refund_operation_id = force_text(order_item['OrderItemId'])
        insurances_by_id[order_item_id].refund_amount = order_item['Amount']
        insurances_by_id[order_item_id].is_external = order_item['IsExternallyLoaded']
        insurances_by_id[order_item_id].reference_id = order_item.get('AgentReferenceId')

    result.passengers = list(passenger_by_id.values())
    result.tickets = list(ticket_by_blank_id.values())
    result.insurances = list(insurances_by_id.values())
    return result


def get_sorted_refund_ticket_items(order_items):
    return sorted((order_item for order_item in order_items if order_item.is_refund and order_item.is_ticket),
                  key=lambda o: o.creation_dt)


def get_status(reference_id, order_items, buy_order_item, sorted_refund_ticket_items):
    if reference_id:
        reference_order_item = next(item for item in order_items if item['AgentReferenceId'] == reference_id)
        return IM_OPERATION_STATUS_TO_OPERATION_STATUS[reference_order_item['SimpleOperationStatus']]
    elif sorted_refund_ticket_items:
        return IM_OPERATION_STATUS_TO_OPERATION_STATUS[sorted_refund_ticket_items[-1]['SimpleOperationStatus']]
    return IM_OPERATION_STATUS_TO_OPERATION_STATUS[buy_order_item['SimpleOperationStatus']]


class OrderItem(UserDict):
    @cached_property
    def reserved_to(self):
        return parse_datetime(self.get('ConfirmTimeLimit'))

    @cached_property
    def expire_set_er(self):
        return parse_datetime(self.get('ElectronicRegistrationExpirationDateTime'))

    @cached_property
    def is_buy(self):
        return self['OperationType'].lower() == 'purchase'

    @cached_property
    def is_refund(self):
        return self['OperationType'].lower() == 'return'

    @cached_property
    def is_ticket(self):
        return 'RailwayFullOrderItemInfo' in self['$type']

    @cached_property
    def is_insurance(self):
        return 'RailwayInsuranceFullOrderItemInfo' in self['$type']

    @cached_property
    def creation_dt(self):
        return parse_datetime(self['CreateDateTime'])
