# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta, datetime

from pytz import UTC

from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.train_partners.base.get_route_info import RouteInfo, RouteStopInfo
from travel.rasp.train_api.train_partners.im.base import get_im_response, measurable, IM_DATETIME_FORMAT
from travel.rasp.train_api.train_purchase.core.models import TrainOrder

log = logging.getLogger(__name__)

IM_TRAIN_ROUTE_METHOD = 'Railway/V1/Search/TrainRoute'


def get_route_info_for_order(order, timeout=None):
    if not getattr(order, 'station_from', None) or not getattr(order, 'station_to', None):
        TrainOrder.fetch_stations([order])

    return get_route_info(
        train_number=order.train_number,
        station_from=order.station_from,
        station_to=order.station_to,
        departure=order.departure,
        partner_credential_id=order.partner_credential_id,
        timeout=timeout,
    )


@measurable()
def get_route_info(train_number, station_from, station_to, departure, partner_credential_id, timeout=None):
    departure_railway = UTC.localize(departure).astimezone(get_railway_tz_by_point(station_from))
    response = _get_route_info(
        {
            'TrainNumber': train_number,
            'Origin': station_from.express_id,
            'Destination': station_to.express_id,
            'DepartureDate': departure_railway.strftime(IM_DATETIME_FORMAT),
        },
        credential_id=partner_credential_id,
        timeout=timeout,
    )
    routes = response['Routes'] if response else None

    if routes and routes[0]['RouteStops']:
        route_stops = routes[0]['RouteStops']

        first_departure_datetime = _get_first_departure_datetime(
            route_stops=route_stops,
            origin_code=station_from.express_id,
            departure_railway=departure_railway,
        )
        if first_departure_datetime:
            first_departure_datetime = first_departure_datetime.astimezone(UTC).replace(tzinfo=None)

        first_stop = RouteStopInfo(
            station_express_code=route_stops[0]['StationCode'],
            station_name=route_stops[0]['StationName'],
            departure_datetime=first_departure_datetime,
        )

        last_stop = RouteStopInfo(
            station_express_code=route_stops[-1]['StationCode'],
            station_name=route_stops[-1]['StationName'],
            departure_datetime=None,
        )

        return RouteInfo(first_stop, last_stop)
    return None


def _get_route_info(params, credential_id, timeout=None):
    response = get_im_response(IM_TRAIN_ROUTE_METHOD, params, credential_id=credential_id, timeout=timeout)
    return response


def _get_first_departure_datetime(route_stops, origin_code, departure_railway):
    cur_stop = None
    departure_date = departure_railway.date()
    departure_time_raw = route_stops[0]['DepartureTime']

    for stop in reversed(route_stops[:-1]):
        if not cur_stop:
            if stop['StationCode'] == origin_code:
                cur_stop = stop
        else:
            if stop['DepartureTime'] >= cur_stop['DepartureTime']:
                departure_date = departure_date - timedelta(days=1)
            cur_stop = stop

    if not cur_stop:
        log.error('Origin express code %s was not found.', origin_code)
        return None

    try:
        departure_time = datetime.strptime(departure_time_raw, '%H:%M').time()
    except ValueError:
        log.error('DepartureTime "%s" is not match', departure_time_raw)
        return None

    return departure_railway.tzinfo.localize(datetime.combine(departure_date, departure_time))
