# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.encoding import force_text

from travel.rasp.train_api.train_partners.base.insurance.checkout import InsuranceCheckoutException
from travel.rasp.train_api.train_partners.im.base import get_im_response, measurable

IM_INSURANCE_CHECKOUT_METHOD = 'Insurance/V1/Travel/Checkout'


@measurable()
def checkout(order, timeout=None):
    if not order.current_partner_data or not order.current_partner_data.operation_id:
        raise InsuranceCheckoutException('operation_id is not defined in order {}'.format(order.uid))

    if not all([passenger.insurance for passenger in order.passengers]):
        raise InsuranceCheckoutException('order {}: not every passenger has insurance'.format(order.uid))

    operation_id_by_customer_id = {
        passenger.customer_id: _checkout_passenger(order, passenger, timeout) for passenger in order.passengers
    }
    return operation_id_by_customer_id


def _checkout_passenger(order, passenger, timeout=None):
    order_item_id = int(order.current_partner_data.operation_id)
    order_customer_id = int(passenger.customer_id)
    params = {
        'MainServiceReference': {
            '$type': 'ApiContracts.Insurance.V1.Messages.Travel.MainServiceReferenceInternal, ApiContracts',
            'OrderItemId': order_item_id,
            'OrderCustomerId': order_customer_id,
        },
        'Product': {
            '$type':
                'ApiContracts.Insurance.V1.Messages.Travel.RailwayInsuranceTravelCheckoutRequest, ApiContracts',
            'Package': passenger.insurance.package,
        },
        'Provider': passenger.insurance.provider,
        'Company': passenger.insurance.company,
        'AgentPaymentId': None,
    }
    result = _checkout(params, credential_id=order.partner_credential_id, timeout=timeout)
    return force_text(result['OrderItemId'])


def _checkout(params, credential_id, timeout=None):
    response = get_im_response(IM_INSURANCE_CHECKOUT_METHOD, params, credential_id=credential_id, timeout=timeout)
    return response
