# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from decimal import Decimal

from django.utils.encoding import force_text

from common.utils.field_masker import FieldMasker
from travel.rasp.train_api.train_partners.base.refund import TicketRefundResult, RefundResult, check_refund_error
from travel.rasp.train_api.train_partners.im.base import get_im_response, measurable

REFUND_ENDPOINT = 'Order/V1/Reservation/AutoReturn'


field_masker = FieldMasker(mask_fields={
    'ServiceAutoReturnRequest': {
        'CheckDocumentNumber': 1
    }
})


@measurable()
def make_refund(order, blank_id, doc_id, reference_id):
    operation_id = order.current_partner_data.operation_id
    with check_refund_error():
        response = get_im_response(REFUND_ENDPOINT, params={
            'ServiceAutoReturnRequest': {
                '$type': 'ApiContracts.Railway.V1.Messages.Return.RailwayAutoReturnRequest, ApiContracts',
                'OrderItemBlankIds': [int(blank_id)] if blank_id else None,
                'CheckDocumentNumber': doc_id,
                'OrderItemId': int(operation_id),
                'AgentReferenceId': reference_id
            }
        }, credential_id=order.partner_credential_id, field_masker=field_masker)
    return _build_result_from_refund(response)


def _build_result_from_refund(response):
    refund_operation_id = force_text(response['ServiceReturnResponse']['ReturnOrderItemId'])
    refund_by_blank_id = {}
    for raw_blank in response['ServiceReturnResponse']['Blanks']:
        blank_id = force_text(raw_blank['PurchaseOrderItemBlankId'])
        refund_by_blank_id[blank_id] = TicketRefundResult(
            amount=Decimal(raw_blank['Amount']),
            refund_blank_id=force_text(raw_blank['ReturnOrderItemBlankId']),
            refund_operation_id=refund_operation_id
        )
    return RefundResult(refund_by_blank_id=refund_by_blank_id)
