# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re
from datetime import datetime

from django.utils.encoding import force_text

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.dynamic_settings.default import conf
from common.models.geo import Country
from common.utils.date import MSK_TZ, UTC_TZ
from common.utils.field_masker import FieldMasker
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.train_partners.base import create_tax
from travel.rasp.train_api.train_partners.base.reserve_tickets import (
    ReserveResponse, get_notices, RESERVE_TICKET_ENDPOINT_NAME, ReserveRequestPassenger
)
from travel.rasp.train_api.train_partners.im.base import (
    IM_DATETIME_FORMAT, get_im_response, DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE, DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE,
    measurable, ImError
)
from travel.rasp.train_api.train_purchase.core.enums import (
    AgeGroup, Gender, GenderChoice, LoyaltyCardType, PlacesOption, PlacesType, Arrangement
)

log = logging.getLogger(__name__)

CREATE_RESERVATION_METHOD = 'Order/V1/Reservation/Create'

AGE_GROUP_TO_CATEGORY_VALUE = {
    AgeGroup.ADULTS: 'Adult',
    AgeGroup.CHILDREN: 'Child',
    AgeGroup.BABIES: 'BabyWithoutPlace'
}
COACH_TYPE_TO_CARTYPE_VALUE = {
    CoachType.PLATZKARTE: 'ReservedSeat',
    CoachType.COMPARTMENT: 'Compartment',
    CoachType.SUITE: 'Luxury',
    CoachType.COMMON: 'Shared',
    CoachType.SITTING: 'Sedentary',
    CoachType.SOFT: 'Soft',
}
GENDER_CHOICE_TO_CABINGENDERKIND_VALUE = {
    GenderChoice.MALE: 'Male',
    GenderChoice.FEMALE: 'Female',
    GenderChoice.MIXED: 'Mixed'
}
CABIN_PLACE_DEMANDS = {
    Arrangement.COMPARTMENT: 'InOneCabin',
    Arrangement.NEAREST: 'InOneCabin',
    Arrangement.NOT_SIDE: 'NoSidePlaces',
    Arrangement.SECTION: 'InOneCompartment'
}
GENDER_TO_SEX_VALUE = {
    Gender.MALE: 'Male',
    Gender.FEMALE: 'Female'
}
SEX_VALUE_TO_GENDER = {v: k for k, v in GENDER_TO_SEX_VALUE.items()}
LOYALTY_CARD_TYPE_TO_CARDTYPE_VALUE = {
    LoyaltyCardType.RZHD_BONUS: 'RzhdBonus',
    LoyaltyCardType.UNIVERSAL: 'UniversalRzhdCard'
}

PLACES_OPTION_TO_ADDITIONAL_PLACE_REQUIREMENTS_VALUE = {
    PlacesOption.WITH_PET: 'WithPetsPlaces'
}
PLACE_TYPE_TO_PLACES_TYPE = {
    'Foldable': PlacesType.FOLDING,
    'Invalids': PlacesType.FOR_IMPAIRED_PASSENGER,
    'Lower': PlacesType.LOWER_TIER,
    'Middle': PlacesType.MIDDLE_TIER,
    'MotherAndBaby': PlacesType.FOR_MOTHER_WITH_CHILD,
    'NearPassengersWithPets': PlacesType.NEAR_PASSENGERS_WITH_PETS,
    'NearPlayground': PlacesType.NEAR_PLAYGROUND,
    'NearTable': PlacesType.NEAR_TABLE,
    'NearTablePlayground': PlacesType.NEAR_PLAYGROUND_TABLE,
    'Negotiations': PlacesType.CONFERENCE_ROOM,
    'NotNearTable': PlacesType.NOT_NEAR_TABLE,
    'SeparateCompartment': PlacesType.LASTOCHKA_COMPARTMENT,
    'Upper': PlacesType.UPPER_TIER,
    'WithChild': PlacesType.FOR_PASSENGER_WITH_CHILDREN,
    'WithPets': PlacesType.FOR_PASSENGER_WITH_PET
}
STOREY_TO_CAR_STOREY = {
    0: 'NoValue',
    1: 'First',
    2: 'Second',
}

ADDITIONAL_PLACE_REQUIREMENTS_IN_RESERVATION_CREATE = {
    'MotherAndBabyPlaces',
    'WithBabyPlaces',
    'WithPetsPlaces',
    'Usual',
    'UsualNearTheTable',
    'AnyNearTheTable',
    'AnyNotNearTheTable',
    'NearThePlayground',
    'NearThePlaygroundAndNotTheTable',
    'NearThePlaygroundAndTheTable',
    'NearThePlacesWithPets',
    'FoldablePlace',
    'Forward',
    'Backward',
    'NearWindow',
    'UnfoldablePlace',
    'NearTheTableAndBackward',
    'NearTheTableAndForward',
    'WithoutTableAndBackward',
    'WithoutTableAndForward',
    'WithoutWindowAndBackward',
    'WithoutWindowAndForward',
    'SingleAndForward',
}
ADDITIONAL_PLACE_REQUIREMENTS = dict(
    PLACES_OPTION_TO_ADDITIONAL_PLACE_REQUIREMENTS_VALUE,
    **{r: r for r in ADDITIONAL_PLACE_REQUIREMENTS_IN_RESERVATION_CREATE}
)

NO_VALUE = 'NoValue'

field_masker = FieldMasker(mask_fields={
    'Customers': [{'DocumentNumber': 1, 'Birthday': 1, 'BirthDate': 1}],
    'ReservationResults': [{'Passengers': [{'Birthday': 1, 'BirthDate': 1}]}]})


def _get_notice_parts(raw_notice):
    if not raw_notice:
        return []

    notice = re.sub(' +', ' ', raw_notice)  # Заменяем двойные/тройные пробелы на одинарный
    parts = [part.strip() for part in notice.split('.')]
    return [p for p in parts if p]


def _parse_tariff_info(category, tariff_type):
    if category == AGE_GROUP_TO_CATEGORY_VALUE[AgeGroup.BABIES]:
        return TariffInfo.objects.get(code=TariffInfo.BABY_CODE)

    return next((
        tariff_info for tariff_info in TariffInfo.objects.all()
        if tariff_type in tariff_info.im_response_codes_list
    ), None)


class WrongPlaces(Exception):
    pass


class PlacesInfo(object):
    """
    Логика преобразования требований к местам, которые пришли с фронта, в то что передаем в ИМ
    """
    gender = None
    storey = None
    lower_places_quantity = None
    upper_places_quantity = None
    max_place_number = None
    min_place_number = None
    cabin_place_demands = None  # В одном отсеке, не боковые, в одном купе
    additional_place_requirements = None

    def __init__(self, coach_type, gender,
                 number_and_is_upper_for_places, place_demands, requirements, additional_place_requirements):
        self.gender = gender

        if place_demands and not additional_place_requirements:
            # Переносим единственно возможный признак "с животными" для обратной совместимости
            additional_place_requirements = place_demands
        self.additional_place_requirements = additional_place_requirements

        self._make_places_location_params(coach_type, number_and_is_upper_for_places, requirements)

    def _make_places_location_params(self, coach_type, number_and_is_upper_for_places, requirements):
        if number_and_is_upper_for_places:
            if len(number_and_is_upper_for_places) != len(set(place['number']
                                                              for place in number_and_is_upper_for_places)):
                raise WrongPlaces('There are some places with same numbers: {}'.format(
                    [place['number'] for place in number_and_is_upper_for_places]))

            self.min_place_number = min(place['number'] for place in number_and_is_upper_for_places)
            self.max_place_number = max(place['number'] for place in number_and_is_upper_for_places)

        if requirements:
            self.storey = requirements['storey']

            if isinstance(requirements['arrangement'], Arrangement):
                self.cabin_place_demands = requirements['arrangement']
            else:
                self.cabin_place_demands = (
                    Arrangement(requirements['arrangement'])
                    if requirements['arrangement']
                    else None
                )

            if requirements['count']:
                # в апи партнера upper для сидячих мест это около прохода, lower - около окна
                upper_places_quantity = requirements['count']['upper'] or requirements['count']['near_passage']
                lower_places_quantity = requirements['count']['bottom'] or requirements['count']['near_window']
                if lower_places_quantity or upper_places_quantity:
                    self.upper_places_quantity = upper_places_quantity
                    self.lower_places_quantity = lower_places_quantity
        elif coach_type not in (CoachType.SITTING, CoachType.COMMON) and number_and_is_upper_for_places:
            upper_places_quantity = len([place for place in number_and_is_upper_for_places if place['is_upper']])
            self.upper_places_quantity = upper_places_quantity
            self.lower_places_quantity = len(number_and_is_upper_for_places) - upper_places_quantity

    @property
    def place_range(self):
        if self.min_place_number is None or self.max_place_number is None:
            return None
        return {
            'From': self.min_place_number,
            'To': self.max_place_number,
        }


class ReservationManager(object):
    def __init__(self, passengers, train_number, when, arrival_point_code, departure_point_code,
                 coach_number, coach_type, service_class, bedding, electronic_registration, places_info,
                 partner_credential_id, give_child_without_place, international_service_class, is_cppk):
        self.passengers = passengers
        self.train_number = train_number
        self.when = when
        self.arrival_point_code = arrival_point_code
        self.departure_point_code = departure_point_code
        self.coach_number = coach_number
        self.coach_type = coach_type
        self.service_class = service_class
        self.bedding = bedding
        self.electronic_registration = electronic_registration
        self.places_info = places_info
        self.partner_credential_id = partner_credential_id
        self.give_child_without_place = give_child_without_place
        self.international_service_class = international_service_class
        self.is_cppk = is_cppk

    @classmethod
    def from_order_data(cls, order_data):
        places_info = PlacesInfo(
            coach_type=order_data['coach_type'],
            gender=order_data['gender'],
            number_and_is_upper_for_places=order_data['places'],
            place_demands=order_data['place_demands'],
            requirements=order_data['requirements'],
            additional_place_requirements=order_data['additional_place_requirements'],
        )
        passengers = [
            ReserveRequestPassenger(
                age_group=passenger['age_group'],
                birth_date=passenger['birth_date'],
                citizenship_country=passenger['citizenship_country'],
                document_number=passenger['doc_id'],
                document_type=passenger['doc_type'],
                first_name=passenger['first_name'],
                gender=passenger['sex'],
                last_name=passenger['last_name'],
                loyalty_cards=passenger['loyalty_cards'],
                patronymic=passenger['patronymic'],
                tariff_info=passenger['tariff_info'],
                phone=passenger['phone'],
                email=passenger['email'],
            )
            for passenger in order_data['passengers']
        ]
        return ReservationManager(
            passengers=passengers,
            train_number=order_data['train_number'],
            when=order_data['departure'].astimezone(get_railway_tz_by_point(order_data['station_from'])),
            arrival_point_code=order_data['station_to'].get_code('express'),
            departure_point_code=order_data['station_from'].get_code('express'),
            coach_number=order_data['coach_number'],
            coach_type=order_data['coach_type'],
            service_class=order_data['service_class'],
            bedding=order_data['bedding'],
            electronic_registration=order_data['electronic_registration'],
            places_info=places_info,
            partner_credential_id=order_data['partner_credential_id'],
            give_child_without_place=order_data['give_child_without_place'],
            international_service_class=order_data['international_service_class'],
            is_cppk=order_data['is_cppk'],
        )

    @classmethod
    def from_order(cls, order, order_info):
        places_info = PlacesInfo(
            coach_type=order.coach_type,
            gender=order.gender,
            number_and_is_upper_for_places=order.rebooking_info.places,
            place_demands=order.rebooking_info.place_demands,
            requirements=order.rebooking_info.requirements,
            additional_place_requirements=order.rebooking_info.additional_place_requirements,
        )

        passengers = []
        doc_id_n_birth_date_by_customer_id = {p.customer_id: (p.doc_id, p.birth_date) for p in order_info.passengers}
        for passenger in order.passengers:
            doc_id, birth_date = doc_id_n_birth_date_by_customer_id[passenger.customer_id]

            passengers.append(ReserveRequestPassenger(
                age_group=passenger.rebooking_info.age_group,
                birth_date=birth_date,
                citizenship_country=Country.objects.get(id=passenger.citizenship_country_id),
                document_number=doc_id,
                document_type=passenger.doc_type,
                first_name=passenger.first_name,
                gender=passenger.sex,
                last_name=passenger.last_name,
                loyalty_cards=passenger.rebooking_info.loyalty_cards,
                patronymic=passenger.patronymic,
                tariff_info=passenger.rebooking_info.tariff_info,
                phone=passenger.phone,
                email=passenger.email,
            ))

        return ReservationManager(
            passengers=passengers,
            train_number=order.train_number,
            when=UTC_TZ.localize(order.departure).astimezone(get_railway_tz_by_point(order.station_from)),
            arrival_point_code=order.station_to.get_code('express'),
            departure_point_code=order.station_from.get_code('express'),
            coach_number=order.coach_number,
            coach_type=order.coach_type,
            service_class=order.rebooking_info.service_class,
            bedding=order.rebooking_info.bedding,
            electronic_registration=order.rebooking_info.electronic_registration,
            places_info=places_info,
            partner_credential_id=order.partner_credential_id,
            give_child_without_place=order.rebooking_info.give_child_without_place,
            international_service_class=order.rebooking_info.international_service_class,
            is_cppk=order.rebooking_info.is_cppk,
        )

    def _build_request_params(self):
        params = {
            'Customers': [
                {
                    '$type': 'ApiContracts.Order.V1.Reservation.OrderFullCustomerRequest, ApiContracts',
                    'Index': i,

                    'DocumentNumber': passenger.document_number,
                    'FirstName': passenger.first_name,
                    'MiddleName': passenger.patronymic,
                    'LastName': passenger.last_name,
                    'DocumentType': DOCUMENT_TYPE_TO_DOCUMENTTYPE_VALUE[passenger.document_type],
                    'CitizenshipCode': passenger.citizenship_country.code,
                    'Sex': GENDER_TO_SEX_VALUE[passenger.gender],
                    'Birthday': passenger.birth_date.strftime(IM_DATETIME_FORMAT)
                }
                for i, passenger in enumerate(self.passengers)
            ],
            'ReservationItems': [
                {
                    '$type': 'ApiContracts.Railway.V1.Messages.Reservation.RailwayReservationRequest, ApiContracts',
                    'Index': 0,

                    # поезд
                    'OriginCode': self.departure_point_code,
                    'DestinationCode': self.arrival_point_code,
                    'DepartureDate': self.when.strftime(IM_DATETIME_FORMAT),
                    'TrainNumber': self.train_number,

                    # требования к вагону и местам
                    'CarType': COACH_TYPE_TO_CARTYPE_VALUE.get(self.coach_type, 'Unknown'),
                    'LowerPlaceQuantity': self.places_info.lower_places_quantity,
                    'UpperPlaceQuantity': self.places_info.upper_places_quantity,
                    'Bedding': self.bedding,
                    'PlaceRange': self.places_info.place_range,
                    'CabinPlaceDemands': CABIN_PLACE_DEMANDS.get(self.places_info.cabin_place_demands, 'NoValue'),
                    'ServiceClass': self.service_class,
                    'InternationalServiceClass': self.international_service_class,
                    'CabinGenderKind': GENDER_CHOICE_TO_CABINGENDERKIND_VALUE.get(self.places_info.gender, 'NoValue'),
                    'AdditionalPlaceRequirements': ADDITIONAL_PLACE_REQUIREMENTS.get(
                        self.places_info.additional_place_requirements, 'NoValue'),
                    'CarStorey': STOREY_TO_CAR_STOREY.get(self.places_info.storey, 'NoValue'),
                    'GiveAdditionalTariffForChildIfPossible': self.give_child_without_place,

                    # регистрация и оплата
                    'SetElectronicRegistration': self.electronic_registration,
                    'ProviderPaymentForm': 'Card',

                    # пассажиры
                    'Passengers': [
                        {
                            'OrderCustomerIndex': i,
                            'Category': AGE_GROUP_TO_CATEGORY_VALUE.get(passenger.age_group, 'Adult'),
                            'PreferredAdultTariffType': passenger.tariff_info.im_request_code,
                            'Phone': passenger.phone if conf.TRAIN_PURCHASE_TRANSMIT_PHONE_EMAIL_TO_IM else None,
                            'ContactEmailOrPhone': (passenger.email if conf.TRAIN_PURCHASE_TRANSMIT_PHONE_EMAIL_TO_IM
                                                    else None),
                            'IsMarketingNewsletterAllowed': False,
                            'RailwayBonusCards': [{
                                'CardType': LOYALTY_CARD_TYPE_TO_CARDTYPE_VALUE[card['type']],
                                'CardNumber': card['number']
                            } for card in passenger.loyalty_cards] if passenger.loyalty_cards else None
                        }
                        for i, passenger in enumerate(self.passengers)
                    ]
                }
            ]
        }

        if not conf.TRAIN_PURCHASE_IM_CHECK_DOUBLE_BOOKING:
            params['CheckDoubleBooking'] = False

        if self.is_cppk:
            params = self._prepare_cppk_reservation(params)

        return self._add_coach_number(params)

    def _prepare_cppk_reservation(self, params):
        for item in params['ReservationItems']:
            item['GiveAdditionalTariffForChildIfPossible'] = False
            item['OnRequestMeal'] = False
            item['Bedding'] = False
            item['CabinGenderKind'] = NO_VALUE
            item['CabinPlaceDemands'] = NO_VALUE
            item['ClientCharge'] = None
            item['InternationalServiceClass'] = None
            item['SpecialPlacesDemand'] = NO_VALUE
            item['CarStorey'] = STOREY_TO_CAR_STOREY[0]
            item['AdditionalPlaceRequirements'] = NO_VALUE

            for passenger in item['Passengers']:
                # Category and PreferredAdultTariffType should be set on the front
                passenger['IsNonRefundableTariff'] = False
                passenger['IsInvalid'] = False
                passenger['DisabledPersonId'] = None
                passenger['TransitDocument'] = NO_VALUE
                passenger['TransportationRequirement'] = None
                passenger['RailwayBonusCards'] = None

        return params

    def _add_coach_number(self, params):
        if self.coach_number:
            for reservation_item in params['ReservationItems']:
                reservation_item['CarNumber'] = self.coach_number
        return params

    def _parse_response(self, partner_data):
        reservation_result_data = partner_data['ReservationResults'][0]
        notice_parts = _get_notice_parts(reservation_result_data.get('TimeDescription'))
        special_notice, time_notice = get_notices(notice_parts)
        customers_by_index = {customer['Index']: customer for customer in partner_data['Customers']}
        blanks_by_id = {blank['OrderItemBlankId']: blank for blank in reservation_result_data['Blanks']}

        response = ReserveResponse(
            im_order_id=partner_data['OrderId'],
            amount=reservation_result_data['Amount'],
            arrival_station_title=reservation_result_data['DestinationStation'],
            coach_owner=reservation_result_data['Carrier'],
            compartment_gender=self.places_info.gender,  # ИМ не возвращает гендерный признак места
            departure_station_title=reservation_result_data['OriginStation'],
            operation_id=force_text(reservation_result_data['OrderItemId']),
            reserved_to=MSK_TZ.localize(datetime.strptime(reservation_result_data['ConfirmTill'], IM_DATETIME_FORMAT)),
            special_notice=special_notice,
            time_notice=time_notice,
            is_three_hours_reservation_available=reservation_result_data.get('IsThreeHoursReservationAvailable', False),
            is_suburban=reservation_result_data.get('IsSuburban', False),
            coach_number=reservation_result_data['CarNumber'],
            is_only_full_return_possible=reservation_result_data.get('IsOnlyFullReturnPossible', False),
        )

        for passenger_data in reservation_result_data['Passengers']:
            blank_id = passenger_data['OrderItemBlankId']
            blank_data = blanks_by_id[blank_id]
            fare_info = blank_data.get('FareInfo')
            tariff_info_data = blank_data['TariffInfo']
            service_price = blank_data['ServicePrice']
            customer = customers_by_index[passenger_data['OrderCustomerReferenceIndex']]
            places_data = passenger_data['PlacesWithType']
            tariff_vat, service_vat, commission_fee_vat = _fill_vats(passenger_data, blanks_by_id)
            response.add_ticket(
                index=customer['Index'],
                customer_id=force_text(customer['OrderCustomerId']),
                amount=passenger_data['Amount'],
                service_price=service_price,
                blank_id=force_text(blank_id),
                carrier_inn=fare_info.get('CarrierTin') if fare_info else None,
                commission_fee_vat=commission_fee_vat,
                passenger_birth_date=datetime.strptime(customer['BirthDate'], IM_DATETIME_FORMAT),
                passenger_citizenship_country=Country.objects.get(code=customer['CitizenshipCode']),
                passenger_document_number=customer['DocumentNumber'],
                passenger_document_type=DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE.get(customer['DocumentType']),
                passenger_first_name=customer['FirstName'],
                passenger_gender=SEX_VALUE_TO_GENDER.get(customer['Sex']),
                passenger_last_name=customer['LastName'],
                passenger_patronymic=customer['MiddleName'],
                places=[place_data['Number'] for place_data in places_data],
                places_type=(PLACE_TYPE_TO_PLACES_TYPE.get(places_data[0]['Type'])
                             if places_data else
                             None),  # XXX мы считаем, что у всех мест один общий тип, а в ИМ у каждого места свой
                raw_tariff_title=tariff_info_data and tariff_info_data['TariffName'],
                service_vat=service_vat,
                tariff_info=_parse_tariff_info(passenger_data['Category'], blank_data['TariffType']),
                tariff_vat=tariff_vat,
            )

        return response

    @measurable(RESERVE_TICKET_ENDPOINT_NAME)
    def reserve(self, timeout=None):
        if conf.TRAIN_PURCHASE_TRANSMIT_PHONE_EMAIL_TO_IM:
            return self._reserve_with_retries(timeout)
        else:
            result = get_im_response(CREATE_RESERVATION_METHOD, self._build_request_params(),
                                     credential_id=self.partner_credential_id, field_masker=field_masker,
                                     timeout=timeout)
            return self._parse_response(result)

    def _reserve_with_retries(self, timeout=None):
        def _empty_passenger_phones():
            self.passengers = map(lambda p: p._replace(phone=None), self.passengers)

        def _empty_passenger_emails():
            self.passengers = map(lambda p: p._replace(email=None), self.passengers)

        current_iteration = 0
        max_iteration = 3

        while True:
            try:
                result = get_im_response(CREATE_RESERVATION_METHOD, self._build_request_params(),
                                         credential_id=self.partner_credential_id, field_masker=field_masker,
                                         timeout=timeout)
                return self._parse_response(result)
            except ImError as im_error:
                current_iteration += 1
                if current_iteration > max_iteration:
                    raise
                if im_error.is_invalid_passenger_phone():
                    log.warning('Not every passenger phone is valid')
                    _empty_passenger_phones()
                elif im_error.is_invalid_passenger_email():
                    log.warning('Not every passenger email is valid')
                    _empty_passenger_emails()
                else:
                    raise


def _fill_vats(passenger_data, blanks_by_id):
    amount = passenger_data['Amount']
    if not amount:
        return None, None, None

    blank_id = passenger_data['OrderItemBlankId']
    vats = blanks_by_id[blank_id]['VatRateValues']
    tariff_vat = create_tax(amount=vats[0]['Value'], rate=vats[0]['Rate'])
    service_vat = create_tax(amount=vats[1]['Value'], rate=vats[1]['Rate'])
    commission_fee_vat = create_tax(amount=vats[2]['Value'], rate=vats[2]['Rate']) if len(vats) > 2 else None

    return tariff_vat, service_vat, commission_fee_vat
