# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from decimal import Decimal, ROUND_HALF_UP
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.functional import cached_property

from common.apps.train.models import TariffInfo
from common.apps.train.tariff_error import TariffError
from common.apps.train_order.enums import CoachType
from common.dynamic_settings.default import conf
from common.models.currency import Price
from common.settings.utils import define_setting, bool_converter
from travel.rasp.train_api.tariffs.train.base.availability_indication import AvailabilityIndication
from travel.rasp.train_api.tariffs.train.base.models import (
    AVAILABILITY_INDICATION_TO_TARIFF_ERROR, PlaceReservationType,
    check_tariff_coach_type, check_place_reservation_types,
)
from travel.rasp.train_api.tariffs.train.base.segments import in_suburban_search
from travel.rasp.train_api.tariffs.train.im.parser import IM_PLACE_RESERVATION_TYPES, IM_AVAILABILITY_INDICATION_TYPES
from travel.rasp.train_api.train_partners.base.train_details import FACILITIES_BY_IM
from travel.rasp.train_api.train_partners.base.train_details.parsers import (
    BaseClassDetails, CurrentClassDetailsError, BaseTrainDetails, BaseCoachDetails, init_places, parse_place_number,
    PlaceDetails
)
from travel.rasp.train_api.train_partners.base.train_details.reservation_variants import ReservationVariant
from travel.rasp.train_api.train_partners.base.train_details.service_classes import get_service_class
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.tariff_calculator import (
    get_pet_place_tariff, get_bedding_tariff
)
from travel.rasp.train_api.train_partners.base.train_details.utils import set_pets
from travel.rasp.train_api.train_partners.im.base import (
    DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE, parse_datetime, PROVIDER_P2, PROVIDER_P1
)
from travel.rasp.train_api.train_purchase.core.directories import TariffCategories, TariffTypes
from travel.rasp.train_api.train_purchase.core.enums import LoyaltyCardType, RoutePolicy
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost

log = logging.getLogger(__name__)

LOYALTY_CARDS = {
    'RzhdBonus': LoyaltyCardType.RZHD_BONUS.value,
    'UniversalRzhdCard': LoyaltyCardType.UNIVERSAL.value,
}

ROUTE_POLICY = {
    'Internal': RoutePolicy.INTERNAL.value,
    'International': RoutePolicy.INTERNATIONAL.value,
    'Finland': RoutePolicy.FINLAND.value,
}

COACH_TYPE_WITH_REQUIREMENTS = {CoachType.COMPARTMENT.value, CoachType.PLATZKARTE.value, CoachType.SITTING.value}
USUAL_PLACE_TYPE = 'Usual'

define_setting('TRAIN_DETAILS_IGNORE_CAR_TRAIN_NUMBER', default=True, converter=bool_converter)


def parse_place_reservation_type(im_coach_group):
    im_place_reservation_type = im_coach_group['PlaceReservationType']
    return IM_PLACE_RESERVATION_TYPES.get(im_place_reservation_type, PlaceReservationType.UNKNOWN)


class TrainDetails(BaseTrainDetails):
    def __init__(self, raw_data, segment, query):
        self.allowed_document_types = [DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE[t].value
                                       for t in raw_data['AllowedDocumentTypes']
                                       if t in DOCUMENTTYPE_VALUE_TO_DOCUMENT_TYPE]
        self.start_number = raw_data['TrainInfo']['TrainNumber'].strip()
        self.ticket_number = (raw_data['TrainInfo']['DisplayTrainNumber'].strip().replace('*', '') or
                              self.start_number)
        self.electronic_ticket = any(c['HasElectronicRegistration'] for c in raw_data['Cars'])
        self.brand = raw_data['TrainInfo'].get('TrainName') or None  # TODO: check this
        self.coaches = []
        self.broken_coaches = []
        self.schemas = []

        self.raw_train_category = raw_data['TrainInfo']['TrainDescription'].strip()
        self.raw_train_name = raw_data['TrainInfo']['TrainName']

        train_categories = self.raw_train_category.split()
        self.is_firm = 'ФИРМ' in train_categories

        self.station_from = query.station_from
        self.express_from = query.express_from
        self.station_to = query.station_to
        self.express_to = query.express_to

        self.departure = parse_datetime(raw_data['TrainInfo']['DepartureDateTime'], query.station_from)
        self.arrival = parse_datetime(raw_data['TrainInfo']['ArrivalDateTime'], query.station_to)

        self.when = query.when
        self.contract = query.contract
        self.query = query
        self.is_suburban = raw_data['TrainInfo']['IsSuburban']
        self.im_initial_station_name = raw_data['TrainInfo']['InitialStationName']
        self.im_final_station_name = raw_data['TrainInfo']['FinalStationName']
        self.yandex_uid = query.yandex_uid
        self.route_policy = ROUTE_POLICY.get(raw_data['RoutePolicy'])

        self.provider = raw_data['TrainInfo'].get('Provider', PROVIDER_P1)
        self.is_cppk = conf.TRAIN_PURCHASE_ENABLE_CPPK_CONDITIONS and self.provider == PROVIDER_P2
        self.tariff_categories = self._get_categories()
        self.is_covid_certificate_required = (
            conf.TRAIN_PURCHASE_COVID_CERT_REQ_COUNTRIES and
            query.station_from.country_id != query.station_to.country_id and
            (query.station_from.country_id in conf.TRAIN_PURCHASE_COVID_CERT_REQ_COUNTRIES or
             query.station_to.country_id in conf.TRAIN_PURCHASE_COVID_CERT_REQ_COUNTRIES)
        )

        coaches = []
        coach_by_key = {}
        for car_raw_data in raw_data['Cars']:
            try:
                class_details = CurrentClassDetails(car_raw_data, self)
            except CurrentClassDetailsError as error:
                log.error(force_text(error))
                continue
            else:
                coach = class_details.get_coach()
                coaches.append(coach)

        self.add_tariffs(
            coaches,
            query.raw_query_data.get('icookie'),
            query.raw_query_data.get('bandit_type'),
            in_suburban_search(segment),
            query.raw_query_data
        )

        for coach in coaches:
            is_valid, errors = coach.validate()
            if not is_valid:
                coach.errors = errors
                self.broken_coaches.append(coach)
                continue

            try:
                base_coach = coach_by_key[coach.key]
            except KeyError:
                coach_by_key[coach.key] = coach
            else:
                for k in base_coach.place_counts:
                    base_coach.place_counts[k] += coach.place_counts[k]
                for t, ps in coach.places_by_type.items():
                    base_coach.places_by_type.setdefault(t, [])
                    base_coach.places_by_type[t].extend(ps)

                base_coach.places_by_compartment.extend(coach.places_by_compartment)
                base_coach.places.extend(coach.places)
                base_coach.current_class_details.min_tariff = min(base_coach.current_class_details.min_tariff,
                                                                  coach.current_class_details.min_tariff)
                base_coach.current_class_details.max_tariff = max(base_coach.current_class_details.max_tariff,
                                                                  coach.current_class_details.max_tariff)
                base_coach.has_non_refundable_tariff = (base_coach.has_non_refundable_tariff
                                                        or coach.has_non_refundable_tariff)

        coaches = list(coach_by_key.values())
        coaches.sort(key=lambda x: x.key)
        for coach in coaches:
            self.coaches.extend(set_pets(coach.current_class_details.owner, self.brand, [coach]))
        self.add_schemas(coaches)
        for coach in self.coaches:
            self._set_place_requirements_available(coach)

    def _get_categories(self):
        if not conf.TRAIN_PURCHASE_TARIFF_DIRECTORIES_ENABLED:
            return None
        return TariffCategories().find_categories(self.route_policy, self.is_suburban, self.is_cppk)

    def _set_place_requirements_available(self, coach):
        not_usual_places = [v for k, v in coach.places_by_type.items() if k != USUAL_PLACE_TYPE and v]
        coach.place_requirements_available = bool(
            coach.places
            and coach.place_reservation_type == PlaceReservationType.USUAL
            and (not self.is_cppk)
            and coach.type in COACH_TYPE_WITH_REQUIREMENTS
            and (coach.schema_id or not_usual_places)
        )


class CurrentClassDetails(BaseClassDetails):
    def __init__(self, car_raw_data, train_details):
        coach_type = CoachType.safe_from_im_code(car_raw_data['CarType'])

        self.car_raw_data = car_raw_data
        self.train_details = train_details
        self.real_coach_type = coach_type
        self.owner = car_raw_data['Carrier']
        self.service_class_code = car_raw_data['ServiceClass']
        self.service_class_description = car_raw_data['ServiceClassTranscript']
        self.service_codes = [
            FACILITIES_BY_IM[code].id for code in car_raw_data['Services']
            if code in FACILITIES_BY_IM
        ]
        self.international_service_class_code = car_raw_data['InternationalServiceClass']
        self.raw_category_traits = car_raw_data['CarDescription']
        self._parse_category_traits()
        self.im_services = car_raw_data['Services']
        self.is_quad_room = car_raw_data['PlaceReservationType'] == 'FourPlacesAtOnce'
        self.is_double_room = car_raw_data['PlaceReservationType'] == 'TwoPlacesAtOnce'
        self.loyalty_cards = self._parse_loyalty_cards(car_raw_data['RzhdCardTypes'])
        self.tariffs_info = self.get_tariffs_info()
        self.tariff_types = self.get_tariff_types()

        self.min_tariff = car_raw_data['MinPrice']
        self.max_tariff = car_raw_data['MaxPrice']
        self.has_non_refundable_tariff = car_raw_data['HasNonRefundableTariff']

    @property
    def coach_type(self):
        return self.real_coach_type.value

    @property
    def key(self):
        return self.coach_type, self.service_class_code, self.raw_category_traits, self.international_service_class_code

    def get_coach(self):
        return CoachDetails(self.car_raw_data, self)

    def get_tariffs_info(self):
        # TODO (syukhno) выпилить после перехода на get_tariff_types
        tariff_codes = list({d['DiscountType'] for d in self.car_raw_data['Discounts']} | {'Full'})
        tariffs_info = TariffInfo.objects.filter(im_request_code__in=tariff_codes)
        return sorted(tariffs_info,
                      key=lambda tariff_info: (tariff_info.max_age - tariff_info.min_age, tariff_info.min_age))

    def get_tariff_types(self):
        if conf.TRAIN_PURCHASE_TARIFF_DIRECTORIES_ENABLED:
            tariff_codes = {'Full'}
            if not self.train_details.is_cppk:
                tariff_codes |= {d['DiscountType'] for d in self.car_raw_data['Discounts']}
            filtered_codes = set(TariffInfo.objects.filter(
                im_request_code__in=tariff_codes).values_list('im_request_code', flat=True))
            return TariffTypes().find_types_by_provider_codes(self.train_details.route_policy, filtered_codes)

        return None

    @cached_property
    def service_tariff(self):
        tariff = 0
        if self.car_raw_data['ServiceCost']:
            tariff = self.car_raw_data['ServiceCost']
        if self.pet_places_only:
            tariff += get_pet_place_tariff(self.train_details.brand, self.train_details.when)
        return tariff

    def _parse_loyalty_cards(self, loyalty_cards):
        return [LOYALTY_CARDS.get(card) for card in loyalty_cards if card in LOYALTY_CARDS]


class CoachDetails(BaseCoachDetails):
    def __init__(self, car_raw_data, current_class_details):
        self.current_class_details = current_class_details
        self.real_coach_type = current_class_details.real_coach_type
        self.type = current_class_details.coach_type
        self.service_class_code = current_class_details.service_class_code
        self.international_service_class_code = current_class_details.international_service_class_code
        self.facilities = current_class_details.service_codes
        self.owner = current_class_details.owner
        self.pet_in_coach = current_class_details.pet_in_coach
        self.pet_places_only = current_class_details.pet_places_only
        self.pets_allowed = False
        self.pets_segregated = False

        self.car_raw_data = car_raw_data
        self.has_dynamic_pricing = car_raw_data['HasDynamicPricing']
        self.number = car_raw_data['CarNumber']
        self.can_choose_bedding = car_raw_data['IsBeddingSelectionPossible']
        self.two_storey = car_raw_data['IsTwoStorey']
        self.electronic_ticket = car_raw_data['HasElectronicRegistration']
        if settings.TRAIN_DETAILS_IGNORE_CAR_TRAIN_NUMBER:
            # train_start_number из вагона передается в оффер и используется для бронирования в ИМ
            self.train_start_number = current_class_details.train_details.start_number
        else:
            self.train_start_number = car_raw_data.get('TrainNumber', current_class_details.train_details.start_number)
        self.is_firm = current_class_details.is_firm
        self.is_for_children = current_class_details.is_for_children

        self.coach_subtype_code = car_raw_data['CarSubType']
        self.road = car_raw_data['Road']
        self.schema_id = None
        self.place_counts = self.get_place_counts()
        self.adult_tariff = None
        self.price_without_places = PlaceDetails(0, None)
        if car_raw_data['FreePlaces']:
            self.places = init_places(car_raw_data['FreePlaces'], current_class_details.coach_type)
        self.places_by_type = {car_raw_data['CarPlaceType']: [p.number for p in self.places]}
        self.places_by_compartment = [
            [parse_place_number(pl, current_class_details.coach_type)[0] for pl in compartment['Places'].split(', ')]
            for compartment in (car_raw_data.get('FreePlacesByCompartments') or [])
        ]
        self.service_class = get_service_class(self.current_class_details, two_storey=self.two_storey)
        self.loyalty_cards = current_class_details.loyalty_cards
        self.tariffs_info = current_class_details.tariffs_info
        self.tariff_types = current_class_details.tariff_types

        self.min_tariff = current_class_details.min_tariff
        self.max_tariff = current_class_details.max_tariff
        self.service_tariff = current_class_details.service_tariff
        self.bedding_tariff = Price(get_bedding_tariff(self.type, current_class_details.service_tariff))

        self.arrival = parse_datetime(car_raw_data['ArrivalDateTime'], current_class_details.train_details.station_to)
        self.through_arrival = self.arrival if car_raw_data['HasNoInterchange'] else None
        self.place_reservation_type = parse_place_reservation_type(car_raw_data)
        self.is_transit_document_required = car_raw_data['IsTransitDocumentRequired']
        self.availability_indication = IM_AVAILABILITY_INDICATION_TYPES.get(
            car_raw_data['AvailabilityIndication'], AvailabilityIndication.UNKNOWN
        )
        self.is_special_sale_mode = car_raw_data.get('IsSpecialSaleMode', False)
        self.has_non_refundable_tariff = current_class_details.has_non_refundable_tariff
        self.place_requirements_available = False
        self.im_scheme_id = car_raw_data.get('RailwayCarSchemeId')

        if current_class_details.train_details.is_suburban:
            self.child_coeff = Decimal(1)

    def validate(self):
        errors = set()
        if not check_tariff_coach_type(self.real_coach_type):
            errors.add(TariffError.UNSUPPORTED_COACH_TYPE)
        if self.min_tariff <= 10:
            errors.add(TariffError.TOO_CHEAP)
        if not check_place_reservation_types(self.place_reservation_type):
            errors.add(TariffError.UNSUPPORTED_RESERVATION_TYPE)
        if self.is_transit_document_required:
            errors.add(TariffError.TRANSIT_DOCUMENT_REQUIRED)
        if self.is_for_children:
            errors.add(TariffError.CHILD_TARIFF)
        if self.availability_indication != AvailabilityIndication.AVAILABLE:
            errors.add(AVAILABILITY_INDICATION_TO_TARIFF_ERROR.get(self.availability_indication, TariffError.UNKNOWN))

        return len(errors) == 0, errors

    @property
    def key(self):
        """
        Описания вагонов с одинаковыми условиями обслуживания и номерами, показываем как один вагон
        """
        return self.number, self.current_class_details.key

    def set_adult_tariffs_without_fee(self):
        if self.current_class_details.min_tariff == self.current_class_details.max_tariff:
            for place in self.iter_place_prices():
                place.ensure_adult_tariff(self.current_class_details.min_tariff)

    def get_place_counts(self):
        result = {'total': self.car_raw_data['PlaceQuantity'], 'upper_coupe': 0, 'lower_coupe': 0,
                  'upper_side': 0, 'lower_side': 0}

        if self.car_raw_data['CarPlaceType'] in ('Upper', 'LastKupeUpper'):
            result['upper_coupe'] = result['total']
        if self.car_raw_data['CarPlaceType'] in ('Lower', 'LastKupeLower'):
            result['lower_coupe'] = result['total']

        if self.car_raw_data['CarPlaceType'] in ('SideUpper', 'SideUpperNearRestroom'):
            result['upper_side'] = result['total']
        if self.car_raw_data['CarPlaceType'] in ('SideLower', 'SideLowerNearRestroom'):
            result['lower_side'] = result['total']

        return result

    def get_reservation_variants(self):
        if self.place_reservation_type == PlaceReservationType.TWO_PLACES_AT_ONCE:
            variants = []
            variants.extend([
                ReservationVariant(2, passengers, False, self.min_tariff, self.service_tariff)
                for passengers in ReservationVariant.generate_passengers_for_place_count(1)
            ])
            variants.extend([
                ReservationVariant(2, passengers, False, self.max_tariff, self.service_tariff * 2)
                for passengers in ReservationVariant.generate_passengers_for_place_count(2, 3)
            ])
            variants.extend([
                ReservationVariant(
                    2, {TariffInfo.FULL_CODE: 1, TariffInfo.CHILD_CODE: 1, TariffInfo.BABY_CODE: 0},
                    True, self.min_tariff, self.service_tariff
                ),
                ReservationVariant(
                    2, {TariffInfo.FULL_CODE: 2, TariffInfo.CHILD_CODE: 1, TariffInfo.BABY_CODE: 0},
                    True, self.max_tariff, self.service_tariff * 2
                ),
            ])
            if not (self.owner == 'ГРАНД' and self.type == CoachType.SOFT.value):
                variants.extend([
                    ReservationVariant(2,
                                       {TariffInfo.FULL_CODE: 2, TariffInfo.CHILD_CODE: 0, TariffInfo.BABY_CODE: 2},
                                       False, self.max_tariff, self.service_tariff * 2),
                    ReservationVariant(2,
                                       {TariffInfo.FULL_CODE: 2, TariffInfo.CHILD_CODE: 1, TariffInfo.BABY_CODE: 1},
                                       True, self.max_tariff, self.service_tariff * 2),
                ])
            return variants
        elif self.place_reservation_type == PlaceReservationType.FOUR_PLACES_AT_ONCE:
            base_tariff = self.min_tariff - self.service_tariff
            base_non_refund_tariff = None
            if self.has_non_refundable_tariff:
                base_non_refund_tariff = self.min_tariff - self.service_tariff
                base_tariff = (base_non_refund_tariff / Decimal('0.9'))\
                    .quantize(Decimal('0.00'), rounding=ROUND_HALF_UP)
            if not conf.TRAIN_PURCHASE_ENABLE_NON_REFUNDABLE:
                base_non_refund_tariff = None
            variants = []
            for passengers_count in (1, 2, 3, 4):
                service_cost = self.service_tariff * passengers_count
                variants.extend([
                    ReservationVariant(4, passengers, False, base_tariff + service_cost, service_cost,
                                       base_non_refund_tariff + service_cost if base_non_refund_tariff else None)
                    for passengers in ReservationVariant.generate_passengers_for_place_count(passengers_count)
                ])
            return variants
        else:
            return None

    def set_reservation_variants_amount(self, yandex_uid):
        if not self.reservation_variants:
            return
        func_calculate_ticket_cost = calculate_ticket_cost
        contract = self.current_class_details.train_details.contract
        for variant in self.reservation_variants:
            price = func_calculate_ticket_cost(
                contract, self.type, variant.original_amount, variant.original_service_amount, yandex_uid=yandex_uid
            )
            variant.set_calculated_ticket_cost(price)
