# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import traceback
from django.utils.encoding import force_text
from pprint import pformat

from common.dynamic_settings.default import conf
from common.models.transport import TransportType
from route_search.shortcuts import search_routes
from travel.rasp.train_api.tariffs.train.base.utils import get_possible_numbers
from travel.rasp.train_api.tariffs.train.segment_builder.helpers.merge_trains import make_meta_trains
from travel.rasp.train_api.train_bandit_api.logging import bandit_log_train_details
from travel.rasp.train_api.train_partners.base import PartnerError
from travel.rasp.train_api.train_partners.im.base import (
    get_im_response, IM_BUS_TRANSPORT_TYPE, PROVIDER_P2, IM_DATETIME_FORMAT
)
from travel.rasp.train_api.train_partners.im.train_details.parsers import TrainDetails
from travel.rasp.train_api.train_partners.im.train_details.yt_log import log_im_car_pricing_response
from travel.rasp.train_api.train_partners.mock_im import get_mock_im_car_pricing
from travel.rasp.train_api.wizard_api.client import train_wizard_api_client

log = logging.getLogger(__name__)
log_data = logging.getLogger(__name__ + '.data')

IM_TRAIN_DETAILS_METHOD = 'Railway/V1/Search/CarPricing'


class ImTrainDetailsError(PartnerError):
    def get_user_message(self):
        return 'Ошибка при получении информации от партнера'


def get_train_details(query):
    im_query = {
        'OriginCode': query.express_from,
        'DestinationCode': query.express_to,
        'DepartureDate': query.railway_dt.replace(second=0).strftime(IM_DATETIME_FORMAT),
        'TrainNumber': query.number,
        'TariffType': 'Full',
    }

    if query.provider:
        im_query['Provider'] = query.provider

    if query.mock_im:
        segment = find_segment(query)
        raw_data = get_mock_im_car_pricing(query, segment)
        check_raw_data(raw_data)
    else:
        raw_data = get_im_response(IM_TRAIN_DETAILS_METHOD, im_query)
        check_raw_data(raw_data)
        segment = find_segment(query)

    try:
        result = TrainDetails(raw_data, segment, query)
    except PartnerError:
        raise
    except Exception:
        log.exception('Неожиданная ошибка при разборе данных с IM')
        raise ImTrainDetailsError(None, 'Неожиданная ошибка при разборе данных с IM', {
            'traceback': traceback.format_exc(),
            'query': im_query,
            'response': force_text(pformat(raw_data))
        })

    if not query.mock_im:
        log_im_car_pricing_response(im_query, raw_data)
        train_wizard_api_client.store_details(result)
    if conf.TRAIN_PURCHASE_BANDIT_LOGGING:
        bandit_log_train_details(result, query.raw_query_data)

    return result


def check_raw_data(raw_data):
    if not raw_data.get('Cars'):
        raise ImTrainDetailsError(None, 'Ответ без вагонов')
    if raw_data['TrainInfo'].get('TransportType', '') == IM_BUS_TRANSPORT_TYPE:
        raise ImTrainDetailsError(None, 'Продажа на автобусы запрещена')


def find_segment(query):
    departure = query.station_from.pytz.localize(query.when)
    search_segments, _, _ = search_routes(
        query.station_from,
        query.station_to,
        departure_date=departure.date(),
        transport_types=[TransportType.objects.get(pk=TransportType.TRAIN_ID)],
        check_date=None,
        include_interval=False,
        add_z_tablos=False
    )
    search_segments = make_meta_trains(search_segments)
    train_numbers = [query.number]
    if (query.provider == PROVIDER_P2 and
            conf.TRAIN_PURCHASE_ENABLE_CPPK_CONDITIONS and
            query.number in conf.TRAIN_PURCHASE_P2_TRAIN_NUMBERS_MAP):
        train_numbers = conf.TRAIN_PURCHASE_P2_TRAIN_NUMBERS_MAP[query.number]
    return next((s for s in search_segments for number in train_numbers
                 if number in get_possible_numbers(s.thread.number) and s.departure == departure), None)
