# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import OrderedDict

import six
from django.conf import settings
from django.utils.encoding import smart_bytes, force_text

from common.utils import marketstat
from travel.rasp.train_api.train_partners.base.train_details.utils import (
    yt_parse_place, yt_build_car_type_flags, yt_build_train_type_flags, prepend_keys
)


yt_im_car_prising_logger = marketstat.DsvSimpleLog(settings.IM_CAR_PRISING_LOG)
log = logging.getLogger(__name__)


class ImToYtParser(object):
    @classmethod
    def _escape_special_symbols(cls, data):
        return (data.replace('\n', '<br/>')
                .replace('\r', ' ')
                .replace('\t', ' '))

    @classmethod
    def _deep_dump(cls, data):
        if isinstance(data, dict):
            data = data.copy()
            for key, value in list(data.items()):
                dumped_value = cls._deep_dump(value)
                if isinstance(dumped_value, dict):
                    del data[key]
                    data.update(prepend_keys(key, dumped_value))
                else:
                    data[key] = dumped_value
            return data
        elif isinstance(data, list):
            return ';;'.join(force_text(cls._deep_dump(v)) for v in data)
        elif isinstance(data, six.string_types):
            return force_text(cls._escape_special_symbols(data))
        else:
            return force_text(data)

    def __init__(self, im_query, im_response):
        self.query_value = {'query__{}'.format(key): value for key, value in im_query.items()}
        self.im_response = im_response

    def build_log_records(self):
        for record in self._build_cars():
            record.update(self.query_value)
            yield record

    def _build_cars(self):
        response_value = self._build_response_value()
        if not self.im_response['Cars']:
            yield response_value
            return

        for car_raw_data in self.im_response['Cars']:
            for record in self._build_car_values(car_raw_data):
                record = prepend_keys('Car', record)
                record.update(response_value)
                yield record

    def _build_response_value(self):
        response_rest = self.im_response.copy()
        del response_rest['Cars']
        del response_rest['TrainInfo']
        result = prepend_keys('Response', self._deep_dump(response_rest))
        result.update(prepend_keys('TrainInfo', self._build_train_info_value(self.im_response['TrainInfo'])))

        return result

    @classmethod
    def _build_train_info_value(cls, train_info_raw_data):
        train_info_value = cls._deep_dump(train_info_raw_data)
        train_info_value.update(prepend_keys('TrainDescription',
                                             yt_build_train_type_flags(train_info_value['TrainDescription'])))

        return train_info_value

    def _build_car_values(self, car_raw_data):
        car_value = self._build_car_info_value(car_raw_data)

        for raw_place in car_raw_data['FreePlaces'].split():
            place_value = prepend_keys('FreePlace', yt_parse_place(raw_place))
            place_value['FreePlace'] = raw_place.rstrip(',')
            place_value.update(car_value)
            yield place_value
        else:
            yield car_value

    def _build_car_info_value(self, car_raw_data):
        car_value = car_raw_data.copy()
        del car_value['FreePlaces']
        car_value = self._deep_dump(car_value)
        car_value.update(prepend_keys('CarDescription', yt_build_car_type_flags(car_value['CarDescription'])))

        return car_value


def log_im_car_pricing_response(im_query, response):
    try:
        parser = ImToYtParser(im_query, response)
        for record in parser.build_log_records():
            record['tskv_format'] = 'rasp-im-details-log'
            yt_im_car_prising_logger.log(OrderedDict(
                (smart_bytes(k), smart_bytes(record[k])) for k in sorted(record.keys())
            ))
    except Exception:
        log.exception('Error while writing im details log')
