# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import datetime
import re
from copy import deepcopy

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.date import get_timezone_from_point, get_pytz
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.helpers.bunker import get_mock_raw_data, get_node
from travel.rasp.train_api.train_partners.im.base import (
    IM_DATETIME_FORMAT, ImNonCriticalError
)

define_setting('ENABLE_MOCK_IM', {Configuration.TESTING: True}, default=False)

CAR_PRICING_MATCHER_NODE = 'travel/trains/mock-im-matchers/carpricing'
TRAIN_PRICING_MATCHER_NODE = 'travel/trains/mock-im-matchers/trainpricing'


def update_mock_im_car_pricing(raw_data, segment, query):
    rw_tz_from = get_railway_tz_by_point(query.station_from)
    tz_from = get_pytz(get_timezone_from_point(query.station_from))
    rw_tz_to = get_railway_tz_by_point(query.station_to)
    tz_to = get_pytz(get_timezone_from_point(query.station_to))
    if segment:
        departure = segment.departure.astimezone(rw_tz_from)
        arrival = segment.arrival.astimezone(rw_tz_to)
    else:
        departure = rw_tz_from.localize(query.railway_dt.replace(second=0))
        arrival = rw_tz_to.localize(query.railway_dt.replace(second=0) + datetime.timedelta(days=1))
    departure_rw_str = departure.strftime(IM_DATETIME_FORMAT)
    arrival_rw_str = arrival.strftime(IM_DATETIME_FORMAT)
    departure_local_str = departure.astimezone(tz_from).strftime(IM_DATETIME_FORMAT)
    arrival_local_str = arrival.astimezone(tz_to).strftime(IM_DATETIME_FORMAT)
    express_from = query.express_from
    express_to = query.express_to
    number = query.number
    raw_data = deepcopy(raw_data)
    raw_data["OriginCode"] = express_from
    raw_data["DestinationCode"] = express_to
    train_info = raw_data["TrainInfo"]
    train_info["TrainNumber"] = number
    train_info["TrainNumberToGetRoute"] = number
    train_info["DisplayTrainNumber"] = number
    train_info["OriginStationCode"] = express_from
    train_info["OriginName"] = query.station_from.title
    train_info["DestinationStationCode"] = express_to
    train_info["DestinationName"] = query.station_to.title
    train_info["DepartureDateTime"] = departure_rw_str
    train_info["LocalDepartureDateTime"] = departure_local_str
    train_info["ArrivalDateTime"] = arrival_rw_str
    train_info["ArrivalDateTimes"] = [arrival_rw_str]
    train_info["LocalArrivalDateTime"] = arrival_local_str
    train_info["LocalArrivalDateTimes"] = [arrival_local_str]
    for car in raw_data["Cars"]:
        car["DestinationStationCode"] = express_to
        car["ArrivalDateTime"] = arrival_rw_str
        car["LocalArrivalDateTime"] = arrival_local_str
        car["TrainNumber"] = number
        car["MealSalesOpenedTill"] = arrival_rw_str
    return raw_data


def get_car_pricing_mock_im_path(station_from, station_to, number):
    matcher = get_node(CAR_PRICING_MATCHER_NODE).json()
    for variant in matcher:
        if (re.match(variant['stationFrom'], station_from) and
                re.match(variant['stationTo'], station_to) and
                re.match(variant['number'], number)):
            return variant['mockImPath']


def get_train_pricing_mock_im_path(station_from, station_to):
    matcher = get_node(TRAIN_PRICING_MATCHER_NODE).json()
    for variant in matcher:
        if (re.match(variant['pointFrom'], station_from) and
                re.match(variant['pointTo'], station_to)):
            return variant['mockImPath']


def get_mock_im_car_pricing(query, segment):
    mock_im_path = query.mock_im_path
    if query.mock_im_auto and not mock_im_path:
        mock_im_path = get_car_pricing_mock_im_path(query.station_from.point_key,
                                                    query.station_to.point_key,
                                                    query.number)
        if not mock_im_path:
            raise ImNonCriticalError(311, "На заданном направлении (или поезде) мест нет",
                                     ["Подходящий mock_im_path не найден"])
    raw_data = get_mock_raw_data(mock_im_path)
    return update_mock_im_car_pricing(raw_data, segment, query)


def _get_im_datetime(datetime_string):
    return datetime.datetime.strptime(datetime_string, IM_DATETIME_FORMAT)


def update_mock_im_train_pricing(raw_data, train_query):
    raw_data = deepcopy(raw_data)
    departure_date = train_query.departure_date
    rw_tz_from = get_railway_tz_by_point(train_query.departure_point)
    tz_from = get_pytz(get_timezone_from_point(train_query.departure_point))
    rw_tz_to = get_railway_tz_by_point(train_query.arrival_point)
    tz_to = get_pytz(get_timezone_from_point(train_query.arrival_point))
    express_from = train_query.departure_point_code
    express_to = train_query.arrival_point_code
    raw_data['OriginStationCode'] = express_from
    raw_data['OriginCode'] = express_from
    raw_data['DestinationStationCode'] = express_to
    raw_data['DestinationCode'] = express_to
    for im_train in raw_data['Trains']:
        rw_departure = rw_tz_from.localize(_get_im_datetime(im_train['DepartureDateTime']))
        rw_arrival = rw_tz_to.localize(_get_im_datetime(im_train['ArrivalDateTime']))
        trip_time = rw_arrival - rw_departure
        rw_departure = rw_tz_from.localize(datetime.datetime.combine(departure_date, rw_departure.time()))
        rw_arrival = rw_departure.astimezone(rw_tz_to) + trip_time
        local_departure = rw_departure.astimezone(tz_from)
        local_arrival = rw_arrival.astimezone(tz_to)

        departure_rw_str = rw_departure.strftime(IM_DATETIME_FORMAT)
        arrival_rw_str = rw_arrival.strftime(IM_DATETIME_FORMAT)
        departure_local_str = local_departure.strftime(IM_DATETIME_FORMAT)
        arrival_local_str = local_arrival.strftime(IM_DATETIME_FORMAT)

        im_train["DepartureDateTime"] = departure_rw_str
        im_train["LocalDepartureDateTime"] = departure_local_str
        im_train["ArrivalDateTime"] = arrival_rw_str
        im_train["ArrivalDateTimes"] = [arrival_rw_str]
        im_train["LocalArrivalDateTime"] = arrival_local_str
        im_train["LocalArrivalDateTimes"] = [arrival_local_str]
    return raw_data


def get_mock_im_train_pricing(query):
    mock_im_path = query.mock_im_path
    if query.mock_im_auto and not mock_im_path:
        mock_im_path = get_train_pricing_mock_im_path(query.departure_point.point_key,
                                                      query.arrival_point.point_key)
        if not mock_im_path:
            raise ImNonCriticalError(311, "На заданном направлении (или поезде) мест нет",
                                     ["Подходящий mock_im_path не найден"])
    raw_data = get_mock_raw_data(mock_im_path)
    return update_mock_im_train_pricing(raw_data, query)
