# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, timedelta

import requests
import six
from django.conf import settings
from django.utils.encoding import force_bytes
from lxml import etree
from pytz import UTC

from common.apps.train.models import UFSErrorDescription
from common.utils.date import MSK_TZ
from common.utils.lxmlutils import get_sub_tag_text, SubTagNotFound
from common.utils.railway import get_railway_tz_by_point
from common.utils.safe_xml_parser import safe_xml_fromstring
from travel.rasp.train_api.train_partners.base import PartnerError, AbstractPartnerMeasurable
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner, OperationStatus

UFS_DATETIME_FORMAT = '%d.%m.%Y %H:%M:%S'
UFS_ENCODING = 'cp1251'

log = logging.getLogger(__name__)


RETRY_ERROR_CODES = [
    1006,  # Повторите запрос
    5067,  # Ошибка связи. Повторите запрос
    5069,  # Сбой при обработке запроса. Повторите запрос
    5380  # Ошибка связи
]


# 1105: Транзакция не найдена или не относится к Агенту авторизации
TRANSACTION_NOT_FOUND_DESCR_ID = 1105


@six.python_2_unicode_compatible
class UfsError(PartnerError):
    def __init__(self, code, description_id, message):
        super(UfsError, self).__init__(code, message)
        self.description_id = description_id

        # Чтобы нормально pickle'лось в кеш
        self.args = code, description_id, message

    def __str__(self):
        return ('<UfsError: code="{self.code}" message="{self.message}"'
                ' description_id={self.description_id}>'.format(self=self))

    @classmethod
    def parse(cls, root_el):
        try:
            code = int(get_sub_tag_text(root_el, 'Code').strip())
            description_id = int(get_sub_tag_text(root_el, 'DescrId').strip())
            description = get_sub_tag_text(root_el, 'Descr', default='').strip()
        except SubTagNotFound:
            return cls(None, None, 'Empty Error')
        return cls(code, description_id, description)

    def is_communication_error(self):
        return self.description_id == 5380

    def is_retry_allowed(self):
        return self.description_id in RETRY_ERROR_CODES

    def is_refund_not_found_error(self):
        return self.description_id == TRANSACTION_NOT_FOUND_DESCR_ID

    def get_user_message(self):
        description = self.message
        try:
            error_description = UFSErrorDescription.objects.get(ufs_code=self.code, show=True)
            description = error_description.L_description() or description
        except UFSErrorDescription.DoesNotExist:
            pass
        return description

    def get_data(self):
        return {
            'code': self.code,
            'description_id': self.description_id,
            'description': self.message
        }


class UfsUnexpectedValue(Exception):
    pass


def check_on_err(root):
    if root.find('./Error') is not None:
        raise UfsError.parse(root)


class UfsResponseStatus(object):
    OK = '0'
    FAILED = '1'


def build_url(path):
    return 'https://{host}/{path}'.format(
        host=settings.UFS_HOST,
        path=path
    )


def get_auth():
    return settings.UFS_USER, settings.UFS_PASSWORD


def get_default_headers():
    return {'user-agent': settings.UFS_TERMINAL}


class TimeType(object):
    MOSCOW = '0'
    LOCAL = '1'
    UTC = '2'


def parse_datetime(element, point=None):
    value = datetime.strptime(element.text, UFS_DATETIME_FORMAT)
    time_type = element.attrib.get('timeType', TimeType.MOSCOW)
    if time_type == TimeType.MOSCOW:
        return MSK_TZ.localize(value)
    elif time_type == TimeType.UTC:
        dt = UTC.localize(value)
        if point:
            dt = dt.astimezone(get_railway_tz_by_point(point))
        return dt

    time_offset = element.attrib.get('timeOffset')
    if not time_offset:
        raise UfsUnexpectedValue('timeOffset expected')
    shift = time_offset[0]
    hours, minutes = map(int, time_offset[1:].split(':'))
    delta = timedelta(hours=hours, minutes=minutes)
    value = value - delta if shift == '+' else value + delta
    dt = UTC.localize(value)
    if point:
        dt = dt.astimezone(get_railway_tz_by_point(point))
    return dt


def encode_ufs_params(params):
    return {name: (
        [force_bytes(item, UFS_ENCODING) for item in value]
        if isinstance(value, list) else
        force_bytes(value, UFS_ENCODING)
    ) for name, value in params.items()}


def get_raw_ufs_response(endpoint, params):
    url = build_url('webservices/Railway/Rest/Railway.svc/' + endpoint)
    log.info('Making request to UFS: {} with parameters\n {}'.format(endpoint, params))
    return requests.get(
        url,
        params=encode_ufs_params(params),
        verify=settings.UFS_SSL,
        timeout=settings.DEFAULT_PARTNER_TIMEOUT,
        auth=get_auth(),
        headers=get_default_headers()
    )


def get_ufs_response(endpoint, params):
    response = get_raw_ufs_response(endpoint, params)

    try:
        tree = safe_xml_fromstring(response.content)
        log.info('For request {response.request.url} got xml response\n{xml_content}\n '
                 'with status code {response.status_code}'.format(xml_content=etree.tounicode(tree, pretty_print=True),
                                                                  response=response))
    except Exception:
        log.error('For request {response.request.url} got response\n {response.text}\n '
                  'with status code {response.status_code}'.format(response=response))
        raise

    check_on_err(tree)

    return tree


UFS_OPERATION_STATUS_TO_OPERATION_STATUS = {
    '0': OperationStatus.OK,
    '1': OperationStatus.FAILED
}


class measurable(AbstractPartnerMeasurable):
    prefix = TrainPartner.UFS.value
