# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from decimal import Decimal

from django.utils.encoding import force_text
from enum import Enum

from common.utils.lxmlutils import get_sub_tag_text, get_bool_from_sub_tag
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_partners.base.get_order_info import TicketInfo, PassengerInfo, OrderInfoResult
from travel.rasp.train_api.train_partners.ufs.base import (
    get_ufs_response, UfsUnexpectedValue, parse_datetime, UFS_OPERATION_STATUS_TO_OPERATION_STATUS, measurable
)
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus


log = logging.getLogger(__name__)

TRANSACTION_INFO_ENDPOINT = 'TransInfo'


class UfsOperationType(Enum):
    SALE = '1'
    REFUND = '14'


@measurable()
def get_order_info_by_reference_id(order, reference_id):
    return _get_order_info({'Stan': reference_id})


@measurable()
def get_order_info(order, timeout=None):
    return _get_order_info({'IdTrans': order.current_partner_data.operation_id})


def _get_order_info(params):
    transinfo_xml = get_ufs_response(TRANSACTION_INFO_ENDPOINT, params)
    expire_set_er_el = transinfo_xml.find('./ExpireSetEr')
    is_refund_operation = get_sub_tag_text(
        transinfo_xml, './Type', default=UfsOperationType.SALE.value).strip() == UfsOperationType.REFUND.value
    trans_id = get_sub_tag_text(transinfo_xml, './TransID').strip()
    order_info = OrderInfoResult(
        buy_operation_id=trans_id,
        status=UFS_OPERATION_STATUS_TO_OPERATION_STATUS.get(
            get_sub_tag_text(transinfo_xml, './TStatus'), OperationStatus.IN_PROCESS
        ),
        order_num=get_sub_tag_text(transinfo_xml, './OrderNum', default=None),
        expire_set_er=parse_datetime(expire_set_er_el) if expire_set_er_el is not None else None
    )

    last_refund_operation_id = get_sub_tag_text(transinfo_xml, './LastRefundTransID', default='').strip() or None

    for passenger_el in transinfo_xml.findall('Passenger'):
        passenger = PassengerInfo(blank_id=passenger_el.attrib['BlankID'],
                                  doc_id=get_sub_tag_text(passenger_el, './DocNum').strip())
        if passenger.doc_id == '':
            raise UfsUnexpectedValue('No documents info about passenger in {}'.format(transinfo_xml))
        order_info.passengers.append(passenger)

    for blank_el in transinfo_xml.findall('Blank'):
        sale_refunded = get_bool_from_sub_tag(blank_el, 'RetFlag', default=False)
        _blank_id = blank_el.attrib['ID']

        if is_refund_operation:
            blank_id = force_text(int(blank_el.get('PrevID', '0').strip()) or _blank_id)
            refund_blank_id = _blank_id
            refund_operation_id = trans_id
            refund_amount = Decimal(get_sub_tag_text(blank_el, './Amount').strip())
            amount = None
        else:
            blank_id = _blank_id
            refund_blank_id = None
            refund_operation_id = None
            amount = Decimal(get_sub_tag_text(blank_el, './Amount').strip())
            refund_amount = None
            if sale_refunded:
                # Настоящий refund_operation_id УФС не предоставляет, но использоваться
                # этот id будет только для возвратов, сделанных не через нашу систему.
                # Поэтому для большинства случаев использования все будет правильно.
                refund_operation_id = last_refund_operation_id
        ticket = TicketInfo(
            blank_id=blank_id,
            rzhd_status=RzhdStatus(
                int(get_sub_tag_text(blank_el, './RzhdStatus').strip())
            ),
            amount=amount,
            pending=get_sub_tag_text(blank_el, 'RemoteCheckIn', default='').strip() == '4',
            refund_blank_id=refund_blank_id,
            refund_operation_id=refund_operation_id,
            refund_amount=refund_amount,
        )
        order_info.tickets.append(ticket)
    return order_info
