# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime
from decimal import Decimal

from django.conf import settings

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.models.geo import Country
from common.utils.lxmlutils import get_sub_tag_text
from travel.rasp.train_api.train_partners.base import create_tax
from travel.rasp.train_api.train_partners.base.reserve_tickets import (
    BaseReserveRequestOld, ReserveResponse, get_notices, RESERVE_TICKET_ENDPOINT_NAME
)
from travel.rasp.train_api.train_partners.ufs.base import parse_datetime, get_ufs_response, measurable
from travel.rasp.train_api.train_purchase.core.enums import (
    DocumentType, Gender, GenderChoice, PlacesOption, PlacesType
)

log = logging.getLogger(__name__)

BUY_TICKET_ENDPOINT = 'BuyTicket'
ADVERT_DOMAIN = 'yandex.ufs-online.ru'

GENDER_CHOICE_TO_SEX_VALUE = {
    GenderChoice.MALE: 'М',
    GenderChoice.FEMALE: 'Ж',
    GenderChoice.MIXED: 'С'
}
COACH_TYPE_TO_TYPE_CAR_VALUE = {
    CoachType.PLATZKARTE: 'П',
    CoachType.COMPARTMENT: 'К',
    CoachType.SUITE: 'Л',
    CoachType.COMMON: 'О',
    CoachType.SITTING: 'С'
}
PLACES_OPTION_TO_PLACEDEMANDS_VALUE = {
    PlacesOption.WITH_PET: 'Ж'
}
CM_TO_PLACES_TYPE = {
    'Н': PlacesType.LOWER_TIER,
    'В': PlacesType.UPPER_TIER,
    'С': PlacesType.MIDDLE_TIER
}
SITTING_CM_TO_PLACES_TYPE = {
    'Ж': PlacesType.FOR_PASSENGER_WITH_PET,
    'М': PlacesType.FOR_MOTHER_WITH_CHILD,
    'Р': PlacesType.FOR_PASSENGER_WITH_CHILDREN,
    'И': PlacesType.FOR_IMPAIRED_PASSENGER,
    'Г': PlacesType.CONFERENCE_ROOM,
    'Н': PlacesType.NOT_NEAR_TABLE,
    'П': PlacesType.NEAR_TABLE,
    'В': PlacesType.NEAR_PLAYGROUND,
    'Б': PlacesType.NEAR_PLAYGROUND_TABLE,
    'Д': PlacesType.NEAR_PASSENGERS_WITH_PETS,
    'О': PlacesType.FOLDING,
    '7': PlacesType.LASTOCHKA_COMPARTMENT
}


def _parse_compartment_gender(gender):
    gender = gender.lower()
    if gender.startswith('ж'):
        return GenderChoice.FEMALE
    elif gender.startswith('м'):
        return GenderChoice.MALE
    else:
        return GenderChoice.MIXED


def _parse_tariff_info(raw_tariff_code):
    return next((
        tariff_info for tariff_info in TariffInfo.objects.all()
        if raw_tariff_code in tariff_info.ufs_response_codes_list
    ), None)


class ReserveRequest(BaseReserveRequestOld):
    _passenger_indexes = None

    def _build_passengers_param(self):
        pass_doc_values = []

        for passenger in self.passengers:
            pass_doc_value = ('{passenger.document_type.value}{passenger.document_number}/'
                              '{passenger.last_name}={passenger.first_name}={patronymic}/'
                              '{passenger.birth_date:%d%m%Y}*{passenger.tariff_info.ufs_request_code}/'
                              '{passenger.citizenship_country.code3}/'
                              '{passenger.gender.value}'
                              .format(passenger=passenger, patronymic=passenger.patronymic or '-'))

            if passenger.loyalty_cards:
                pass_doc_value += ''.join([
                    '|{type}:{number}'.format(type=card['type'].value, number=card['number'])
                    for card in passenger.loyalty_cards]
                )

            pass_doc_values.append(pass_doc_value)

        return {'pass_doc': pass_doc_values}

    def _build_optional_params(self):
        optional_params = (
            ('bedding', int(self.bedding) if self.bedding is not None else None),
            ('diapason', ('{self.min_place_number}-{self.max_place_number}'.format(self=self)
                          if self.min_place_number is not None and self.max_place_number is not None else
                          None)),
            ('lang', self.lang),
            ('n_car', self.coach_number),
            ('n_down', self.lower_places_quantity),
            ('n_up', self.upper_places_quantity),
            ('placedemands', PLACES_OPTION_TO_PLACEDEMANDS_VALUE.get(self.places_option)),
            ('service_class', self.service_class),
            ('sex', GENDER_CHOICE_TO_SEX_VALUE.get(self.gender)),
            ('storey', self.storey),
        )
        return {param: value for param, value in optional_params if value is not None}

    def _build_request_params(self):
        request_params = {
            'advertDomain': ADVERT_DOMAIN,
            'creditCard': 1,  # XXX В документации такого параметра нет. Нужен ли он?
            'day': self.when.day,
            'from': self.departure_point_code,
            # TODO Передавать in_one_kupe в self.places_location (SINGLE_COMPARTMENT, SINGLE_SECTION, COMPARTMENTS_ONLY)
            'in_one_kupe': 0,
            'month': self.when.month,
            'payType': 'CC',
            'remoteCheckIn': int(self.electronic_registration),
            'terminal': settings.UFS_TERMINAL,
            'time': self.when.strftime('%H:%M'),
            'to': self.arrival_point_code,
            'train': self.train_number,
            'type_car': COACH_TYPE_TO_TYPE_CAR_VALUE[self.coach_type]
        }
        request_params.update(self._build_passengers_param())
        request_params.update(self._build_optional_params())

        return request_params

    def _parse_passenger_fio(self, raw_fio):
        return raw_fio.split('=', 2)

    def _parse_passenger_document(self, raw_document):
        return DocumentType(raw_document[:2]), raw_document[2:]

    def _parse_response(self, root):
        departure_station_title, arrival_station_title = (station_el.text.strip() for station_el in root.findall('./C'))
        car_type_code = get_sub_tag_text(root, 'KV', default='')
        notice_parts = _get_notice_parts(root)
        special_notice, time_notice = get_notices(notice_parts)

        response = ReserveResponse(
            amount=Decimal(get_sub_tag_text(root, 'Amount').strip()),
            arrival_station_title=arrival_station_title,
            coach_owner=get_sub_tag_text(root, 'VB', default='').strip(),
            compartment_gender=_parse_compartment_gender(get_sub_tag_text(root, 'R', default='')),
            departure_station_title=departure_station_title,
            operation_id=get_sub_tag_text(root, 'IDTrans', default='').strip(),
            reserved_to=parse_datetime(root.find('./ConfirmTimeLimit')),
            special_notice=special_notice,
            time_notice=time_notice,
            is_three_hours_reservation_available=False
        )

        for et in root.findall('./ET'):
            amount = get_sub_tag_text(et, 'TF', default='').strip()
            tariff_vat, service_vat, commission_fee_vat = _fill_vats(et)

            pi_list = et.findall('./PI')
            for pi in pi_list:
                raw_document = get_sub_tag_text(pi, 'PS').strip()
                passenger_document_type, passenger_document_number = self._parse_passenger_document(raw_document)

                if passenger_document_type == DocumentType.BIRTH_CERTIFICATE and len(pi_list) > 1:
                    # это детский до 5 лет без места
                    amount = None
                    places = []
                    places_type = None
                else:
                    places = [place.strip() for place in get_sub_tag_text(et, 'H', default='').split(',') if place]
                    places_type = (
                        SITTING_CM_TO_PLACES_TYPE
                        if car_type_code == 'С' else
                        CM_TO_PLACES_TYPE).get(get_sub_tag_text(et, 'CM', default=None))

                passenger_last_name, passenger_first_name, passenger_patronymic = \
                    self._parse_passenger_fio(get_sub_tag_text(pi, 'IZ'))
                raw_tariff_title = get_sub_tag_text(et, 'GT', default='').strip()
                response.add_ticket(
                    amount=Decimal(amount) if amount else Decimal(0),
                    blank_id=get_sub_tag_text(et, 'IDBlank', default='').strip(),
                    passenger_birth_date=datetime.strptime(get_sub_tag_text(pi, 'GD'), '%d%m%Y'),
                    passenger_citizenship_country=Country.objects.get(code3=get_sub_tag_text(pi, 'GR')),
                    passenger_document_number=passenger_document_number,
                    passenger_document_type=passenger_document_type,
                    passenger_first_name=passenger_first_name,
                    passenger_gender=Gender(get_sub_tag_text(pi, 'FM')),
                    passenger_last_name=passenger_last_name,
                    passenger_patronymic=passenger_patronymic,
                    places=places,
                    places_type=places_type,
                    tariff_info=_parse_tariff_info(raw_tariff_title),
                    raw_tariff_title=raw_tariff_title,
                    tariff_vat=tariff_vat,
                    service_vat=service_vat,
                    commission_fee_vat=commission_fee_vat,
                )

        return response

    @measurable(RESERVE_TICKET_ENDPOINT_NAME)
    def send(self):
        tree = get_ufs_response(BUY_TICKET_ENDPOINT, self._build_request_params())
        return self._parse_response(tree)


def _get_notice_parts(root):
    parts = [get_sub_tag_text(root, 'GA', default='').strip()]
    gb_parts = [part.strip() for part in get_sub_tag_text(root, 'GB', default='').split('.')]
    parts.extend(gb_parts)
    return [p for p in parts if p]


def _fill_vats(et):
    tariff_vat = _get_vat(et, 'TF4', 'STV1')
    service_vat = _get_vat(et, 'TF5', 'STV2')
    commission_fee_vat = _get_vat(et, 'TFB', 'STV3')
    return tariff_vat, service_vat, commission_fee_vat


def _get_vat(el, amount_tag, rate_tag):
    return create_tax(
        amount=get_sub_tag_text(el, amount_tag, default='').strip() or None,
        rate=get_sub_tag_text(el, rate_tag, default='').strip() or None
    )
