# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime
from decimal import Decimal

from django.utils.functional import cached_property

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.models.currency import Price
from common.utils.exceptions import SimpleUnicodeException
from common.utils.lxmlutils import get_sub_tag_text
from common.utils.railway import get_railway_tz_by_point
from common.utils.text import split_string
from travel.rasp.train_api.train_partners.base.train_details.parsers import (
    BaseTrainDetails, BaseClassDetails, BaseCoachDetails, parse_place_number, init_places
)
from travel.rasp.train_api.train_partners.base.train_details.service_classes import get_service_class
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.tariff_calculator import (
    get_pet_place_tariff, get_bedding_tariff
)
from travel.rasp.train_api.train_partners.base.train_details.utils import set_pets
from travel.rasp.train_api.train_partners.ufs.base import parse_datetime

log = logging.getLogger(__name__)


class TrainDetails(BaseTrainDetails):
    def __init__(self, n_el, pp_el, query):
        self.allowed_document_types = split_string(get_sub_tag_text(pp_el, 'AllowedDocTypes').strip())
        self.start_number = get_sub_tag_text(n_el, 'N1').strip()
        self.ticket_number = get_sub_tag_text(n_el, 'N2', self.start_number).strip().replace('*', '')
        self.electronic_ticket = n_el.find('./ER') is not None
        self.brand = get_sub_tag_text(n_el, 'BRN', default='').strip() or None
        self.coaches = []
        self.schemas = []
        self.raw_train_category = get_sub_tag_text(n_el, 'KN', '').strip()
        self.raw_train_name = get_sub_tag_text(n_el, 'NN',
                                               default=get_sub_tag_text(n_el, 'BRN', default='')).strip()
        train_categories = self.raw_train_category.split()
        self.is_firm = 'ФИРМ' in train_categories

        self.station_from = query.station_from
        self.express_from = query.express_from
        self.station_to = query.station_to
        self.express_to = query.express_to

        self.departure = parse_datetime(n_el.find('DepartureTime'), self.station_from)
        self.arrival = parse_datetime(n_el.find('ArrivalTime'), self.station_to)

        self.when = query.when
        self.contract = query.contract
        self.query = query

        for ck_el in n_el.findall('./CK'):
            try:
                current_class = CurrentClassDetails(ck_el, self)
            except CurrentClassDetailsError as e:
                log.error(e)
            else:
                coaches = current_class.get_coaches()
                coaches = set_pets(current_class.owner, self.brand, coaches)
                self.coaches += coaches

        self.add_schemas(self.coaches)
        self.add_tariffs(self.coaches, None, None, {})


class CurrentClassDetailsError(SimpleUnicodeException):
    pass


class CurrentClassDetails(BaseClassDetails):
    def __init__(self, ck_el, train_details):
        ufs_coach_type = get_sub_tag_text(ck_el, 'KV')

        try:
            coach_type = CoachType.from_ufs_code(ufs_coach_type)
        except KeyError:
            raise CurrentClassDetailsError('Неизвестный класс вагона {}'.format(ufs_coach_type))

        if coach_type == CoachType.SOFT:
            raise CurrentClassDetailsError('"Мягкие" вагоны недостпны для продажи')

        self.ck_el = ck_el
        self.has_dynamic_pricing = ck_el.find('./UD') is not None
        self.train_details = train_details
        self.raw_category_traits = get_sub_tag_text(ck_el, 'R', '')
        self._parse_category_traits()
        self.coach_type = coach_type.value
        self.owner = get_sub_tag_text(ck_el, 'VB', '').strip()
        self.road = get_sub_tag_text(ck_el, 'W2', '').strip()
        self.service_class_code = get_sub_tag_text(ck_el, 'CO', '').strip()
        self.international_service_class_code = None
        self.service_class_description = get_sub_tag_text(ck_el, 'CO_DESC', '').strip()
        self.service_codes = [code for code in get_sub_tag_text(ck_el, 'CO_SRV', '').strip().split(',') if code]
        self.is_quad_room = ck_el.find('./QM') is not None
        self.is_double_room = ck_el.find('./DM') is not None
        self.loyalty_cards = [card for card in split_string(get_sub_tag_text(ck_el, 'LoyaltyCards', ''))
                              if card != 'RzhdSU']
        self.tariffs_info = self.get_tariffs_info()

    def get_coaches(self):
        if not self.has_saleable_coaches():
            return []
        return [CoachDetails(cv_el, self) for cv_el in self.ck_el.findall('./CV')]

    def get_tariffs_info(self):
        ufs_codes = split_string(get_sub_tag_text(self.ck_el, 'AvailableTariffs', ''), ';')
        tariffs_info = TariffInfo.objects.filter(ufs_request_code__in=ufs_codes)
        return (sorted(tariffs_info,
                       key=lambda tariff_info: (tariff_info.max_age - tariff_info.min_age, tariff_info.min_age))
                if tariffs_info else
                [TariffInfo.objects.get(code=TariffInfo.FULL_CODE)])

    @cached_property
    def min_tariff(self):
        return Decimal(get_sub_tag_text(self.ck_el, 'TF'))

    @cached_property
    def max_tariff(self):
        if self.ck_el.find('./TF2') is not None:
            return Decimal(get_sub_tag_text(self.ck_el, 'TF2'))
        return self.min_tariff

    @cached_property
    def service_tariff(self):
        tariff = 0

        if self.ck_el.find('./TF3') is not None:
            tariff = Decimal(get_sub_tag_text(self.ck_el, 'TF3'))

        if self.pet_places_only:
            tariff += get_pet_place_tariff(self.train_details.brand, self.train_details.when)

        return tariff


class CoachDetails(BaseCoachDetails):
    def __init__(self, cv_el, current_class_details):
        self.current_class_details = current_class_details
        self.type = current_class_details.coach_type
        self.service_class_code = current_class_details.service_class_code
        self.facilities = current_class_details.service_codes
        self.owner = current_class_details.owner
        self.road = current_class_details.road
        self.pet_in_coach = current_class_details.pet_in_coach
        self.pet_places_only = current_class_details.pet_places_only
        self.pets_allowed = False
        self.pets_segregated = False
        self.has_dynamic_pricing = current_class_details.has_dynamic_pricing

        self.cv_el = cv_el
        self.number = get_sub_tag_text(cv_el, 'VH')
        self.can_choose_bedding = cv_el.find('./BL') is not None
        self.two_storey = bool(len(cv_el.xpath('TwoStorey')))
        self.electronic_ticket = cv_el.find('./ER') is not None
        self.train_start_number = current_class_details.train_details.start_number
        self.is_firm = current_class_details.is_firm

        self.coach_subtype_code = get_sub_tag_text(cv_el, 'PT', '').strip()
        self.schema_id = None
        self.place_counts = self.get_place_counts()
        self.adult_tariff = None
        self.places = init_places(get_sub_tag_text(cv_el, 'H', default=None), current_class_details.coach_type)

        self.service_class = get_service_class(self.current_class_details, two_storey=self.two_storey)
        self.loyalty_cards = current_class_details.loyalty_cards
        self.tariffs_info = current_class_details.tariffs_info

        self.min_tariff = current_class_details.min_tariff
        self.max_tariff = current_class_details.max_tariff
        self.service_tariff = current_class_details.service_tariff
        self.bedding_tariff = Price(get_bedding_tariff(self.type, current_class_details.service_tariff))

        self.through_arrival = (
            get_through_arrival_dt(current_class_details.train_details.when,
                                   current_class_details.train_details.station_to,
                                   get_sub_tag_text(cv_el, 'D1'), get_sub_tag_text(cv_el, 'T4'))
            if get_sub_tag_text(cv_el, 'AA', default=None) == 'БП' else
            None
        )

    def get_place_counts(self):
        total = int(get_sub_tag_text(self.cv_el, 'M4', '0'))
        lower_coupe = int(get_sub_tag_text(self.cv_el, 'M5', '0'))
        upper_coupe = int(get_sub_tag_text(self.cv_el, 'M6', '0'))
        lower_side = int(get_sub_tag_text(self.cv_el, 'M7', '0'))
        upper_side = int(get_sub_tag_text(self.cv_el, 'M8', '0'))
        if total == 0:
            total = lower_coupe + upper_coupe + lower_side + upper_side
        return {
            'total': total,
            'lower_coupe': lower_coupe,
            'upper_coupe': upper_coupe,
            'lower_side': lower_side,
            'upper_side': upper_side
        }

    def set_adult_tariffs_without_fee(self):
        tariffs = {}
        children = self.cv_el.getchildren()
        for child_el in children:
            amount = child_el.get('Amount')
            if amount is None:
                continue
            amount = Decimal(amount)
            place_numbers = [parse_place_number(raw_number, self.type) for raw_number in split_string(child_el.text)]
            tariffs.update({str(place_number): amount for (place_number, gender) in place_numbers})

        for place in self.places:
            place.ensure_adult_tariff(tariffs.get(str(place.number)))


def get_through_arrival_dt(departure_dt, arrival_station, arrival_date_str, arrival_time_str):
    arrival_dt = datetime.combine(datetime.strptime(arrival_date_str, '%d.%m').date(),
                                  datetime.strptime(arrival_time_str, '%H:%M').time()).replace(year=departure_dt.year)
    if arrival_dt <= departure_dt:
        arrival_dt = arrival_dt.replace(year=departure_dt.year + 1)
    return get_railway_tz_by_point(arrival_station).localize(arrival_dt)
