# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import time as os_time
import traceback

import pymongo
from django.conf import settings
from lxml import etree

from common.db import mongo
from common.db.mongo.counter import MongoCounter
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import global_cache_sync_delete
from common.utils.safe_xml_parser import safe_xml_fromstring
from travel.rasp.train_api.tariffs.train.base.service import (
    UnknownResultType, MAX_DOWNLOAD_ATTEMPTS, WorkerNetworkError, WorkerParseError
)
from travel.rasp.train_api.tariffs.train.base.worker import WorkerApiError, WorkerResult, Worker, WorkerEmptyResultError
from travel.rasp.train_api.train_partners.base import ApiError
from travel.rasp.train_api.train_partners.base.train_details import TrainDetailsQuery
from travel.rasp.train_api.train_partners.base.train_details.serialization import TrainDetailsSchema
from travel.rasp.train_api.train_partners.ufs.base import UfsError, get_ufs_response
from travel.rasp.train_api.train_partners.ufs.train_details.parsers import TrainDetails
from travel.rasp.train_api.train_partners.ufs.train_details.utils import log_ufs_response
from travel.rasp.train_api.wizard_api.client import train_wizard_api_client

# TODO: Нужно выпиливать этот способ получения мест, с кешированием и пользоваться ручкой async
TARIFF_CACHE_TIMEOUT = 60 * 5  # Время хранения информации по местам в вагоне

log = logging.getLogger(__name__)
log_data = logging.getLogger(__name__ + '.data')


class UfsTrainDetailsQuery(TrainDetailsQuery):
    @property
    def cache_key(self):
        return '{root}/backend_ufs_train_details_{express_from}_{express_to}_{number}_{dt}'.format(
            root=settings.CACHEROOT,
            express_from=self.express_from,
            express_to=self.express_to,
            number=self.number,
            dt=self.railway_dt.strftime('%Y-%m-%dT%H:%M'),
        )


class UfsTrainDetailsResult(WorkerResult):
    def __init__(self, query, status, train_details=None, error=None):
        super(UfsTrainDetailsResult, self).__init__(query, status, error=error)

        self.train_details = train_details

    @property
    def cache_timeout(self):
        if self.status == self.STATUS_PENDING:
            return settings.TARIFF_SUPPLIERWAIT_TIMEOUT
        elif self.status == self.STATUS_SUCCESS:
            return TARIFF_CACHE_TIMEOUT
        elif self.status == self.STATUS_ERROR:
            if isinstance(self.error, WorkerNetworkError):  # не кешируем сетевые ошибки
                return 0
            elif isinstance(self.error, UfsError):
                return self.get_ufs_error_cache_timeout()
            else:
                return settings.UFS_TRAIN_DETAILS_ERROR_TIMEOUT

        raise UnknownResultType('Unknown result type status={} error_type={}'.format(self.status, type(self.error)))

    def get_ufs_error_cache_timeout(self):
        if self.error.is_retry_allowed():
            return 0
        return settings.UFS_TRAIN_DETAILS_ERROR_TIMEOUT

    def delete_from_cache(self):
        global_cache_sync_delete(self.query.cache_key)


def _format_exception(exception):
    lines = traceback.format_exception_only(type(exception), exception)
    return ''.join(lines).strip()


class TrainDetailsException(WorkerApiError):
    pass


class ParsingError(WorkerParseError):
    pass


class NetworkError(WorkerNetworkError):
    pass


class TrainDetailsEmptyError(WorkerEmptyResultError):
    pass


def get_result(query, send_query=False):
    result = UfsTrainDetailsResult.get_from_cache(query)

    if result and result.expired_at > environment.now_aware():
        if result.status != result.STATUS_PENDING:
            result.delete_from_cache()  # кешируем результат только до момента его получения
        return result

    if getattr(settings, 'TEST_UFS_TRAIN_DETAILS_SETTINGS', {}).get('SYNCHRONOUS_RESPONSE', False):
        return process_query(query)

    if not send_query:
        return None

    return do_ufs_query(query)


def do_ufs_query(ufs_query):
    result = UfsTrainDetailsResult(ufs_query, UfsTrainDetailsResult.STATUS_PENDING)
    result.update_cache()

    worker = Worker(target=process_query, args=(ufs_query,))
    worker.start_daemon()

    return result


def process_query(query):
    log.info('Обрабатываем запрос %s %s %s %s', query.express_from, query.express_to,
             query.railway_dt, query.number)
    try:
        train_details = get_train_details(query)
    except ApiError as error:
        result = UfsTrainDetailsResult(query, UfsTrainDetailsResult.STATUS_ERROR, error=error)
    else:
        result = UfsTrainDetailsResult(query, UfsTrainDetailsResult.STATUS_SUCCESS,
                                       train_details=TrainDetailsSchema().dump(train_details).data)

    result.update_cache()

    return result


def get_train_details(query):
    try:
        if getattr(settings, 'TEST_UFS_TRAIN_DETAILS_SETTINGS', {}).get('USE_COLLECTED_DATA', False):
            tree = get_test_tree(query)
        else:
            tree = download_tree(query)
    except ApiError:
        raise
    except Exception as e:
        log.exception('Неожиданная ошибка при получении данных с UFS')
        raise ParsingError(_format_exception(e))

    try:
        result = parse_tree(tree, query)
    except ApiError:
        raise
    except Exception as e:
        log.exception('Неожиданная ошибка при разборе данных с UFS:\n%s',
                      etree.tounicode(tree, pretty_print=True))
        raise ParsingError(_format_exception(e))

    train_wizard_api_client.store_details(result)

    return result


CARLIST_ENDPOINT = 'CarListEx'


def download_tree(query):
    params = {
        'from': query.express_from,
        'to': query.express_to,
        'day': query.railway_dt.day,
        'month': query.railway_dt.month,
        'train': query.number,
        'terminal': settings.UFS_TERMINAL,
        'time': query.railway_dt.strftime('%H:%M'),
        'grouppingType': 3
    }

    last_exception = None
    for attempt_number in range(1, MAX_DOWNLOAD_ATTEMPTS + 1):
        try:
            log.info('Спрашиваем ufs CarListEx. Попытка: %s. Reuqest params: %s', attempt_number, repr(params))
            start = os_time.time()
            response_tree = get_ufs_response(CARLIST_ENDPOINT, params)
            log.info('%s fetch time %.3f', query.cache_key, os_time.time() - start)
            break
        except UfsError:
            raise
        except Exception as last_exception:
            log.exception('Не удалось получить данные от UFS')
    else:
        raise NetworkError(_format_exception(last_exception))

    log_data.info(
        'Получили ответ:\n'
        'QUERY: %(express_from)s %(express_to)s %(railway_dt)s %(number)s\n'
        'CACHE_KEY: %(cache_key)s\nRESPONSE:\n%(response)s\nEND_RESPONSE',
        {
            'express_from': query.express_from,
            'express_to': query.express_to,
            'railway_dt': query.railway_dt,
            'cache_key': query.cache_key,
            'number': query.number,
            'response': etree.tounicode(response_tree)
        }
    )

    log.debug('\n%s', etree.tounicode(response_tree, pretty_print=True))
    log_ufs_response(params, response_tree)
    return response_tree


def get_test_tree(query):
    test_ufs_train_details_settings = getattr(settings, 'TEST_UFS_TRAIN_DETAILS_SETTINGS', {})
    mode = test_ufs_train_details_settings.get('GET_DATA_MODE', 'requested')
    ufs_data_collection = test_ufs_train_details_settings['DATA_COLLECTION']
    collection = getattr(mongo.database, ufs_data_collection)

    data = None

    if mode.strip().lower() == 'any':
        counter = test_ufs_train_details_settings['RESPONSE_COUNTER']
        response_number = MongoCounter(counter).next_value()
        data = collection.find_one({'id': response_number})
    else:
        cursor = collection.find({
            'express_from': query.express_from,
            'express_to': query.express_to,
            'day': query.railway_dt.day,
            'month': query.railway_dt.month,
            'number': query.number,
        }).sort('received_at', direction=pymongo.DESCENDING).limit(1)
        for d in cursor:
            data = d

    if data is None:
        data = collection.find_one()

    return safe_xml_fromstring(data['xml'])


def parse_tree(tree, query):
    n_elements = tree.findall('./S/N')
    if not n_elements:
        raise TrainDetailsEmptyError()

    pp_el = tree.find('./S/Z3/PP')

    train_details_list = [TrainDetails(n_el, pp_el, query) for n_el in n_elements]
    train_details = merge_train_details_list(train_details_list)
    return train_details


def merge_train_details_list(train_details_list):
    if len(train_details_list) == 1:
        return train_details_list[0]

    # Функция получилась грязная. Она изменяет первый элемент в списке train_details_list.
    train_details = train_details_list[0]
    for item in train_details_list[1:]:
        if item.is_firm and not train_details.is_firm:
            train_details.is_firm = True
        if item.brand and not train_details.brand:
            train_details.brand = item.brand
        if item.electronic_ticket and not train_details.electronic_ticket:
            train_details.electronic_ticket = True

        train_details.coaches += item.coaches
        train_details.add_schemas(item.coaches)
        train_details.add_tariffs(item.coaches)

    return train_details
