# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta, datetime

from django.conf import settings
from django.utils.encoding import smart_bytes
from lxml import etree
from pytz import UTC

from common.utils import marketstat
from common.utils.date import MSK_TZ
from common.utils.lxmlutils import get_sub_tag_text
from travel.rasp.train_api.train_partners.base.train_details.utils import (
    yt_parse_place, yt_build_car_type_flags, yt_build_train_type_flags, prepend_keys
)

yt_ufs_response_logger = marketstat.DsvSimpleLog(settings.UFS_RESPONSE_LOG)
log = logging.getLogger(__name__)


def is_error_response(tree):
    status = get_sub_tag_text(tree, 'Status', '')
    return tree.find('./Error') is not None or status == '1'


class UfsToYtParses(object):
    def __init__(self, params, response_xml_tree):
        self.query = {'query__{}'.format(key): value for key, value in params.items()}
        self.xml_tree = response_xml_tree

    def build_log_records(self):
        if is_error_response(self.xml_tree):
            error_record = self._build_error()
            error_record.update(self.query)
            yield error_record
            return

        for record in self._build_s():
            record.update(self.query)
            yield record

    def _build_error(self):
        result = {
            el_name: get_sub_tag_text(self.xml_tree, el_name, '').strip()
            for el_name in ('Code', 'Descr', 'DescrId', 'Status')
        }
        return prepend_keys('Error', result)

    def _build_s(self):
        z3_value = self._build_z3(self.xml_tree.find('./S/Z3'))
        n_list = self.xml_tree.findall('./S/N')
        if not n_list:
            yield prepend_keys('S', z3_value)
            return

        for n in n_list:
            for record in self._build_n(n):
                record.update(z3_value)
                yield prepend_keys('S', record)

    def _build_n(self, n):
        n_result = {}
        for element in n:
            if element.tag == 'KN':
                kn_value = self._build_kn_value(element)
                n_result.update(kn_value)
            elif element.tag != 'CK':
                n_result[element.tag] = self._get_element_content(element)
                if element.tag in ('DepartureTime', 'ArrivalTime'):
                    column_name = '{}_MSK'.format(element.tag)
                    n_result[column_name] = self._get_msk_datetime(element)

        for ck in n.findall('./CK'):
            for record in self._build_ck(ck):
                record.update(n_result)
                yield prepend_keys('N', record)

    def _build_ck(self, ck):
        ck_result = {}
        for element in ck:
            if element.tag == 'R':
                r_value = self._build_r_value(element)
                ck_result.update(r_value)
            if element.tag != 'CV':
                ck_result[element.tag] = self._get_element_content(element)

        for cv in ck.findall('./CV'):
            for record in self._build_cv(cv):
                record.update(ck_result)
                yield prepend_keys('CK', record)

    def _build_cv(self, cv):
        cv_result = {}
        for element in cv:
            if element.tag != 'H':
                cv_result[element.tag] = self._get_element_content(element)
        h_elem = cv.find('./H')
        if not h_elem.text:
            yield prepend_keys('CV', cv_result)
            return

        h_list = h_elem.text.split(',')
        for h in h_list:
            record = self._build_h_value(h.strip())
            record.update(cv_result)
            yield prepend_keys('CV', record)

    def _build_z3(self, zf):
        return prepend_keys('Z3', {
            e.tag: self._get_element_content(e) for e in zf
        })

    def _build_h_value(self, h):
        h_result = yt_parse_place(h)
        result = prepend_keys('H', h_result)
        result['H'] = h
        return result

    def _build_kn_value(self, kn):
        kn_result = yt_build_train_type_flags(kn.text)

        return prepend_keys('KN', kn_result)

    def _build_r_value(self, r):
        r_result = yt_build_car_type_flags(r.text if r.text else '')

        return prepend_keys('R', r_result)

    @staticmethod
    def _get_element_content(element):
        if len(element):
            value = etree.tostring(element, encoding='utf8').strip()
        else:
            value = element.text
            if value:
                value = value.strip()
        return value

    @staticmethod
    def _get_msk_datetime(element):
        dt = datetime.strptime(element.text, '%d.%m.%Y %H:%M:%S')
        if element.attrib.get('timeType') != '2':
            tz_offset = element.attrib['timeOffset']
            shift = tz_offset[0]
            hours, minutes = map(int, tz_offset[1:].split(':'))
            delta = timedelta(hours=hours, minutes=minutes)
            dt = dt - delta if shift == '+' else dt + delta
        utc_dt = UTC.localize(dt)
        return utc_dt.astimezone(MSK_TZ).strftime('%Y-%m-%d %H:%M:%S')


def log_ufs_response(params, response_xml_tree):
    try:
        parser = UfsToYtParses(params, response_xml_tree)
        for record in parser.build_log_records():
            record['tskv_format'] = 'rasp-ufs-response-log'
            yt_ufs_response_logger.log({smart_bytes(k): smart_bytes(v) for k, v in record.items()})
    except Exception:
        log.exception('Error while writing ufs log')
