# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import base64
import json

from django.conf import settings
from rest_framework import exceptions
from rest_framework.authentication import BaseAuthentication
from rest_framework.decorators import authentication_classes, api_view
from rest_framework.pagination import LimitOffsetPagination
from rest_framework.viewsets import ViewSet
from ylog.context import put_to_context

from common.apps.train_order.models import BackofficeUser
from common.settings.utils import define_setting
from common.utils.blackbox_wrapper import get_session, SessionError, get_session_params_from_request

"""
Для того, чтобы авторизация в ручках бэкофиса всегда проходила, нужно добавить в local_settings:
BYPASS_BACKOFFICE_AUTH = True
"""
define_setting('BYPASS_BACKOFFICE_AUTH', default=False)


class BackofficeAuth(BaseAuthentication):
    def authenticate_header(self, request):
        if request.META.get('HTTP_AUTHORIZATION', None):
            return request.META.get('HTTP_AUTHORIZATION', None)
        else:
            return get_session_params_from_request(request)

    def authenticate(self, request):
        user, token = self._authenticate(request)
        if user:
            put_to_context("backoffice_user", user.username)
        return user, token

    def _authenticate(self, request):
        if settings.BYPASS_BACKOFFICE_AUTH:
            return BackofficeUser(username='anonymous', is_active=True, is_admin=True, id=123), None

        auth = self.authenticate_header(request)
        if isinstance(auth, dict):
            return self.blackbox_auth(auth)
        else:
            try:
                auth = json.loads(base64.b64decode(auth))
            except (TypeError, ValueError):
                raise exceptions.AuthenticationFailed('Bad auth header')

        if auth.get('type') == 'direct':
            return self.direct_auth(auth)
        if auth.get('type') == 'blackbox':
            return self.blackbox_auth(auth)

        raise exceptions.AuthenticationFailed('Unknown auth type')

    @classmethod
    def blackbox_auth(cls, auth):
        """
        Для тестирования. Нужно сделать доменное имя заканчивающееся на .yandex.ru.
        Тогда можно будет ходить в ручки бекофиса под своим пользователем без генерации ключей,
        без дополнительных параметров и заголовков.
        """
        try:
            bbsession = get_session(auth['userip'], auth['host'],
                                    auth['sessionid'], auth['ssl_sessionid'])
        except KeyError:
            raise exceptions.AuthenticationFailed('Bad auth header params')
        except SessionError:
            raise exceptions.AuthenticationFailed('Bad auth session')

        if not bbsession:
            raise exceptions.APIException('Blackbox is not available')

        login = bbsession.fields.get('login')
        try:
            user = BackofficeUser.objects.get(username=login, is_active=True)
        except BackofficeUser.DoesNotExist:
            raise exceptions.PermissionDenied('Has no permission')

        return user, None

    @classmethod
    def direct_auth(cls, auth):
        try:
            user = BackofficeUser.objects.get(username=auth.get('login'), is_active=True)
        except BackofficeUser.DoesNotExist:
            raise exceptions.PermissionDenied('Has no permission')

        return user, None


class BackofficeAdminAuth(BackofficeAuth):
    def authenticate(self, request):
        user, token = super(BackofficeAdminAuth, self).authenticate(request)
        if not user.is_admin:
            raise exceptions.PermissionDenied('Has no admin permission')
        return user, token


class BackofficeViewSet(ViewSet):
    authentication_classes = [BackofficeAuth]


class BackofficeAdminViewSet(ViewSet):
    authentication_classes = [BackofficeAdminAuth]


class MongoCursorLimitOffsetPagination(LimitOffsetPagination):
    def paginate_queryset(self, cursor, request, view=None):
        self.limit = self.get_limit(request)
        if self.limit is None:
            return None

        self.offset = self.get_offset(request)
        self.count = cursor.count()
        self.request = request

        if self.count > self.limit and self.template is not None:
            self.display_page_controls = True

        cursor.skip(self.offset)
        cursor.limit(self.limit)

        return list(cursor)


def backoffice_api_view(http_method_names=None, admin_only=False):
    def decorator(func):
        auth_class = BackofficeAdminAuth if admin_only else BackofficeAuth
        func = authentication_classes([auth_class])(func)
        func = api_view(http_method_names)(func)
        return func

    return decorator
