# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from bson import ObjectId
from django.utils.translation import get_language
from mongoengine import Q
from pymongo.collation import CollationStrength

from common.utils.date import UTC_TZ
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, Payment
from travel.rasp.train_api.train_purchase.core.utils import hash_doc_id

ORDER_FAILED_STATUSES = [
    OrderStatus.CONFIRM_FAILED, OrderStatus.PAYMENT_FAILED, OrderStatus.PAYMENT_OUTDATED,
    OrderStatus.START_PAYMENT_FAILED
]


def to_utc(dt):
    if dt.tzinfo is not None:
        dt = dt.astimezone(UTC_TZ).replace(tzinfo=None)

    return dt


class SearchOrdersQuery(object):
    search_filters = {
        'date_from': lambda x: Q(id__gte=ObjectId.from_datetime(to_utc(x))),
        'date_to': lambda x: Q(id__lte=ObjectId.from_datetime(to_utc(x))),

        'last_name': lambda x: Q(passengers__last_name=x),
        'first_name': lambda x: Q(passengers__first_name=x),
        'patronymic': lambda x: Q(passengers__patronymic=x),
        'email': lambda x: Q(user_info__email=x),
        'phone': lambda x: Q(user_info__reversed_phone=SearchOrdersQuery.get_reversed_phones_filter(x)),
        'doc_id': lambda x: Q(passengers__doc_id_hash=hash_doc_id(x)),

        'order_num': lambda x: Q(partner_data_history__order_num=str(x)),
        'ticket': lambda x: Q(passengers__tickets__blank_id=x),

        'status': lambda x: Q(status=x),
        'process_state': lambda x: Q(process__state=x),

        'is_failed': lambda x: Q(status__in=ORDER_FAILED_STATUSES) if x else Q(status__nin=ORDER_FAILED_STATUSES),
        'has_refunds': lambda x: Q(
            **{'passengers__tickets__rzhd_status{}'.format('' if x else '__ne'): RzhdStatus.REFUNDED}
        ),

        'departure_date_from': lambda x: Q(departure__gte=to_utc(x)),
        'departure_date_to': lambda x: Q(departure__lte=to_utc(x)),
        'station_from_id': lambda x: Q(station_from_id=x),
        'train_number': lambda x: Q(train_number=x),
    }

    collation_filters = {'last_name', 'first_name', 'patronymic', 'email', 'train_number'}

    @classmethod
    def get_reversed_phones_filter(cls, phone_str):
        if not phone_str:
            return phone_str
        country_codes = ('+375', '+380', '+7', '7', '8')
        for prefix in country_codes:
            if phone_str.startswith(prefix):
                phone_str = phone_str[len(prefix):]
                break
        return {"$regex": '^{}'.format(phone_str[::-1])}

    def __init__(self, data):
        self.data = data

    def build_queryset(self):
        filters = Q()

        if 'purchase_token' in self.data:
            try:
                payment = Payment.objects.get(purchase_token=self.data['purchase_token'])
                filters = filters & Q(uid=payment.order_uid)
            except Payment.DoesNotExist:
                return TrainOrder.objects.none()

        for field, filter_func in self.search_filters.items():
            if field in self.data:
                filters = filters & filter_func(self.data[field])

        queryset = TrainOrder.objects.filter(filters)
        if 'order_by' in self.data:
            queryset = queryset.order_by(self.data['order_by'])

        # где не нужен учёт регистра и диакритических знаков, добавляем collation
        if self.collation_filters.intersection(self.data):
            queryset = queryset.collation(locale=get_language(), strength=CollationStrength.PRIMARY)

        return queryset
