# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from codecs import BOM_UTF8
from io import BytesIO
from operator import itemgetter

import unicodecsv
from marshmallow import Schema, fields, pre_dump, post_load, validate
from marshmallow_enum import EnumField

from common.utils.date import MSK_TZ, UTC_TZ
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_purchase.backoffice.orders.search import SearchOrdersQuery
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder
from travel.rasp.train_api.train_purchase.serialization import OrderBaseDumpSchema, SourceSchema


class BaseOrdersQuerySchema(Schema):
    date_from = fields.DateTime(load_from='dateFrom')
    date_to = fields.DateTime(load_from='dateTo')

    last_name = fields.String(load_from='lastName')
    first_name = fields.String(load_from='firstName')
    patronymic = fields.String()
    email = fields.Email()
    phone = fields.String(load_from='phone')
    doc_id = fields.String(load_from='docId')

    order_num = fields.Integer(load_from='orderNum')
    ticket = fields.Integer(load_from='ticket')
    status = EnumField(OrderStatus, by_value=True)
    is_failed = fields.Boolean(load_from='isFailed')
    process_state = fields.String(load_from='processState')
    has_refunds = fields.Boolean(load_from='hasRefunds')
    purchase_token = fields.String(load_from='purchaseToken')

    departure_date_from = fields.DateTime(load_from='departureDateFrom')
    departure_date_to = fields.DateTime(load_from='departureDateTo')
    station_from_id = fields.Integer(load_from='stationFromId')
    train_number = fields.String(load_from='trainNumber')

    @post_load
    def build_query(self, data):
        return SearchOrdersQuery(data)


class SearchOrdersQuerySchema(BaseOrdersQuerySchema):
    order_by = fields.String(load_from='orderBy', missing='reserved_to', extra={
        'default': 'reserved_to',
        'examples': ['reserved_to', '-reserved_to', 'refund_expired_at', '-refund_expired_at', '_id', '-_id']
    })


class CountOrdersQuerySchema(BaseOrdersQuerySchema):
    pass


class RefundSchema(Schema):
    uuid = fields.String()
    blank_ids = fields.List(fields.String(), dump_to='blankIds')
    insurance_ids = fields.List(fields.String(), dump_to='insuranceIds')
    status = fields.String()
    real_payment_status = fields.String(dump_to='paymentStatus')
    real_trust_refund_id = fields.String(dump_to='trustRefundId')
    real_payment_resized = fields.Boolean(dump_to='paymentResized')

    @pre_dump
    def fill_payment_info(self, refund):
        """
        :param refund: train_api.train_purchase.core.models.TrainRefund
        """
        payment_status = None
        payment_resized = False
        trust_refund_id = None
        refund_payment = refund.refund_payment
        if refund_payment:
            payment_status = refund_payment.refund_payment_status
            payment_resized = refund_payment.payment_resized
            trust_refund_id = refund_payment.trust_reversal_id if payment_resized else refund_payment.trust_refund_id
        refund.real_payment_status = payment_status
        refund.real_payment_resized = payment_resized
        refund.real_trust_refund_id = trust_refund_id


class BackofficeOrderSchema(OrderBaseDumpSchema):
    fio = fields.Method('_format_passengers')
    order_price = fields.Method('_calculate_amount', dump_to='orderPrice')
    station_from = fields.Method('_get_station_from_title', dump_to='stationFrom')
    station_to = fields.Method('_get_station_to_title', dump_to='stationTo')
    created_at = fields.DateTime(attribute='id.generation_time', dump_to='createdAt')
    refunds = fields.Nested(RefundSchema, many=True)
    purchase_token = fields.String(dump_to='purchaseToken', attribute='current_billing_payment.purchase_token')
    purchase_tokens_history = fields.List(fields.String, dump_to='purchaseTokensHistory',
                                          attribute='purchase_tokens_history')
    orders_created = fields.List(fields.String, dump_to='ordersCreated')
    invalid_payments_unholded = fields.Boolean(dump_to='invalidPaymentsUnholded')
    source = fields.Nested(SourceSchema)
    insurance_price = fields.Function(
        lambda o: sum(p.insurance.amount for p in o.passengers if p.insurance and p.insurance.trust_order_id),
        dump_to='insurancePrice'
    )

    @staticmethod
    def _format_passengers(order):
        return '\n'.join(passenger.fio for passenger in order.passengers)

    @staticmethod
    def _calculate_amount(order):
        return sum(ticket.payment.total for ticket in order.iter_tickets())

    def _get_station_from_title(self, order):
        return self.get_station_title_by_field_name(order, 'station_from')

    def _get_station_to_title(self, order):
        return self.get_station_title_by_field_name(order, 'station_to')

    @staticmethod
    def get_station_title_by_field_name(order, field_name):
        station = getattr(order, field_name, None)
        if station:
            return station.L_title()
        id_field_name = '{}_id'.format(field_name)
        return getattr(order, id_field_name)

    @pre_dump(pass_many=True)
    def _fetch_stations(self, obj, many):
        orders = obj if many else [obj]
        TrainOrder.fetch_stations(orders)

    @pre_dump(pass_many=True)
    def _fill_refunds(self, obj, many):
        orders = obj if many else [obj]
        for order in orders:
            order.refunds = list(order.iter_refunds())


class UpdateUserInfoSchema(Schema):
    email = fields.Email(allow_none=False, validate=validate.Length(min=3))
    phone = fields.String(dump_to='phone')


class UpdateOrderSchema(Schema):
    user_info = fields.Nested(UpdateUserInfoSchema, load_from='userInfo')

    @classmethod
    def build_update(self, order, data):
        update = {}
        if 'user_info' in data:
            user_info = data['user_info']
            if 'email' in user_info and user_info['email'] != order.user_info.email:
                update['user_info__email'] = user_info['email']
            if 'phone' in user_info and user_info['phone'] != order.user_info.phone:
                update['user_info__phone'] = user_info['phone']
                update['user_info__reversed_phone'] = user_info['phone'][::-1]

        return update


class ResendTicketEmailSchema(Schema):
    email = fields.Email()


class ResendRefundedBlanksEmailSchema(Schema):
    refund_uuid = fields.String(required=True, load_from='refundUuid')
    email = fields.Email()


class DocumentSchema(Schema):
    fio = fields.String()
    doc_type = fields.String(dump_to='docType', attribute='doc_type.value')
    doc_id = fields.String(dump_to='docId')


class DocumentsSchema(Schema):
    documents = fields.Nested(DocumentSchema, many=True)


class ExportOrdersQuerySchema(SearchOrdersQuerySchema):
    pass


def _get_partner_data(order):
    return order['partner_data_history'][-1]


def _prepare_order(order):
    order['payments'].sort(key=lambda p: p['trust_created_at'], reverse=True)


class ExportToCsvOrderSchema(object):
    common_fields = (
        ('дата', lambda o: _date_in_msk_tz(o['finished_at']), ['finished_at']),
        ('мск. время завершения', lambda o: _time_in_msk_tz(o['finished_at']), ['finished_at']),
        ('мск. время оплаты', lambda o: _time_in_msk_tz(o['payments'][0].get('hold_at', o['payments'][0]['clear_at'])),
            ['payments.hold_at', 'payments.clear_at']),
        ('номер заказа партнера', lambda o: _get_partner_data(o)['im_order_id'],
            ['partner_data_history.im_order_id']),
        ('номер заказа РЖД', lambda o: _get_partner_data(o)['order_num'], ['partner_data_history.order_num']),
        ('UID заказа', itemgetter('uid'), ['uid']),
        ('статус заказа', itemgetter('status'), ['status']),
        ('кол-во билетов в заказе', lambda o: len(_list_tickets(o)), ['passengers.tickets.blank_id']),
        ('номер поезда', itemgetter('train_number'), ['train_number']),
        ('номер вагона', itemgetter('car_number'), ['car_number']),
        ('тип вагона', itemgetter('car_type'), ['car_type']),
        ('класс сервиса', lambda o: o['rebooking_info']['service_class'], ['rebooking_info.service_class']),
        ('перевозчик', itemgetter('coach_owner'), ['coach_owner']),
        ('название станции от', lambda o: _get_partner_data(o)['station_from_title'],
            ['partner_data_history.station_from_title']),
        ('ID станции от', itemgetter('station_from_id'), ['station_from_id']),
        ('название станции до', lambda o: _get_partner_data(o)['station_to_title'],
            ['partner_data_history.station_to_title']),
        ('ID станции до', itemgetter('station_to_id'), ['station_to_id']),
        ('дата отправления со станции пользователя', lambda o: _date_in_msk_tz(o['departure']), ['departure']),
        ('время отправления со станции пользователя', lambda o: _time_in_msk_tz(o['departure']), ['departure']),
        ('дата прибытия', lambda o: _date_in_msk_tz(o['arrival']), ['arrival']),
        ('время прибытия', lambda o: _time_in_msk_tz(o['arrival']), ['arrival']),
        ('почта', lambda o: o['user_info']['email'], ['user_info.email']),
        ('телефон', lambda o: o['user_info']['phone'], ['user_info.phone']),
        ('возможно бронирование на 3 часа', lambda o: _get_partner_data(o)['is_three_hours_reservation_available'],
            ['partner_data_history.is_three_hours_reservation_available']),
        ('предыдущий заказ', lambda o: o['orders_created'][0], ['orders_created']),
    )

    order_fields = (
        ('purchaseToken', lambda o: o['payments'][0]['purchase_token'], ['payments.purchase_token']),
        ('purchaseTokensHistory', lambda o: '[{}]'.format(','.join([p['purchase_token'] for p in o['payments'][1:]])),
            ['payments.purchase_token']),
        ('есть возвраты', lambda o: any(t['rzhd_status'] == RzhdStatus.REFUNDED for t in _list_tickets(o)),
            ['passengers.tickets.rzhd_status']),
        ('общая стоимость билетов', lambda o: sum(t['payment']['amount'] for t in _list_tickets(o)),
            ['passengers.tickets.payment.amount']),
        ('общая комиссия по заказу', lambda o: sum(t['payment']['fee'] for t in _list_tickets(o)),
            ['passengers.tickets.payment.fee']),
        ('id схемы', itemgetter('scheme_id'), ['scheme_id']),
    )

    passenger_fields = (
        ('фамилия', itemgetter('last_name'), ['passengers.last_name']),
        ('имя', itemgetter('first_name'), ['passengers.first_name']),
        ('отчество', itemgetter('patronymic'), ['passengers.patronymic']),
        ('пол', itemgetter('sex'), ['passengers.sex']),
        ('возраст', lambda p: int(p['age']), ['passengers.age']),
    )

    ticket_fields = (
        ('номер билета', itemgetter('blank_id'), ['passengers.tickets.blank_id']),
        ('стоимость билета', lambda t: t['payment']['amount'], ['passengers.tickets.payment.amount']),
        ('стоимость белья', lambda t: t['payment']['service_amount'], ['passengers.tickets.payment.service_amount']),
        ('комиссия', lambda t: t['payment']['fee'], ['passengers.tickets.payment.fee']),
        ('тариф', itemgetter('raw_tariff_title'), ['passengers.tickets.raw_tariff_title']),
        ('место', lambda t: ','.join(t['places']), ['passengers.tickets.places']),
        ('статус', itemgetter('rzhd_status'), ['passengers.tickets.rzhd_status']),
    )

    source_fields = (
        ('req_id колдунщика', lambda o: o['source']['req_id'], ['source.req_id']),
        ('устройство пользователя', lambda o: o['source']['device'], ['source.device']),
        ('utm_source', lambda o: o['source']['utm_source'], ['source.utm_source']),
        ('utm_medium', lambda o: o['source']['utm_medium'], ['source.utm_medium']),
        ('utm_campaign', lambda o: o['source']['utm_campaign'], ['source.utm_campaign']),
        ('utm_term', lambda o: o['source']['utm_term'], ['source.utm_term']),
        ('utm_content', lambda o: o['source']['utm_content'], ['source.utm_content']),
        ('from', lambda o: o['source']['from_'], ['source.from_']),
        ('gclid', lambda o: o['source']['gclid'], ['source.gclid']),
        ('terminal', lambda o: o['source']['terminal'], ['source.terminal']),
        ('is_transfer', lambda o: o['source']['is_transfer'], ['source.is_transfer']),
        ('partner', lambda o: o['source']['partner'], ['source.partner']),
        ('subpartner', lambda o: o['source']['subpartner'], ['source.subpartner']),
        ('partner_uid', lambda o: o['source']['partner_uid'], ['source.partner_uid']),
    )

    def iter_to_csv(self, queryset, layout):
        def _open_writer():
            s = BytesIO()
            return s, unicodecsv.writer(s, encoding='utf-8', dialect='excel')

        stream, writer = _open_writer()
        stream.write(BOM_UTF8)

        order_fields = []
        passenger_fields = []
        ticket_fields = []
        if layout == 'orders':
            order_fields.extend(self.common_fields + self.order_fields + self.source_fields)
        elif layout == 'tickets':
            order_fields.extend(self.common_fields)
            passenger_fields.extend(self.passenger_fields)
            ticket_fields.extend(self.ticket_fields)
        else:
            raise AttributeError('layout="%s" is not supported' % layout)

        row_fields = order_fields + passenger_fields + ticket_fields
        headers = [key[0] for key in row_fields]
        writer.writerow(headers)
        yield stream.getvalue()

        project_dict = {'payments.trust_created_at': 1}
        for _, _, project_fields in row_fields:
            project_dict.update({k: 1 for k in project_fields})
        queryset = queryset.aggregate(*[
            {'$lookup': {'from': 'payment', 'localField': 'uid', 'foreignField': 'order_uid', 'as': 'payments'}},
            {'$project': project_dict},
        ])
        for order in queryset:  # values
            _prepare_order(order)
            order_output = [_try_get_value(order, key[1]) for key in order_fields]
            if ticket_fields:
                for passenger in order['passengers']:
                    psngr_output = order_output + [_try_get_value(passenger, key[1]) for key in passenger_fields]
                    for ticket in passenger['tickets']:
                        stream, writer = _open_writer()
                        writer.writerow(psngr_output + [_try_get_value(ticket, key[1]) for key in ticket_fields])
                        yield stream.getvalue()
            else:
                stream, writer = _open_writer()
                writer.writerow(order_output)
                yield stream.getvalue()


def _list_tickets(order):
    tickets = [ticket for passenger in order['passengers'] for ticket in passenger['tickets']]
    return tickets


def _try_get_value(dictionary, getter):
    try:
        return getter(dictionary)
    except (KeyError, IndexError):
        return None


def _to_msk_tz(datetime):
    return UTC_TZ.localize(datetime).astimezone(MSK_TZ) if datetime else None


def _date_in_msk_tz(datetime):
    return _to_msk_tz(datetime).date() if datetime else None


def _time_in_msk_tz(datetime):
    return _to_msk_tz(datetime).time().replace(microsecond=0) if datetime else None
