# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import traceback
from collections import defaultdict, namedtuple
from datetime import datetime
from functools import wraps

import bson
import six
from django.core.serializers.json import DjangoJSONEncoder
from django.http import StreamingHttpResponse
from rest_framework import status
from rest_framework.decorators import detail_route, list_route
from rest_framework.pagination import LimitOffsetPagination
from rest_framework.response import Response
from ylog.context import log_context

from common.data_api.billing.trust_client import TrustClient
from common.dev_tools.swagger import swagger_aware_view_set
from common.utils.date import UTC_TZ, MSK_TZ
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.backoffice.base import BackofficeViewSet
from travel.rasp.train_api.train_purchase.backoffice.orders.serialization import (
    BackofficeOrderSchema, CountOrdersQuerySchema, ResendRefundedBlanksEmailSchema, ResendTicketEmailSchema,
    SearchOrdersQuerySchema, UpdateOrderSchema, ExportOrdersQuerySchema, ExportToCsvOrderSchema, DocumentsSchema
)
from travel.rasp.train_api.train_purchase.backoffice.serialization import ErrorSchema
from travel.rasp.train_api.train_purchase.core.models import (
    TrainOrder, BackofficeActionHistory, EmbeddedBackofficeUser, RefundPayment
)
from travel.rasp.train_api.train_purchase.core.utils import hash_doc_id
from travel.rasp.train_api.train_purchase.tasks.unhold_invalid_payments import unhold_order_payments
from travel.rasp.train_api.train_purchase.utils.order import update_tickets_statuses, UpdateTicketStatusMode
from travel.rasp.train_api.train_purchase.utils.order_tickets import rebooking
from travel.rasp.train_api.train_purchase.utils.refund_email import send_refund_email
from travel.rasp.train_api.train_purchase.utils.tickets_email import do_resend_tickets_email

log = logging.getLogger(__name__)

PAGE_SIZE = 40


class DjangoJSONEncoderMskTZ(DjangoJSONEncoder):
    def default(self, o):
        if isinstance(o, bson.objectid.ObjectId):
            return
        if isinstance(o, datetime):
            return UTC_TZ.localize(o).astimezone(MSK_TZ).replace(tzinfo=None).isoformat()
        return super(DjangoJSONEncoderMskTZ, self).default(o)


def backoffice_order_detail_route(*args, **kwargs):
    log_call = kwargs.pop('log_call', True)
    builtin = kwargs.pop('builtin', False)

    def decorator(view_func):
        if not builtin:
            view_func = detail_route(*args, **kwargs)(view_func)

        @wraps(view_func)
        def wrapper(self, request, pk):
            try:
                order = TrainOrder.objects.get(uid=pk)
            except TrainOrder.DoesNotExist:
                return Response({
                    'errors': {'uid': 'Order was not found'},
                }, status=status.HTTP_404_NOT_FOUND)

            with log_context(order_uid=order.uid):
                if log_call:
                    body = '\nDATA: {}'.format(request.data) if request.data else ''
                    log.info('%s: %s%s', request.method, request.get_full_path(), body)
                return view_func(self, request, order)
        return wrapper
    return decorator


@swagger_aware_view_set
class BackofficeOrderViewSet(BackofficeViewSet):
    @backoffice_order_detail_route(builtin=True)
    def retrieve(self, request, order):
        """Получить заказ
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    $ref: 'BackofficeOrderSchema'
        """
        return Response(BackofficeOrderSchema().dump(order).data)

    @backoffice_order_detail_route()
    def json(self, request, order):
        """Получить заказ
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: json c заказом
        """
        if order.process.get('history', None):
            order.process['history'] = None
        raw_order = order.to_mongo().to_dict()

        raw_order['refunds'] = []
        for refund in order.iter_refunds():
            raw_refund = refund.to_mongo().to_dict()
            raw_refund['refund_payments'] = [
                refund_payment.to_mongo().to_dict()
                for refund_payment in RefundPayment.objects.filter(refund_uuid=refund.uuid)
            ]
            raw_order['refunds'].append(raw_refund)

        raw_order['payments'] = [payment.to_mongo().to_dict() for payment in order.payments]
        return Response(json.dumps(raw_order, cls=DjangoJSONEncoderMskTZ))

    @backoffice_order_detail_route(builtin=True)
    def update(self, request, order):
        """Обновить заказ
        ---
        parameters:
          - in: body
            name: query_params
            schema:
                $ref: 'UpdateOrderSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    $ref: 'BackofficeOrderSchema'
        """
        data, errors = UpdateOrderSchema().load(request.data)
        if errors:
            return Response({'errors': errors}, status.HTTP_400_BAD_REQUEST)

        order_state = order.to_mongo()
        if 'history' in order_state.get('process', {}):
            del order_state['process']['history']

        update = UpdateOrderSchema.build_update(order, data)
        if update:
            log.info('Before update: {order.user_info: %s, request.user: %s}', order.user_info.to_json(), request.user)
            history = BackofficeActionHistory.objects.create(
                uid=order.uid,
                user=EmbeddedBackofficeUser(user_id=request.user.id, username=request.user.username),
                prev_state=order_state,
                action='update',
                details=update,
            )
            order.modify(**update)
            history.modify(completed=True)

        return Response(BackofficeOrderSchema().dump(order).data)

    def list(self, request):
        """Получить список заказов
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'SearchOrdersQuerySchema'
        responses:
            200:
                description: Найденные заказы
                schema:
                    $ref: 'BackofficeOrderSchema'
        """
        search_query, errors = SearchOrdersQuerySchema().load(request.query_params)
        if errors:
            return Response({'errors': errors}, status.HTTP_400_BAD_REQUEST)

        paginator = LimitOffsetPagination()
        paginator.default_limit = PAGE_SIZE
        queryset = search_query.build_queryset().exclude('process')
        try:
            # перехват ошибок выборки для логирования параметров запросов
            return paginator.get_paginated_response(
                BackofficeOrderSchema().dump(paginator.paginate_queryset(queryset, request), many=True).data
            )
        except Exception:
            log.exception('Ошибка выборки: query_params=%s', request.query_params)
            raise

    @list_route(methods=['get'])
    def count(self, request):
        """Количество заказов, удовлетворяющих запросу
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'CountOrdersQuerySchema'
        responses:
            200:
                description: количество найденных заказов
        """

        search_query, errors = CountOrdersQuerySchema().load(request.query_params)
        if errors:
            return Response({
                'errors': errors
            }, status.HTTP_400_BAD_REQUEST)

        return Response({'count': search_query.build_queryset().count()})

    @list_route(methods=['get'])
    def export_orders(self, request):
        """Экспорт в csv заказов, удовлетворяющих запросу
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'ExportOrdersQuerySchema'
        responses:
            200:
                description: список найденных заказов в csv-формате с выделенным набором полей
        """
        search_query, errors = ExportOrdersQuerySchema().load(request.query_params)
        if errors:
            return Response({
                'errors': errors
            }, status.HTTP_400_BAD_REQUEST)

        queryset = search_query.build_queryset()
        try:
            response = StreamingHttpResponse(
                content_type='text/csv; charset=utf-8',
                streaming_content=ExportToCsvOrderSchema().iter_to_csv(queryset, layout='orders'),
            )
            response['Content-Disposition'] = 'attachment; filename="export.csv"'
            return response
        except Exception:
            log.exception('Ошибка выборки: query_params=%s', request.query_params)
            raise

    @list_route(methods=['get'])
    def export_tickets(self, request):
        """Экспорт в csv билетов, удовлетворяющих запросу по заказам
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'ExportOrdersQuerySchema'
        responses:
            200:
                description: список найденных билиетов в csv-формате с выделенным набором полей
        """
        search_query, errors = ExportOrdersQuerySchema().load(request.query_params)
        if errors:
            return Response({
                'errors': errors
            }, status.HTTP_400_BAD_REQUEST)

        queryset = search_query.build_queryset()
        try:
            response = StreamingHttpResponse(
                content_type='text/csv; charset=utf-8',
                streaming_content=ExportToCsvOrderSchema().iter_to_csv(queryset, layout='tickets'),
            )
            response['Content-Disposition'] = 'attachment; filename="export.csv"'
            return response
        except Exception:
            log.exception('Ошибка выборки: query_params=%s', request.query_params)
            raise

    @backoffice_order_detail_route(methods=['post'], url_path='resend-tickets')
    def resend_tickets(self, request, order):
        """Повторно отправить билеты
        ---
        parameters:
          - in: body
            name: post_schema
            schema:
                $ref: 'ResendTicketEmailSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: все получилось
            500:
                schema:
                    $ref: 'ErrorSchema'
        """

        data, errors = ResendTicketEmailSchema().load(request.data)
        if errors:
            return Response({
                'errors': errors,
            }, status=status.HTTP_400_BAD_REQUEST)

        email = data.get('email', order.user_info.email)
        try:
            do_resend_tickets_email(order.uid, email)
        except Exception as e:
            log.exception('Ошибка при отправке письма')
            return Response(ErrorSchema().dump({
                'status': 'error',
                'exception': six.text_type(type(e)),
                'traceback': traceback.format_exc()
            }).data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
        return Response({'status': 'ok'})

    @backoffice_order_detail_route(methods=['post'], url_path='resend-refunded-blanks')
    def resend_refunded_blanks(self, request, order):
        """Повторно отправить бланки возврата
        ---
        parameters:
          - in: body
            name: post_schema
            schema:
                $ref: 'ResendRefundedBlanksEmailSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: все получилось
            500:
                schema:
                    $ref: 'ErrorSchema'

        """
        data, errors = ResendRefundedBlanksEmailSchema().load(request.data)
        if errors:
            return Response({
                'errors': errors,
            }, status=status.HTTP_400_BAD_REQUEST)
        refund_uuid = data['refund_uuid']
        email = data.get('email', order.user_info.email)
        try:
            refund = order.get_refund(refund_uuid)
            send_refund_email(refund, order=order, email=email)
        except Exception as e:
            log.exception('Ошибка при отправке письма о возврате')
            return Response(ErrorSchema().dump({
                'status': 'error',
                'exception': six.text_type(type(e)),
                'traceback': traceback.format_exc()
            }).data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
        else:
            return Response({'status': 'ok'})

    @backoffice_order_detail_route(url_path='unhold-payments')
    def unhold_payments(self, request, order):
        """Разблокирует все платежи по заказу (в статусе Authorized)
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: все получилось
            500:
                schema:
                    $ref: 'ErrorSchema'

        """
        try:
            trust_client = TrustClient()
            update_spec = unhold_order_payments(trust_client, order)
            if update_spec:
                order.modify(**update_spec)
        except Exception as e:
            log.exception('Error in unholding')
            return Response(ErrorSchema().dump({
                'status': 'error',
                'exception': six.text_type(type(e)),
                'traceback': traceback.format_exc()
            }).data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
        else:
            return Response({'status': 'ok'})

    @backoffice_order_detail_route(methods=['get'], url_path='documents')
    def get_documents(self, request, order):
        """Получить документы пассажиров (ФИО + тип документа + номер докумета)
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Документы пассажиров
                schema:
                    $ref: 'DocumentsSchema'
        """
        Document = namedtuple('Document', ('fio', 'doc_type', 'doc_id'))

        order_passengers_by_blank_id = defaultdict(dict)
        for passenger in order.passengers:
            for tiket in passenger.tickets:
                order_passengers_by_blank_id[tiket.blank_id][passenger.doc_id_hash] = passenger

        order_info = get_order_info(order)

        documents = []
        for passenger_info in order_info.passengers:
            doc_id_hash = hash_doc_id(passenger_info.doc_id)
            order_passenger = order_passengers_by_blank_id[passenger_info.blank_id][doc_id_hash]
            documents.append(Document(order_passenger.fio, order_passenger.doc_type, passenger_info.doc_id))

        return Response(DocumentsSchema().dump({'documents': documents}).data)

    @backoffice_order_detail_route(url_path='update-tickets-statuses')
    def update_tickets_statuses(self, request, order):
        """Обновляет статусы билетов у партнера и возвращает заказ
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    $ref: 'BackofficeOrderSchema'
        """

        update_tickets_statuses(order, mode=UpdateTicketStatusMode.TRY_UPDATE_FROM_EXPRESS)
        return Response(BackofficeOrderSchema().dump(order).data)

    @backoffice_order_detail_route()
    def rebooking(self, request, order):
        """Обновляет статусы билетов у партнера и возвращает заказ
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    $ref: 'BackofficeOrderSchema'
        """

        try:
            rebooking_status = rebooking(order)
            return Response({'status': rebooking_status})
        except Exception as e:
            log.exception('Error in rebooking')
            return Response(ErrorSchema().dump({
                'status': 'error',
                'exception': six.text_type(type(e)),
                'traceback': traceback.format_exc()
            }).data, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
