# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import wraps

from rest_framework import status
from rest_framework.decorators import detail_route
from rest_framework.response import Response
from ylog.context import log_context

from common.data_api.billing.trust_client import TrustClient, TrustRefundStatuses, TrustClientException
from common.dev_tools.swagger import swagger_aware_view_set
from travel.rasp.library.python.common23.date import environment
from common.workflow.registry import get_process
from travel.rasp.train_api.train_purchase.backoffice.base import BackofficeViewSet
from travel.rasp.train_api.train_purchase.core.models import TrainRefund, RefundPaymentStatus
from travel.rasp.train_api.train_purchase.utils.order import send_event_to_refund
from travel.rasp.train_api.train_purchase.workflow.ticket_refund import TICKET_REFUND_PROCESS
from travel.rasp.train_api.train_purchase.workflow.user_events import RefundUserEvents

log = logging.getLogger(__name__)

REFUND_PAYMENT_STATUS_ALLOWED_TO_CREATE_ANOTHER_REFUND_PAYMENT = (
    RefundPaymentStatus.FAILED,
    RefundPaymentStatus.UNKNOWN,
    RefundPaymentStatus.TRY_LATER
)


def backoffice_refund_detail_route(*args, **kwargs):
    log_call = kwargs.pop('log_call', True)

    def decorator(view_func):
        @detail_route(*args, **kwargs)
        @wraps(view_func)
        def wrapper(self, request, pk):
            try:
                refund = TrainRefund.objects.get(uuid=pk)
            except TrainRefund.DoesNotExist:
                return Response({
                    'errors': {'uuid': 'Refund was not found'}
                }, status=status.HTTP_404_NOT_FOUND)

            with log_context(order_uid=refund.order_uid):
                if log_call:
                    body = '\nDATA: {}'.format(request.data) if request.data else ''
                    log.info('%s: %s%s', request.method, request.get_full_path(), body)
                return view_func(self, request, refund)

        return wrapper
    return decorator


@swagger_aware_view_set
class BackofficeRefundViewSet(BackofficeViewSet):
    @backoffice_refund_detail_route(methods=['post'])
    def create_another_refund_payment(self, request, refund):
        """Повторить возврат денег пользователю
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uuid возврата
            schema:
                type: string
        responses:
            200:
                description: все получилось
            404:
                description: не нашли нужный возврат
            409:
                description: Неправильный статус платежа для повтора возврата денег пользователю,
                             или возврат находится в unhandled exception state
        """
        refund_payment = refund.refund_payment
        if not refund_payment:
            log.error('Нет ни одного возврата средств для возврата %s', refund.uuid)
            return Response({'errors': {'refund_payment': 'no refund payment'}},
                            status=status.HTTP_400_BAD_REQUEST)

        if refund_payment.payment_resized:
            log.error('Невозможно создать еще один возврат средств для ресайза для возврата %s', refund.uuid)
            return Response({'errors': {'refund_payment': 'Can not do operations on resized payment'}},
                            status=status.HTTP_400_BAD_REQUEST)

        refund_payment_status = refund_payment.refund_payment_status

        if refund_payment_status not in REFUND_PAYMENT_STATUS_ALLOWED_TO_CREATE_ANOTHER_REFUND_PAYMENT:
            log.error('payment_status не подходит для создания нового воврата средств для возврата %s', refund.uuid)
            return Response({
                'errors': {
                    'refund': 'Payment status is not FAILED/UNKNOWN/TRY_LATER, can not create another refund payment'
                },
            }, status=status.HTTP_409_CONFLICT)

        process = get_process(TICKET_REFUND_PROCESS, refund)
        if process.process and process.process.get('state') == process.EXCEPTION_STATE:
            log.error('Нельзя делать возврат, когда процесс находится в unhandled_exception_state, для возврата %s',
                      refund.uuid)
            return Response({
                'errors': {'refund': 'Refund is in unhandled exception state'},
            }, status=status.HTTP_409_CONFLICT)

        send_event_to_refund(refund, RefundUserEvents.CREATE_ANOTHER_REFUND_PAYMENT)
        log.info('Успешно создали возврат средств')
        return Response({
            'ok': True
        })

    @backoffice_refund_detail_route(methods=['post'])
    def update_refund_payment_status(self, request, refund):
        """Проверить и обновить данные возврата денег пользователю
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uuid возврата
            schema:
                type: string
        responses:
            200:
                description: если в ответе есть errors, то возврат денег пользователю еще не прошел,
                             иначе все ок
            400:
                description: нет trust_refund_id или неподходящий payment_status
            404:
                description: не нашли нужный возврат
            500:
                description: не смогли узнать статус возврата в трасте
        """
        refund_payment = refund.refund_payment
        if not refund_payment:
            return Response({
                'errors': {'refund': 'No refund payment'},
            }, status=status.HTTP_400_BAD_REQUEST)

        if refund_payment.payment_resized:
            return update_resized_refund_payment_status(refund_payment)
        else:
            return update_refund_payment_status(refund_payment)


def update_refund_payment_status(refund_payment):
    trust_refund_id = refund_payment.trust_refund_id
    if not trust_refund_id:
        log.error('Нет trust_refund_id для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'refund': 'No trust_refund_id'},
        }, status=status.HTTP_400_BAD_REQUEST)

    refund_payment_status = refund_payment.refund_payment_status
    if refund_payment_status != RefundPaymentStatus.UNKNOWN:
        log.error('payment_status не UNKNOWN для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'refund': 'Invalid payment_status'},
        }, status=status.HTTP_400_BAD_REQUEST)

    try:
        trust_refund_status = TrustClient().get_refund_status(trust_refund_id)
    except TrustClientException:
        log.error('Не смогли узнать статус возврата денег для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'trust': 'Can not get refund status'},
        }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    if trust_refund_status != TrustRefundStatuses.SUCCESS:
        log.error('Статус возврата денег пользователю еще не success для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'trust': 'Refund payment status is not success'},
        }, status=status.HTTP_200_OK)

    refund_payment.modify(
        set__refund_payment_status=RefundPaymentStatus.DONE,
        set__refund_payment_finished_at=environment.now_utc()
    )
    log.info('Успешно обновили статус возврата денег пользователю для возврата %s', refund_payment.refund_uuid)
    return Response({
        'ok': True
    })


def update_resized_refund_payment_status(refund_payment):
    try:
        payment_info = TrustClient().get_payment_info(refund_payment.purchase_token)
    except TrustClientException:
        log.error('Не смогли узнать статус возврата денег для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'trust': 'Can not get refund status'},
        }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    if not payment_info.reversal_id:
        log.error('Еще нет reversal_id в информации о платеже для возврата %s', refund_payment.refund_uuid)
        return Response({
            'errors': {'trust': 'No reversal_id'},
        }, status=status.HTTP_200_OK)

    refund_payment.modify(
        set__refund_payment_status=RefundPaymentStatus.DONE,
        set__refund_payment_finished_at=environment.now_utc(),
        set__trust_reversal_id=payment_info.reversal_id
    )
    log.info('Успешно обновили статус возврата денег пользователю для возврата %s', refund_payment.refund_uuid)
    return Response({
        'ok': True
    })
