# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, fields, post_load
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
from six.moves.urllib_parse import urlencode

from travel.rasp.train_api.train_purchase.backoffice.orders.search import SearchOrdersQuery

BACKOFFICE_ORDERS_URL = 'https://train-purchase-backoffice-ext.rasp.yandex.ru/?'
BACKOFFICE_ORDERS_STANDARD_ARGS = {'offset': 0, 'orderBy': '-reserved_to'}


class SupportOrdersQuerySchema(Schema):
    email = fields.Email(required=True)

    @post_load
    def build_query(self, data):
        return SearchOrdersQuery(data)


class SupportOrdersCountSchema(Schema):
    count = fields.Integer()
    url = fields.String()


@api_view(['GET'])
def orders_count_by_email(request):
    """
    Количество заказов с определенного адреса электронной почты.

    В службе поддержки работа с пользователем завязана на его email.
    Поэтому нам нужно отдавать количественную информацию о заказах по email пока без авторизации.
    В будущем тут должена быть авторизация по TVM.
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'SupportOrdersQuerySchema'
    responses:
        200:
            description: количество найденных заказов
            schema:
                $ref: 'SupportOrdersCountSchema'
    """
    search_query, errors = SupportOrdersQuerySchema().load(request.query_params)
    if errors:
        return Response({
            'errors': errors
        }, status.HTTP_400_BAD_REQUEST)

    url_args = {'email': search_query.data['email']}
    url_args.update(BACKOFFICE_ORDERS_STANDARD_ARGS)
    return Response(SupportOrdersCountSchema().dump({
        'count': search_query.build_queryset().count(),
        'url': BACKOFFICE_ORDERS_URL + urlencode(url_args),
    }).data)
