# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

from rest_framework import status
from rest_framework.response import Response

from common.data_api.billing.trust_client import TrustClientException, TrustClient
from common.utils.try_hard import try_hard
from travel.rasp.train_api.train_purchase.backoffice.base import backoffice_api_view
from travel.rasp.train_api.train_purchase.backoffice.trust_info.serialization import (
    TrustPurchaseInfoSchema, TrustRefundInfoSchema
)
from travel.rasp.train_api.train_purchase.core.models import TrainRefund

TRUST_INFO_MAX_RETRIES = 3
TRUST_INFO_SLEEP_DURATION = timedelta(seconds=1).total_seconds()


@backoffice_api_view(['GET'], admin_only=True)
def purchase_info(request):
    """
    Получение информации из траста о платеже
    Пример запроса:
    /ru/train-purchase-backoffice/trust-info/purchase/?purchaseToken=bd90bacb0390b6ca1781360780afe42f
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrustPurchaseInfoSchema'
    responses:
        200:
            description: информация из траста о платеже
            schema:
               properties:
                  status:
                    description: ok
                    type: string
                  info:
                    description: json напрямую из биллинга
                    type: object
        400:
            description: если нет параметра purchaseToken
        500:
            description: если не смогли получить данные из биллинга
    """
    data, errors = TrustPurchaseInfoSchema().load(request.query_params)
    if errors:
        return Response({'errors': errors}, status.HTTP_400_BAD_REQUEST)

    trust_client = TrustClient()
    info = _get_purchase_info(trust_client, data['purchase_token'])

    return Response({'status': 'ok', 'info': info})


@try_hard(max_retries=TRUST_INFO_MAX_RETRIES, sleep_duration=TRUST_INFO_SLEEP_DURATION,
          retriable_exceptions=(TrustClientException,))
def _get_purchase_info(trust_client, purchase_token):
    return trust_client.get_raw_payment_info(purchase_token)


@backoffice_api_view(['GET'], admin_only=True)
def refund_info(request):
    """
    Получение информации из траста о возврате
    Пример запроса:
    предпочтительно: /ru/train-purchase-backoffice/trust-info/refund/?refundUuid=0e9c75c1b386462688eb1e7c5486663d
    устаревший вызов: /ru/train-purchase-backoffice/trust-info/refund/?trustRefundId=59ea0830795be248dc83d7b6
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrustRefundInfoSchema'
        description: предпочтительно вызывать с параметром refundUuid
    responses:
        200:
            description: информация из траста о возврате
            schema:
               properties:
                  status:
                    description: ok
                    type: string
                  info:
                    description: json из биллинга для возврата, для ресайза такая же информация собранная вручную
                    type: object
        400:
            description: если нет параметра refundUuid или он неправильный, и нет параметра trustRefundId
        500:
            description: если не смогли получить данные из биллинга
    """
    data, errors = TrustRefundInfoSchema().load(request.query_params)
    if errors:
        return Response({'errors': errors}, status.HTTP_400_BAD_REQUEST)

    trust_refund_id = data.get('trust_refund_id')

    trust_client = TrustClient()
    if data.get('refund_uuid'):
        try:
            refund = TrainRefund.objects.get(uuid=data['refund_uuid'])
        except TrainRefund.DoesNotExist:
            return Response({'errors': 'Can not find refund by uuid'}, status.HTTP_400_BAD_REQUEST)

        refund_payment = refund.refund_payment
        if not refund_payment:
            return Response({'errors': 'No refund payment'}, status.HTTP_400_BAD_REQUEST)

        if refund_payment.payment_resized:
            payment_info = try_hard_trust_method(trust_client.get_raw_payment_info, refund_payment.purchase_token)
            info = {
                'status': payment_info.get('status'),
                'fiscal_receipt_url': payment_info.get('fiscal_receipt_clearing_url'),
            }
            return Response({'status': 'ok', 'info': info})

        trust_refund_id = refund_payment.trust_refund_id

    info = try_hard_trust_method(trust_client.get_refund_info, trust_refund_id)
    return Response({'status': 'ok', 'info': info})


@try_hard(max_retries=TRUST_INFO_MAX_RETRIES, sleep_duration=TRUST_INFO_SLEEP_DURATION,
          retriable_exceptions=(TrustClientException,))
def try_hard_trust_method(method, *args, **kwargs):
    return method(*args, **kwargs)
