# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict, namedtuple

from travel.rasp.train_api.train_purchase.core.enums import AgeCode, RoutePolicy

log = logging.getLogger(__name__)

TariffCategory = namedtuple(
    'TariffCategory',
    [
        'code',
        'route_policy',
        'without_place',
        'need_document',
        'min_age',
        'min_age_includes_birthday',
        'max_age',
        'max_age_includes_birthday',
    ],
)

TariffType = namedtuple(
    'TariffType',
    [
        'code',
        'route_policy',
        'provider_code',
        'discount',
        'validators',
    ],
)


class TariffCategories(object):
    categories = [
        TariffCategory(
            code=AgeCode.BABY.value,
            route_policy=RoutePolicy.INTERNAL.value,
            without_place=True,
            need_document=False,
            min_age=0,
            min_age_includes_birthday=False,
            max_age=5,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.CHILD.value,
            route_policy=RoutePolicy.INTERNAL.value,
            without_place=False,
            need_document=False,
            min_age=5,
            min_age_includes_birthday=False,
            max_age=10,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.FULL.value,
            route_policy=RoutePolicy.INTERNAL.value,
            without_place=False,
            need_document=False,
            min_age=10,
            min_age_includes_birthday=False,
            max_age=150,
            max_age_includes_birthday=False,
        ),
        TariffCategory(
            code=AgeCode.BABY.value,
            route_policy=RoutePolicy.INTERNATIONAL.value,
            without_place=True,
            need_document=False,
            min_age=0,
            min_age_includes_birthday=False,
            max_age=4,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.CHILD.value,
            route_policy=RoutePolicy.INTERNATIONAL.value,
            without_place=False,
            need_document=False,
            min_age=4,
            min_age_includes_birthday=False,
            max_age=12,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.FULL.value,
            route_policy=RoutePolicy.INTERNATIONAL.value,
            without_place=False,
            need_document=False,
            min_age=12,
            min_age_includes_birthday=False,
            max_age=150,
            max_age_includes_birthday=False,
        ),
        TariffCategory(
            code=AgeCode.BABY.value,
            route_policy=RoutePolicy.FINLAND.value,
            without_place=True,
            need_document=False,
            min_age=0,
            min_age_includes_birthday=False,
            max_age=6,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.CHILD.value,
            route_policy=RoutePolicy.FINLAND.value,
            without_place=False,
            need_document=False,
            min_age=6,
            min_age_includes_birthday=False,
            max_age=17,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.FULL.value,
            route_policy=RoutePolicy.FINLAND.value,
            without_place=False,
            need_document=False,
            min_age=17,
            min_age_includes_birthday=False,
            max_age=150,
            max_age_includes_birthday=False,
        ),
        TariffCategory(
            code=AgeCode.BABY.value,
            route_policy=RoutePolicy.SUBURBAN.value,
            without_place=True,
            need_document=False,
            min_age=0,
            min_age_includes_birthday=False,
            max_age=5,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.CHILD.value,
            route_policy=RoutePolicy.SUBURBAN.value,
            without_place=False,
            need_document=False,
            min_age=5,
            min_age_includes_birthday=False,
            max_age=7,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.FULL.value,
            route_policy=RoutePolicy.SUBURBAN.value,
            without_place=False,
            need_document=False,
            min_age=7,
            min_age_includes_birthday=False,
            max_age=150,
            max_age_includes_birthday=False,
        ),
        TariffCategory(
            code=AgeCode.CHILD.value,
            route_policy=RoutePolicy.CPPK.value,
            without_place=False,
            need_document=False,
            min_age=0,
            min_age_includes_birthday=False,
            max_age=7,
            max_age_includes_birthday=True,
        ),
        TariffCategory(
            code=AgeCode.FULL.value,
            route_policy=RoutePolicy.CPPK.value,
            without_place=False,
            need_document=False,
            min_age=7,
            min_age_includes_birthday=False,
            max_age=150,
            max_age_includes_birthday=False,
        ),
    ]

    def __init__(self):
        self.directory = defaultdict(list)
        for category in self.categories:
            self.directory[category.route_policy].append(category)

    def find_categories(self, route_policy, is_suburban, is_cppk):
        if is_cppk:
            policy = RoutePolicy.CPPK.value
        elif is_suburban:
            policy = RoutePolicy.SUBURBAN.value
        else:
            policy = route_policy

        result = self.directory.get(policy)
        if not result:
            log.warning('Cannot find TariffCategory for policy "%s"', policy)
            return result

        return result


class TariffTypes(object):
    types = [
        TariffType(
            code='full',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Full',
            discount=0,
            validators='{}',
        ),
        TariffType(
            code='junior',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Junior',
            discount=0.3,
            validators='{"age":{"min":12,"minIncludesBirthday":false,"max":26,"maxIncludesBirthday":true}}',
        ),
        TariffType(
            code='senior',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Senior',
            discount=0.3,
            validators='{"age":{"min":60,"minIncludesBirthday":true}}',
        ),
        TariffType(
            code='happy',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Happy',
            discount=0.35,
            validators='{"birthday":{"daysBefore":7,"daysAfter":7}}',
        ),
        TariffType(
            code='wedding',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Wedding',
            discount=0.35,
            validators='{}',
        ),
        TariffType(
            code='family',
            route_policy=RoutePolicy.INTERNATIONAL.value,
            provider_code='Family',
            discount=0.3,
            validators='{"adultsCount":{"min":2,"max":2},"maxChildrenAges":[[4,4],[12]]}',
        ),
        TariffType(
            code='full',
            route_policy=RoutePolicy.INTERNAL.value,
            provider_code='Full',
            discount=0,
            validators='{}',
        ),
        TariffType(
            code='junior',
            route_policy=RoutePolicy.INTERNAL.value,
            provider_code='Junior',
            discount=0.3,
            validators='{"age":{"min":10,"minIncludesBirthday":false,"max":21,"maxIncludesBirthday":true}}',
        ),
        TariffType(
            code='senior',
            route_policy=RoutePolicy.INTERNAL.value,
            provider_code='Senior',
            discount=0.3,
            validators='{"age":{"min":60,"minIncludesBirthday":true}}',
        ),
    ]

    def __init__(self):
        self._directory_by_provider_codes = {(t.route_policy, t.provider_code): t for t in self.types}

    def find_types_by_provider_codes(self, route_policy, provider_codes):
        result = filter(None, [self._directory_by_provider_codes.get((route_policy, provider_code))
                               for provider_code in provider_codes])

        if not result:
            log.warning('Cannot find TariffType for route policy "%s" and provider codes "%s"',
                        route_policy, provider_codes)
        return result
