# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import random
from datetime import timedelta
from decimal import Decimal

import factory
import faker
from factory.fuzzy import FuzzyChoice
from factory.mongoengine import MongoEngineFactory
from six.moves import zip_longest

from common.apps.train_order.enums import CoachType
from common.models.geo import Country, Station
from common.tester.factories import create_station, create_settlement
from travel.rasp.library.python.common23.tester.helpers.class_counter import inc_class_counter
from common.utils import gen_hex_uuid
from travel.rasp.library.python.common23.date import environment
from common.utils.exceptions import SimpleUnicodeException
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_purchase.core.enums import (
    DocumentType, Gender, GenderChoice, TrainPartner, OrderStatus, TrainPurchaseSource, TrainPartnerCredentialId,
    TravelOrderStatus, RoutePolicy
)
from travel.rasp.train_api.train_purchase.core.models import (
    BillingPayment, ClientContract, ClientContracts, Passenger, RefundStatus, RefundUserInfo, Ticket,
    TicketPayment, PartnerData, TrainOrder, UserInfo, TicketRefund, Tax, RefundBlank, RefundPayment,
    RefundPaymentStatus, RefundEmail, Source, TrainRefund, Payment, Insurance, OrderRouteInfo, StationInfo,
    RebookingInfo, InsuranceProcess, PassengerRebookingInfo, TrainPurchaseSmsVerification
)
from travel.rasp.train_api.train_purchase.core.utils import hash_doc_id


class TicketPaymentFactory(MongoEngineFactory):
    class Meta:
        model = TicketPayment

    amount = 100
    ticket_order_id = '100500'
    fee = 70
    fee_order_id = '200300'
    fee_percent = Decimal('0.1')
    partner_fee = 30
    service_amount = 0
    service_vat = Tax(amount=0, rate=0)

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        if 'fee' not in kwargs and 'amount' in kwargs:
            kwargs['fee'] = 0.15 * kwargs['amount']
        return kwargs


class TicketRefundFactory(MongoEngineFactory):
    class Meta:
        model = TicketRefund

    operation_id = '123'
    amount = 100
    tariff_vat = Tax(amount=1, rate=1)
    service_vat = Tax(amount=2, rate=2)
    commission_fee_vat = Tax(amount=3, rate=3)
    refund_commission_fee_vat = Tax(amount=4, rate=4)


class TicketFactory(MongoEngineFactory):
    class Meta:
        model = Ticket

    blank_id = '123456789'
    places = ['1']
    rzhd_status = 0

    @factory.post_generation
    def refund(obj, create, extracted, **kwargs):
        if extracted is not None:
            if isinstance(extracted, TicketRefundFactory._meta.model):
                obj.refund = extracted
            else:
                obj.refund = TicketRefundFactory(**extracted)
        elif kwargs:
            obj.refund = TicketRefundFactory.simple_generate(create, **kwargs)

    payment = factory.SubFactory(TicketPaymentFactory)


class PassengerRebookingInfoFactory(MongoEngineFactory):
    class Meta:
        model = PassengerRebookingInfo


class PassengerFactory(MongoEngineFactory):
    class Meta:
        model = Passenger

    first_name = factory.Faker('first_name')
    last_name = factory.Faker('last_name')

    sex = FuzzyChoice(Gender)

    doc_type = FuzzyChoice(DocumentType)
    doc_id_hash = None

    citizenship_country_id = Country.RUSSIA_ID

    phone = '79222020123'
    email = 'email@email.com'

    tickets = factory.LazyAttribute(lambda _x: [TicketFactory()])
    customer_id = factory.LazyAttribute(
        lambda _x: str(inc_class_counter(PassengerFactory, 'customer_id', start_value=300000))
    )
    rebooking_info = factory.SubFactory(PassengerRebookingInfoFactory)

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        doc_id = kwargs.pop('doc_id', None)
        if doc_id:
            kwargs['doc_id_hash'] = hash_doc_id(doc_id)
        return kwargs


class UserInfoFactory(MongoEngineFactory):
    class Meta:
        model = UserInfo

    ip = factory.Faker('ipv4')
    region_id = 123
    email = factory.Faker('email')
    phone = '+71234567890'
    yandex_uid = '1212121212121212121'

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        if 'phone' in kwargs:
            kwargs['reversed_phone'] = kwargs['phone'][::-1]
        return kwargs


class RefundUserInfoFactory(MongoEngineFactory):
    class Meta:
        model = RefundUserInfo

    ip = factory.Faker('ipv4')
    region_id = 54


class TrainRefundFactory(MongoEngineFactory):
    class Meta:
        model = TrainRefund

    order_uid = factory.LazyFunction(gen_hex_uuid)
    status = RefundStatus.NEW
    user_info = factory.SubFactory(RefundUserInfoFactory)
    uuid = factory.LazyFunction(gen_hex_uuid)
    blank_ids = ['1']
    created_at = factory.LazyFunction(environment.now_utc)

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        extra_params = kwargs.pop('factory_extra_params', {})
        create_order = extra_params.get('create_order', False)
        if create_order:
            create_order_kwargs = extra_params.get('create_order_kwargs', {})
            kwargs['order_uid'] = create_order_kwargs.get('uid') or kwargs.get('order_uid', gen_hex_uuid())
            all_blank_ids = create_order_kwargs.pop('blank_ids', [])
            passengers = create_order_kwargs.pop('passengers', None)
            if not passengers:
                passengers = [
                    PassengerFactory(tickets=[TicketFactory(
                        blank_id=blank_id,
                        rzhd_status=RzhdStatus.REFUNDED,
                        payment=TicketPaymentFactory(amount=100),
                        refund=TicketRefundFactory(amount=10)
                    )])
                    for blank_id in kwargs['blank_ids']
                ] + [
                    PassengerFactory(tickets=[TicketFactory(
                        blank_id=blank_id,
                        rzhd_status=RzhdStatus.REMOTE_CHECK_IN,
                        payment=TicketPaymentFactory(amount=100)
                    )])
                    for blank_id in all_blank_ids if blank_id not in kwargs['blank_ids']
                ]
            order_kwargs = {
                'uid': kwargs['order_uid'],
                'payments': [{}],
                'passengers': passengers,
            }
            order_kwargs.update(create_order_kwargs)
            TrainOrderFactory(**order_kwargs)
        return kwargs


class BillingPaymentFactory(MongoEngineFactory):
    class Meta:
        model = BillingPayment

    clear_at = factory.Faker('date_time')


class PartnerDataFactory(MongoEngineFactory):
    class Meta:
        model = PartnerData

    im_order_id = factory.Faker('random_int', min=1)
    operation_id = factory.LazyFunction(lambda: str(random.randint(1, 99999)))


class RefundBlankFactory(MongoEngineFactory):
    class Meta:
        model = RefundBlank

    content = b'pdfcontent'


class StationInfoFactory(MongoEngineFactory):
    class Meta:
        model = StationInfo

    @classmethod
    def get_or_create_station(cls, station, station_id, express_code, title, settlement_title):
        if station:
            return station
        if not express_code:
            express_code = faker.Faker().uuid4()
        if station_id:
            try:
                return Station.objects.get(id=station_id)
            except Station.DoesNotExist:
                pass
        return create_station(
            id=station_id,
            title=title,
            __={'codes': {'express': express_code}},
            settlement=create_settlement(title=settlement_title) if settlement_title else None,
        )

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        station = cls.get_or_create_station(
            kwargs.pop('station', None),
            kwargs.pop('id', None),
            kwargs.pop('express', None),
            kwargs.get('title', u'Откуда поезд'),
            kwargs.get('settlement_title', None),
        )
        kwargs['id'] = station.id
        return kwargs


class OrderRouteInfoFactory(MongoEngineFactory):
    class Meta:
        model = OrderRouteInfo

    start_station = factory.SubFactory(StationInfoFactory)
    end_station = factory.SubFactory(StationInfoFactory)
    from_station = factory.SubFactory(StationInfoFactory)
    to_station = factory.SubFactory(StationInfoFactory)


class RebookingInfoFactory(MongoEngineFactory):
    class Meta:
        model = RebookingInfo


class InsuranceProcessFactory(MongoEngineFactory):
    class Meta:
        model = InsuranceProcess


class TrainOrderFactory(MongoEngineFactory):
    class Meta:
        model = TrainOrder

    status = OrderStatus.RESERVED
    travel_status = TravelOrderStatus.RESERVED
    partner = TrainPartner.IM
    partner_credential_id = TrainPartnerCredentialId.IM
    train_number = '001A'
    train_ticket_number = '002A'
    gender = FuzzyChoice(GenderChoice)
    departure = factory.Faker('date_time')
    arrival = factory.Faker('date_time')
    coach_type = CoachType.COMPARTMENT
    coach_number = '2'
    coach_owner = 'ФПК СЕВЕРНЫЙ'
    displayed_coach_owner = 'ФПК'
    user_info = factory.SubFactory(UserInfoFactory)
    partner_data_history = factory.LazyAttribute(lambda _x: [PartnerDataFactory()])
    passengers = factory.LazyAttribute(lambda _x: [PassengerFactory()])
    reserved_to = factory.LazyAttribute(lambda _x: environment.now_utc() + timedelta(minutes=15))
    process = {}
    rebooking_info = factory.SubFactory(RebookingInfoFactory)
    route_info = factory.SubFactory(OrderRouteInfoFactory)
    route_policy = RoutePolicy.INTERNAL
    insurance = factory.SubFactory(InsuranceProcessFactory)

    @classmethod
    def get_or_create_station(cls, station, station_id, express_code, title):
        if station:
            return station
        if not express_code:
            express_code = faker.Faker().uuid4()
        if station_id:
            try:
                return Station.objects.get(id=station_id)
            except Station.DoesNotExist:
                return create_station(id=station_id, title=title, __={'codes': {'express': express_code}})
        return create_station(title=title, __={'codes': {'express': express_code}})

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        cls._check_dts(kwargs)
        uid = kwargs.setdefault('uid', gen_hex_uuid())
        partner_data = kwargs.pop('partner_data', None)
        if partner_data:
            kwargs['partner_data_history'] = [partner_data]
        updated_from_billing_at = kwargs.get('updated_from_billing_at')
        payments = kwargs.pop('payments', [{}])
        for p in payments:
            if isinstance(p, dict):
                p.setdefault('order_uid', uid)
                p.setdefault('updated_from_billing_at', updated_from_billing_at)
                PaymentFactory(**p)
            elif isinstance(p, Payment):
                p.update(set__order_uid=uid)
            else:
                raise SimpleUnicodeException('платежи должны быть типа Payment или dict')

        station_from = cls.get_or_create_station(
            kwargs.pop('station_from', None),
            kwargs.pop('station_from_id', None),
            kwargs.pop('express_from', None),
            kwargs.pop('station_from_title', u'Откуда'),
        )
        kwargs['station_from_id'] = station_from.id

        station_to = cls.get_or_create_station(
            kwargs.pop('station_to', None),
            kwargs.pop('station_to_id', None),
            kwargs.pop('express_to', None),
            kwargs.pop('station_to_title', u'Куда'),
        )
        kwargs['station_to_id'] = station_to.id

        return kwargs

    @classmethod
    def _check_dts(cls, kwargs):
        for key in ('arrival', 'departure'):
            if key in kwargs:
                dt = kwargs[key]
                if dt and dt.tzinfo is not None:
                    raise SimpleUnicodeException('{} должен быть наивным'.format(key))


class ClientContractFactory(MongoEngineFactory):
    class Meta:
        model = ClientContract

    start_dt = factory.Faker('date_time')
    finish_dt = factory.Faker('date_time')
    is_active = True
    is_cancelled = False
    is_suspended = False
    partner_commission_sum = Decimal('10.0')
    partner_commission_sum2 = Decimal('10.0')


class ClientContractsFactory(MongoEngineFactory):
    class Meta:
        model = ClientContracts

    updated_at = factory.LazyFunction(environment.now)
    partner = TrainPartner.IM
    contracts = factory.LazyAttribute(lambda _x: [ClientContractFactory()])


class RefundPaymentFactory(MongoEngineFactory):
    """
    RefundPaymentFactory(factory_extra_params={'create_order': True})
    создаст еще и заказ с возвратом, который соответствует создаваемому RefundPayment
    """

    class Meta:
        model = RefundPayment

    order_uid = factory.LazyFunction(gen_hex_uuid)
    purchase_token = 'some-purchase-token'
    trust_refund_id = 'some-trust-refund-id'
    refund_uuid = factory.LazyFunction(gen_hex_uuid)
    refund_payment_status = RefundPaymentStatus.UNKNOWN
    refund_created_at = factory.LazyFunction(environment.now_utc)
    refund_blank_ids = ['1']
    user_info = factory.SubFactory(RefundUserInfoFactory)

    @classmethod
    def _adjust_kwargs(cls, **kwargs):
        extra_params = kwargs.pop('factory_extra_params', {})
        create_order = extra_params.get('create_order', False)
        if create_order:
            kwargs['order_uid'] = kwargs.get('order_uid', gen_hex_uuid())
            kwargs['refund_uuid'] = kwargs.get('refund_uuid', gen_hex_uuid())
            kwargs['refund_payment_status'] = kwargs.get('refund_payment_status', RefundPaymentStatus.UNKNOWN)
            kwargs['refund_blank_ids'] = kwargs.get('refund_blank_ids', ['1'])
            kwargs['refund_insurance_ids'] = kwargs.get('refund_insurance_ids', [])
            kwargs['trust_refund_id'] = kwargs.get('trust_refund_id', 'some-trust-refund-id')
            kwargs['refund_created_at'] = kwargs.get('refund_created_at', environment.now_utc())
            kwargs['refund_payment_finished_at'] = kwargs.get('refund_payment_finished_at')
            kwargs['purchase_token'] = kwargs.get('purchase_token', 'some-purchase-token')
            passengers = []
            for blank_id, insurance_id in zip_longest(kwargs['refund_blank_ids'], kwargs['refund_insurance_ids']):
                ticket_kwargs = dict(payment=TicketPaymentFactory(amount=100), refund=TicketRefundFactory(amount=10))
                if blank_id:
                    ticket_kwargs['blank_id'] = blank_id
                insurance = InsuranceFactory(operation_id=insurance_id) if insurance_id else InsuranceFactory()
                passengers.append(PassengerFactory(tickets=[TicketFactory(**ticket_kwargs)], insurance=insurance))
            TrainOrderFactory(
                uid=kwargs['order_uid'],
                payments=[
                    dict(purchase_token=kwargs['purchase_token'])
                ],
                passengers=passengers,
            )
            is_active_refund = extra_params.get('is_active_refund', True)
            TrainRefundFactory(
                is_active=is_active_refund,
                order_uid=kwargs['order_uid'],
                uuid=kwargs['refund_uuid'],
                blank_ids=kwargs['refund_blank_ids'],
                insurance_ids=kwargs['refund_insurance_ids'],
            )
            if not is_active_refund:
                TrainRefundFactory(
                    is_active=True,
                    order_uid=kwargs['order_uid'],
                )
        return kwargs


class RefundEmailFactory(MongoEngineFactory):
    class Meta:
        model = RefundEmail

    order_uid = factory.LazyFunction(gen_hex_uuid)
    refund_uuid = factory.LazyFunction(gen_hex_uuid)
    created_at = factory.LazyFunction(environment.now_utc)


class SourceFactory(MongoEngineFactory):
    class Meta:
        model = Source

    req_id = factory.LazyFunction(lambda: str(random.randint(1, 99999)))
    gclid = factory.LazyFunction(lambda: str(random.randint(1, 999999999)))
    device = TrainPurchaseSource.DESKTOP
    from_ = 'some from'
    terminal = 'some terminal'


class PaymentFactory(MongoEngineFactory):
    class Meta:
        model = Payment

    uid = factory.LazyFunction(gen_hex_uuid)
    order_uid = factory.LazyFunction(gen_hex_uuid)
    trust_created_at = factory.LazyFunction(environment.now_utc)


class InsuranceFactory(MongoEngineFactory):
    class Meta:
        model = Insurance

    operation_id = factory.LazyFunction(lambda: str(random.randint(1, 999999999)))
    amount = Decimal(100)
    compensation = Decimal(1000)
    compensation_variants = []
    company = 'Renessans'
    package = 'AccidentWithFloatPremium'
    provider = 'P3'


class TrainPurchaseSmsVerificationFactory(MongoEngineFactory):
    class Meta:
        model = TrainPurchaseSmsVerification

    phone = '1111'
    code = 'verification_code'
    used = False
    sent_at = factory.Faker('date_time')
    message = 'message'
    action_name = 'testing'
