# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from decimal import Decimal

import six
from django.conf import settings
from mongoengine import Document, BinaryField, ObjectIdField, QuerySet
from mongoengine.document import EmbeddedDocument
from mongoengine.fields import (
    BooleanField, EmbeddedDocumentField, DateTimeField, DecimalField, DynamicField, IntField,
    ListField, StringField, DictField, EmbeddedDocumentListField,
)
from pymongo.collation import Collation, CollationStrength

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.data_api.billing.trust_client import TrustClient
from common.db.mongo.fields import StringEnumField
from common.db.mongo.representable import representable_document
from common.models.geo import Country, Station
from common.settings import MONTHS
from common.utils import gen_hex_uuid
from travel.rasp.library.python.common23.date import environment
from common.utils.date import UTC_TZ
from common.workflow.process import Process
from travel.rasp.train_api.train_purchase.core import config
from travel.rasp.train_api.train_purchase.core.enums import (
    DocumentType, Gender, GenderChoice, OrderStatus, PlacesType, TrainPartner, TrainPurchaseSource, OperationStatus,
    RebookingStatus, Arrangement, AgeGroup, LoyaltyCardType, PlacesOption, TrainPartnerCredentialId, TravelOrderStatus,
    InsuranceStatus, RoutePolicy
)

log = logging.getLogger(__name__)
GET_ROUTE_INFO_TIMEOUT_SEC = 3


def _make_case_insensitive_meta_index(*fields, **extra_meta):
    return dict({
        'fields': list(fields),
        'collation': Collation(locale=settings.LANGUAGE_CODE, strength=CollationStrength.PRIMARY)
    }, **extra_meta)


class TrainPurchaseBaseDocument(Document):
    meta = {
        'abstract': True,
        'strict': False,
        'auto_create_index': False,
        'db_alias': 'train_purchase',
    }


class NonStrictEmbeddedDocument(EmbeddedDocument):
    meta = {
        'abstract': True,
        'strict': False
    }


RepresentableEmbeddedDocument = representable_document(NonStrictEmbeddedDocument)


class Place(RepresentableEmbeddedDocument):
    number = IntField(min_value=1, required=True)
    is_upper = BooleanField(required=True)


class RefundBlank(TrainPurchaseBaseDocument):
    content = BinaryField(required=True)


class Tax(RepresentableEmbeddedDocument):
    """ Ставка и сумма налога """
    amount = DecimalField(required=True, precision=2, verbose_name='Сумма')
    rate = DecimalField(required=True, precision=2, verbose_name='Ставка в процентах')

    def calculate_full_amount(self):
        if not self.rate:  # невозможно рассчитать облагаемую сумму
            return None

        amount = self.amount
        full_rate = self.rate / (Decimal('100') + self.rate)
        full_amount = amount / full_rate

        for exp in (Decimal('0.'), Decimal('0.0'), Decimal('0.00')):
            rounded_amount = full_amount.quantize(exp)
            if (rounded_amount * full_rate).quantize(amount) == amount:
                return rounded_amount

        return full_amount


class TicketPayment(RepresentableEmbeddedDocument):
    """ суммы в рублях """
    amount = DecimalField(required=True, precision=2, verbose_name='Стоимость билета в РЖД')
    ticket_order_id = StringField(verbose_name='trust.order_id для билета')
    service_order_id = StringField(verbose_name='trust.order_id для сервиса')
    service_amount = DecimalField(required=True, precision=2,
                                  verbose_name='Стоимость постельного белья, уже включена в amount')
    service_fee = DecimalField(default=Decimal(0), precision=2, verbose_name='Сбор за белье, уже включен в fee')
    fee = DecimalField(required=True, precision=2, verbose_name='Комиссия Яндекса')
    fee_order_id = StringField(verbose_name='trust.order_id для комиссии Яндекса')
    fee_percent = DecimalField(precision=4, verbose_name='Процент комиссии Яндекса')
    fee_percent_range = DecimalField(precision=4, verbose_name='Диапазон процента комиссии Яндекса')
    partner_fee = DecimalField(required=True, precision=2, verbose_name='Комиссия партнёра')
    partner_refund_fee = DecimalField(precision=2, default=Decimal('31.70'),
                                      verbose_name='Комиссия партнёра за возврат')
    tariff_vat = EmbeddedDocumentField(Tax, verbose_name='НДС со стоимости перевозки')
    service_vat = EmbeddedDocumentField(Tax, verbose_name='НДС со стоимости сервиса')
    commission_fee_vat = EmbeddedDocumentField(
        Tax, verbose_name='НДС со стоимости комиссионного сбора и дополнительных услуг')

    @property
    def total(self):
        return self.amount + self.fee

    @property
    def ticket_amount(self):
        return self.amount - self.service_amount

    @property
    def refundable_yandex_fee_amount(self):
        refundable_fee_amount = self.fee - self.partner_fee - self.partner_refund_fee
        return max(Decimal(0), refundable_fee_amount)


class TicketRefund(RepresentableEmbeddedDocument):
    operation_id = StringField()
    amount = DecimalField(precision=2, verbose_name='Сумма к возврату')
    tariff_vat = EmbeddedDocumentField(Tax, verbose_name='Возвращаемый НДС со стоимости тарифа')
    service_vat = EmbeddedDocumentField(Tax, verbose_name='Возвращаемый НДС со стоимости сервиса')
    commission_fee_vat = EmbeddedDocumentField(Tax, verbose_name='Возвращаемый НДС со стоимости комиссионного сбора')
    refund_commission_fee_vat = EmbeddedDocumentField(
        Tax, verbose_name='Возвращаемый НДС со стоимости комиссионного сбора за возврат')
    refund_yandex_fee_amount = DecimalField(precision=2, default=Decimal(0),
                                            verbose_name='Возвращаемая комиссия Яндекса')
    blank_id = ObjectIdField()

    def get_blank(self):
        if not self.blank_id:
            return None
        return RefundBlank.objects(pk=self.blank_id).first()


class Ticket(RepresentableEmbeddedDocument):
    blank_id = StringField(required=True)
    places = ListField(StringField(), default=[])
    places_type = StringEnumField(PlacesType)
    pending = BooleanField(help_text='Статус бланка не уточнен. Менять статус электронной регистрации нельзя',
                           default=False)
    rzhd_status = IntField()  # возможные значения в train_api.train_partners.base.RzhdStatus
    payment = EmbeddedDocumentField(TicketPayment)
    refund = EmbeddedDocumentField(TicketRefund)
    tariff_info_code = StringField()
    raw_tariff_title = StringField()
    carrier_inn = StringField(help_text='ИНН перевозчика')

    @property
    def tariff_info(self):
        tariff_info_code = self.tariff_info_code
        if tariff_info_code is not None:
            return TariffInfo.objects.get(code=tariff_info_code)


class PartnerData(RepresentableEmbeddedDocument):
    im_order_id = IntField(min_value=0, null=True)
    operation_id = StringField()
    order_num = StringField()
    check_transaction_counter = IntField(default=0)
    expire_set_er = DateTimeField()
    special_notice = StringField()
    time_notice = StringField()
    station_from_title = StringField()
    station_to_title = StringField()
    start_station_title = StringField(help_text='Название станции отправления поезда')
    end_station_title = StringField(help_text='Название конечной станции поезда')
    compartment_gender = StringEnumField(GenderChoice, default=GenderChoice.MIXED)
    is_three_hours_reservation_available = BooleanField(default=False)
    is_reservation_prolonged = BooleanField(default=False, verbose_name='Продлено ли время брони. '
                                                                        'Фронт будет передавать этот флаг в метрику')
    is_order_cancelled = BooleanField(default=False)
    operation_status = StringEnumField(OperationStatus)
    reservation_datetime = DateTimeField(verbose_name='Время бронирования')
    is_suburban = BooleanField(default=None)
    is_only_full_return_possible = BooleanField(default=False, verbose_name='Возможен только полный возврат')


class BillingPayment(RepresentableEmbeddedDocument):
    clear_at = DateTimeField()
    purchase_token = StringField(verbose_name='Идентификатор корзины в TRUST')
    payment_url = StringField(verbose_name='URL для iframe-а оплаты')
    create_payment_counter = IntField(default=0)
    last_payment_created_at = DateTimeField()
    immediate_return = BooleanField(default=False, verbose_name='Используем продовый биллинг в тестинге')
    status = StringField()
    resp_code = StringField()
    resp_desc = StringField()

    @property
    def trust_created_at(self):
        return self.last_payment_created_at


class PassengerLoyaltyCards(RepresentableEmbeddedDocument):
    type = StringEnumField(LoyaltyCardType, help_text='Тип карты')
    number = StringField(help_text='Номер карты')

    @classmethod
    def to_dict(cls, model):
        return {
            'type': model.type,
            'number': model.number,
        } if not isinstance(model, dict) else model


class PassengerTariffInfo(RepresentableEmbeddedDocument):
    im_request_code = StringField(help_text='Код тарифа для запроса в ИМ')

    @classmethod
    def to_dict(cls, model):
        return {
            'im_request_code': model.im_request_code
        } if not isinstance(model, dict) else model


class PassengerRebookingInfo(RepresentableEmbeddedDocument):
    age_group = StringEnumField(AgeGroup, help_text='Возрастная группа пассажира')
    loyalty_cards = EmbeddedDocumentListField(PassengerLoyaltyCards, help_text='Информация о картах лояльности')
    tariff_info = EmbeddedDocumentField(PassengerTariffInfo, help_text='Информация о тарифе')


class InsuranceCompensation(RepresentableEmbeddedDocument):
    event = StringField(required=True, help_text='Вид страхового случая')
    compensation = DecimalField(required=True, help_text='Размер компенсации')


class Insurance(RepresentableEmbeddedDocument):
    amount = DecimalField(required=True, help_text='Стоимость страховки')
    package = StringField(help_text='Код страхового продукта')
    provider = StringField(help_text='Код провайдера')
    company = StringField(help_text='Наименование страховой компании')
    compensation = DecimalField(required=True, help_text='Страховая сумма')
    compensation_variants = EmbeddedDocumentListField(InsuranceCompensation, default=[], help_text='deprecated')
    operation_id = StringField(help_text='Id позиции страхового продукта в заказе')
    trust_order_id = StringField(help_text='trust.order_id для страховки')
    operation_status = StringEnumField(OperationStatus)
    refund_uuid = StringField(max_length=64, min_length=32,
                              help_text='UUID возврата страховки или билета со страховкой')


class Passenger(RepresentableEmbeddedDocument):
    first_name = StringField(required=True)
    last_name = StringField(required=True)
    patronymic = StringField()
    sex = StringEnumField(Gender, required=True)  # TODO: Переименовать в gender
    birth_date_hash = StringField()
    age = DecimalField(
        help_text='Возраст пассажира на дату отправления с начальной станции (в ж/д часовой зоне станции)')
    doc_type = StringEnumField(DocumentType, required=True)
    doc_id_hash = StringField()
    citizenship_country_id = IntField(min_value=1, required=True)
    tickets = ListField(EmbeddedDocumentField(Ticket))
    rebooking_info = EmbeddedDocumentField(PassengerRebookingInfo, help_text='Данные для перебронирования')
    customer_id = StringField(help_text='Id заказчика билета')
    insurance = EmbeddedDocumentField(Insurance, help_text='Данные для страхования пассажира')
    phone = StringField(help_text='Контактный номер телефона')
    email = StringField(help_text='Контактная электронная почта')

    @property
    def citizenship_country(self):
        return Country.objects.get(id=self.citizenship_country_id)

    @property
    def fio(self):
        return '{} {} {}'.format(self.last_name, self.first_name, self.patronymic or '').rstrip()

    @property
    def total_amount(self):
        return sum(t.payment.amount for t in self.tickets)

    @property
    def total_ticket_amount(self):
        return sum(t.payment.ticket_amount for t in self.tickets)

    @property
    def total_service_amount(self):
        return sum(t.payment.service_amount for t in self.tickets)

    @property
    def total_tickets_fee(self):
        return sum(t.payment.fee for t in self.tickets)

    @property
    def total_partner_fee(self):
        return sum(t.payment.partner_fee for t in self.tickets)

    @property
    def total_insurance(self):
        return self.insurance.amount if self.insurance and self.insurance.trust_order_id else Decimal(0)

    @property
    def total(self):
        return self.total_amount + self.total_tickets_fee + self.total_insurance

    @property
    def total_fee(self):
        return self.total_tickets_fee + self.insurance_fee

    @property
    def insurance_fee(self):
        return Decimal('0.65') * self.total_insurance


class Source(RepresentableEmbeddedDocument):
    req_id = StringField(help_text='req_id при переходе человека из колдунщика')
    device = StringEnumField(TrainPurchaseSource, help_text='Тип устройства, с которого сделан заказ')
    utm_source = StringField()
    utm_medium = StringField()
    utm_campaign = StringField()
    utm_term = StringField()
    utm_content = StringField()
    from_ = StringField()
    gclid = StringField(help_text='Уникальный идентификатор при клике по объявлению в Гугле')
    terminal = StringField(help_text='Описание терминала для передачи в Trust')
    partner = StringField(help_text='Идентификатор партнёрской сети')
    subpartner = StringField(help_text='Идентификатор вебмастера')
    partner_uid = StringField(help_text='Идентификатор перехода из партнёрской сети')
    test_id = StringField(help_text='ID эксперимента в колдунщиках/поиске')
    test_buckets = StringField(help_text='Корзины колдовских/поисковых экспериментов')
    icookie = StringField(help_text='Новый идентификатор пользователя, замена yandexuid')
    serp_uuid = StringField(help_text='Идентификатор пользователя в поисковом приложении')
    is_transfer = BooleanField(help_text='Является частью общего маршрута с пересадками')


class OrderEmailFlags(RepresentableEmbeddedDocument):
    """
    Отдельная коллекция для флагов отправки служебных писем,
    например, об ошибках или странном состоянии заказа.

    Странных состояний заказа может быть много, значит, и разных писем может быть много,
    поэтому флаги вынесены в отдельный поддокумент.
    """
    problem_order_email_is_sent = BooleanField(default=False, help_text='Письмо о проблемном заказе было отправлено')


class UserInfoBase(RepresentableEmbeddedDocument):
    meta = {'abstract': True}

    ip = StringField(required=True)
    uid = StringField(help_text='uid пользователя в Яндекс.Паспорте, он же passport_id')
    region_id = IntField(required=True)


class UserInfo(UserInfoBase):
    email = StringField(required=True)
    is_mobile = BooleanField(default=False)
    phone = StringField(default=None)
    reversed_phone = StringField(default=None)
    yandex_uid = StringField(default=None)


class RefundUserInfo(UserInfoBase):
    pass


class RefundStatus(object):
    NEW = 'new'
    FAILED = 'failed'
    PARTNER_REFUND_UNKNOWN = 'partner_refund_unknown'
    PARTNER_REFUND_DONE = 'partner_refund_done'
    DONE = 'done'

    @classmethod
    def get_choices(cls):
        return [
            (cls.NEW, cls.NEW),
            (cls.FAILED, cls.FAILED),
            (cls.PARTNER_REFUND_UNKNOWN, cls.PARTNER_REFUND_UNKNOWN),
            (cls.PARTNER_REFUND_DONE, cls.PARTNER_REFUND_DONE),
            (cls.DONE, cls.DONE),
        ]


class RefundPaymentStatus(object):
    NEW = 'new'
    FAILED = 'failed'
    UNKNOWN = 'unknown'
    DONE = 'done'
    TRY_LATER = 'try_later'  # TODO: remove this
    CREATED = 'created'

    @classmethod
    def get_choices(cls):
        return [
            (cls.NEW, cls.NEW),
            (cls.FAILED, cls.FAILED),
            (cls.UNKNOWN, cls.UNKNOWN),
            (cls.DONE, cls.DONE),
            (cls.TRY_LATER, cls.TRY_LATER),
            (cls.CREATED, cls.CREATED),
        ]


class TrainRefund(TrainPurchaseBaseDocument):
    # general fields
    uuid = StringField(max_length=64, min_length=32, required=True, unique=True, default=gen_hex_uuid)
    order_uid = StringField(max_length=64, min_length=32, required=True)
    is_active = BooleanField(default=False, help_text='Признак возврата, находящегося в обработке')
    user_info = EmbeddedDocumentField(RefundUserInfo, required=True)
    email_is_sent = BooleanField(default=False, help_text='Письмо поставленно в очередь гарантированной отправки')
    problem_refund_email_is_sent = BooleanField(default=False)
    process = DynamicField(default={})  # workflow data

    # partner refund fields
    status = StringField(required=True, default=RefundStatus.NEW, choices=RefundStatus.get_choices())
    blank_ids = ListField(StringField(min_length=1), default=[])
    insurance_ids = ListField(StringField(min_length=1), default=[])
    has_sending_attempt = BooleanField(default=False)
    is_partial_failed = BooleanField(help_text='Часть билетов вернули, а часть нет', default=False)
    is_external = BooleanField(default=False, help_text='Признак кассового возврата')
    created_at = DateTimeField(required=True, help_text='Время создания возврата, '
                                                        'для старых записей не будет совпадать с временем в _id')
    finished_at = DateTimeField(help_text='Время завершения операции возврата билета '
                                          'для ежемесячных отчётов')  # RASPFRONT-4426

    @property
    def order(self):
        return TrainOrder.objects.get(uid=self.order_uid)

    @property
    def refund_payment(self):
        refund_payments = RefundPayment.objects.filter(refund_uuid=self.uuid).order_by('-_id')
        return refund_payments[0] if refund_payments.count() > 0 else None

    meta = {
        'indexes': [
            'uuid',
            # Текущий возврат в заказе может быть только один
            {'fields': ('order_uid', 'is_active'), 'unique': True, 'partialFilterExpression': {'is_active': True}},
            'order_uid',
            ('order_uid', '-created_at'),
            'status',
            'is_partial_failed',
            'created_at',
            'finished_at',
            'process.external_events_count',
            'process.lock_modified',
            'process.lock_uid',
            'process.state',
        ],
        'index_background': True
    }


class ErrorInfo(RepresentableEmbeddedDocument):
    type = StringField(required=True)
    message = StringField(default=None)
    data = DictField()


class QuerySetWithCollation(QuerySet):
    _collation = None

    def collation(self, **kwargs):
        queryset = self.clone()
        queryset._collation = Collation(**kwargs)
        return queryset

    def clone(self):
        res = super(QuerySetWithCollation, self).clone()
        res._collation = self._collation
        return res

    @property
    def _cursor(self):
        if self._cursor_obj is not None:
            return self._cursor_obj
        c = super(QuerySetWithCollation, self)._cursor
        if self._collation:
            c.collation(self._collation)
        return c


class Requirements(RepresentableEmbeddedDocument):
    arrangement = StringEnumField(Arrangement, help_text='Требования к местам')
    storey = IntField(min_value=0, max_value=2, help_text='Требуемый этаж вагона')
    count = DynamicField(help_text='Требуемое распределение мест')


class RebookingInfo(RepresentableEmbeddedDocument):
    enabled = BooleanField(help_text='Автоматическое перебронирование билетов включено')
    cycle_until = DateTimeField(help_text='Время жизни цикла перебронирования')
    cycle_number = IntField(help_text='Номер цикла перебронирования')

    status = StringEnumField(RebookingStatus, help_text='Статус запущенного вручную перебронирования')
    bedding = BooleanField(help_text='Требуется постельное белье')
    electronic_registration = BooleanField(help_text='Требуется электронная регистрация')
    place_demands = StringEnumField(PlacesOption, help_text='Дополнительные требования к местам')
    service_class = StringField(help_text='Класс обслуживания')
    international_service_class = StringField(help_text='Международный класс обслуживания')
    places = DynamicField(help_text='Выбранные места')
    requirements = EmbeddedDocumentField(Requirements, help_text='Требования к местам')
    additional_place_requirements = StringField(help_text='Дополнительные требования к местам')
    give_child_without_place = BooleanField(help_text='Оформить детский билет без места')
    is_cppk = BooleanField(help_text='Признак ЦППК')


class StationInfo(RepresentableEmbeddedDocument):
    id = IntField(min_value=1, required=True, help_text='ID станции')
    title = StringField(help_text='Название станции')
    settlement_title = StringField(help_text='Название населенного пункта')
    departure = DateTimeField(help_context='Время отправления поезда')

    @classmethod
    def create_from_station(cls, station):
        info = StationInfo(
            id=station.id,
            title=station.L_title(),
            settlement_title=station.settlement.L_title() if station.settlement else None,
        )
        return info


class OrderRouteInfo(RepresentableEmbeddedDocument):
    start_station = EmbeddedDocumentField(StationInfo, help_text='Станция отправления поезда')
    end_station = EmbeddedDocumentField(StationInfo, help_text='Станция прибытия поезда')
    from_station = EmbeddedDocumentField(StationInfo, help_text='Станция отправления пассажира')
    to_station = EmbeddedDocumentField(StationInfo, help_text='Станция прибытия пассажира')


class InsuranceProcess(RepresentableEmbeddedDocument):
    status = StringEnumField(InsuranceStatus, help_text='Текущий статус процесса страховок')


@six.python_2_unicode_compatible
class TrainOrder(TrainPurchaseBaseDocument):
    class RefundNotFound(Exception):
        pass

    uid = StringField(max_length=64, min_length=32, unique=True, required=True)
    partner = StringEnumField(TrainPartner, required=True)
    partner_credential_id = StringEnumField(TrainPartnerCredentialId, default=TrainPartnerCredentialId.IM,
                                            required=True, help_text='Реквизиты обращения к партнеру')
    status = StringEnumField(OrderStatus, required=True)
    finished_at = DateTimeField(help_text='Время завершения '
                                          'операции покупки для ежемесячных отчётов')  # RASPFRONT-4426
    error = EmbeddedDocumentField(ErrorInfo)
    lang = StringField()  # XXX зачем это поле?
    process = DynamicField(default={})  # workflow data
    admin_comment = StringField(help_text='Заполняем при ручных действиях с заказом, '
                                          'например, при полном возврате денег')
    station_from_id = IntField(min_value=1, required=True)  # перенести в route_info?
    station_to_id = IntField(min_value=1, required=True)
    train_number = StringField(required=True)
    train_ticket_number = StringField(required=True)
    train_name = StringField(help_text='Фирменное название поезда, например, "Малахит"')
    departure = DateTimeField(required=True)
    arrival = DateTimeField(required=True)

    coach_type = StringEnumField(CoachType, required=True, db_field='car_type')
    coach_number = StringField(required=True, db_field='car_number')
    coach_owner = StringField(null=True, help_text='Перевозчик из ответа партнера на создание брони.'
                                                   'Например, "ФПК СЕВЕРНЫЙ"')
    displayed_coach_owner = StringField(null=True, help_text='Перевозчик из ответа партнера о повагонной информации.'
                                                             'Показываем в ЛК, например, "ФПК"')
    two_storey = BooleanField(default=False)
    gender = StringEnumField(GenderChoice, null=True)

    passengers = ListField(EmbeddedDocumentField(Passenger))
    reserved_to = DateTimeField()
    is_order_expired = BooleanField(default=None, help_text='Признак что время брони истекло')
    refund_expired_at = DateTimeField()
    partner_data_history = ListField(EmbeddedDocumentField(PartnerData))
    user_info = EmbeddedDocumentField(UserInfo, required=True)
    invalid_payments_unholded = BooleanField(default=None, help_text='Анхолд кривых платежей: None - не производился, '
                                                                     'True - успешно произведен, False - неуспешно')
    updated_from_billing_at = DateTimeField(help_text='Время обновления статусов и расшифровок платежей из биллинга')
    orders_created = ListField(StringField(min_length=32, max_length=64))
    source = EmbeddedDocumentField(Source, help_text='Подмножество данных для аналитики')
    email_flags = EmbeddedDocumentField(OrderEmailFlags,
                                        help_text='Флаги отправки сообщений о странных состояниях заказа')
    max_pending_till = DateTimeField(help_text='Время максимального ожидания обновления')  # TRAINS-33
    rebooking_info = EmbeddedDocumentField(RebookingInfo, help_text='Данные для перебронирования')
    # TODO вынести insurance-свойства в общую подструктуру
    insurance = EmbeddedDocumentField(InsuranceProcess, help_text='Информация по процессу страховок')
    insurance_enabled = BooleanField(help_text='Использование страховок разрешено')
    insurance_auto_return_uuid = StringField(
        max_length=64, min_length=32, help_text='UUID автоматического возврата страховки после неудачного оформления'
    )
    route_info = EmbeddedDocumentField(OrderRouteInfo, required=True, help_text='Точки маршрута для поиска')
    travel_status = StringEnumField(TravelOrderStatus, help_text='Статус портала путешествий')
    price_exp_id = StringField(help_text='Идентификатор выдачи с фронта')
    scheme_id = IntField(help_text='Идентификатор схемы')
    route_policy = StringEnumField(RoutePolicy, help_text='Направление')

    def try_update_route_info(self):
        from travel.rasp.train_api.train_partners.base.get_route_info import get_route_info_for_order
        result = get_route_info_for_order(self, GET_ROUTE_INFO_TIMEOUT_SEC)
        if result:
            try:
                start_station = Station.get_by_code('express', result.first_stop.station_express_code)
                self.route_info.start_station = StationInfo.create_from_station(start_station)
            except Station.DoesNotExist:
                log.warning('Cannot get station by code %s', result.first_stop.station_express_code)

            try:
                end_station = Station.get_by_code('express', result.last_stop.station_express_code)
                self.route_info.end_station = StationInfo.create_from_station(end_station)
            except Station.DoesNotExist:
                log.warning('Cannot get station by code %s', result.last_stop.station_express_code)

            if self.partner_data_history:
                self.current_partner_data.start_station_title = result.first_stop.station_name
                self.current_partner_data.end_station_title = result.last_stop.station_name
            self.save()

    use_new_partner_data = BooleanField(default=True)
    use_new_payments = BooleanField(default=True)
    use_booking_workflow = BooleanField(default=True)

    sync_personal_data_status = IntField(required=False, help_text='Статус миграции заказа в datasync')
    migrate_logs_status = IntField(required=False, help_text='Статус миграции логов')
    migrate_process_uid = StringField(required=False, help_text='Идентификатор процесса, мигрирующего логи')

    removed = BooleanField(default=False)
    removal_reason = StringField(null=True)

    @property
    def payments(self):
        return list(Payment.objects.filter(order_uid=self.uid).order_by('-trust_created_at'))

    @property
    def current_partner_data(self):
        return self.partner_data_history[-1]

    @property
    def current_partner_data_lookup_name(self):
        return 'partner_data_history__{}'.format(len(self.partner_data_history) - 1)

    def iter_lookup_partner_data(self):
        for i, pd in enumerate(self.partner_data_history):
            yield 'partner_data_history__{}'.format(i), pd

    @property
    def current_billing_payment(self):
        return Payment.objects.filter(order_uid=self.uid).order_by('-trust_created_at')[0]

    @property
    def purchase_tokens_history(self):
        """ История токенов, сначала самый новый """
        return [payment.purchase_token for payment in self.payments[1:]]

    def iter_ticket_to_lookup_name(self):
        for passenger_index, passenger in enumerate(self.passengers):
            for ticket_index, ticket in enumerate(passenger.tickets):
                yield ticket, 'passengers__{}__tickets__{}'.format(passenger_index, ticket_index)

    def iter_tickets(self):
        return (t for p in self.passengers for t in p.tickets)

    def iter_insurances(self):
        return [p.insurance for p in self.passengers if p.insurance and p.insurance.operation_id]

    class StationNotFound(Exception):
        pass

    @classmethod
    def fetch_stations(cls, orders, raise_on_station_not_found=True, fetch_start_and_end_stations=False,
                       fetch_settlements=False):
        if not orders:
            return orders

        station_ids = set()
        for order in orders:
            station_ids |= {order.station_from_id, order.station_to_id}
            if fetch_start_and_end_stations:
                if not (order.route_info.start_station and order.route_info.end_station):
                    order.try_update_route_info()
                if order.route_info.start_station:
                    station_ids.add(order.route_info.start_station.id)
                if order.route_info.end_station:
                    station_ids.add(order.route_info.end_station.id)
        station_query = Station.objects.filter(id__in=station_ids)
        if fetch_settlements:
            station_query = station_query.prefetch_related('settlement')
        station_by_ids = {s.id: s for s in station_query}

        def get_station(station_id):
            try:
                return station_by_ids[station_id]
            except KeyError:
                if raise_on_station_not_found:
                    raise cls.StationNotFound('No station for id {}'.format(station_id))
                else:
                    return None

        for order in orders:
            order.station_from = get_station(order.station_from_id)
            order.station_to = get_station(order.station_to_id)
            if fetch_start_and_end_stations and order.route_info.start_station:
                order.start_station = get_station(order.route_info.start_station.id)
            if fetch_start_and_end_stations and order.route_info.end_station:
                order.end_station = get_station(order.route_info.end_station.id)
        return orders

    def iter_refunds(self):
        for refund in TrainRefund.objects.filter(order_uid=self.uid).order_by('created_at'):
            yield refund

    def get_refund(self, refund_uuid):
        try:
            return next(r for r in self.iter_refunds() if r.uuid == refund_uuid)
        except StopIteration:
            raise self.RefundNotFound()

    @property
    def last_refund(self):
        try:
            return TrainRefund.objects.filter(order_uid=self.uid).order_by('-created_at')[0]
        except IndexError:
            return None

    @property
    def refunded_payments(self):
        return RefundPayment.objects.filter(order_uid=self.uid, refund_payment_status=RefundPaymentStatus.DONE)

    def get_fiscal_title(self):
        if not getattr(self, 'station_from', None):
            TrainOrder.fetch_stations([self])
        departure_local = UTC_TZ.localize(self.departure).astimezone(self.station_from.pytz)
        departure_string = '{0:%d} {1} {0:%Y} года в {0:%H:%M}'.format(departure_local, MONTHS[departure_local.month])
        res = 'Заказ {ticket_str} на поезд {train_number} из {st_from} {v} {st_to} {date}, вагон {coach_number}'\
            .format(ticket_str=('билетов' if len(list(self.iter_tickets())) > 1 else 'билета'),
                    train_number=self.train_ticket_number,
                    st_from=self.station_from.L_title(case='genitive'),
                    v=self.station_to.L_title(case='preposition_v_vo_na'),
                    st_to=self.station_to.L_title(case='accusative'),
                    date=departure_string,
                    coach_number=self.coach_number)
        return res

    def get_actual_travel_status(self):
        if self.process.get('state') == Process.EXCEPTION_STATE:
            return TravelOrderStatus.CANCELLED
        payment = self.current_billing_payment
        if payment and payment.process.get('state') == Process.EXCEPTION_STATE:
            return TravelOrderStatus.CANCELLED
        if self.status == OrderStatus.RESERVED:
            return TravelOrderStatus.RESERVED
        if self.status == OrderStatus.PAID:
            return TravelOrderStatus.IN_PROGRESS
        if self.status in [OrderStatus.CANCELLED, OrderStatus.START_PAYMENT_FAILED, OrderStatus.PAYMENT_FAILED,
                           OrderStatus.PAYMENT_OUTDATED, OrderStatus.CONFIRM_FAILED, OrderStatus.REBOOKING_FAILED]:
            return TravelOrderStatus.CANCELLED
        if self.status == OrderStatus.DONE:
            refund = self.last_refund
            if not refund:
                return TravelOrderStatus.DONE
            else:
                refund_status = (RefundStatus.FAILED if refund.process.get('state') == Process.EXCEPTION_STATE
                                 else refund.status)
                if refund_status in [RefundStatus.DONE, RefundStatus.FAILED]:
                    return TravelOrderStatus.DONE
                if refund_status in [RefundStatus.NEW, RefundStatus.PARTNER_REFUND_DONE,
                                     RefundStatus.PARTNER_REFUND_UNKNOWN]:
                    return TravelOrderStatus.IN_PROGRESS
        return TravelOrderStatus.UNKNOWN

    meta = {
        'indexes': [
            'uid',
            'process.external_events_count',
            'process.lock_modified',
            'process.lock_uid',
            'process.state',
            'status',
            'reserved_to',
            _make_case_insensitive_meta_index('passengers.first_name', name='passenger_first_name_with_collation'),
            _make_case_insensitive_meta_index('passengers.last_name', name='passenger_last_name_with_collation'),
            _make_case_insensitive_meta_index('passengers.patronymic', name='passenger_patronymic_with_collation'),
            _make_case_insensitive_meta_index('user_info.email', name='user_email_with_collation'),
            'user_info.reversed_phone',
            ('user_info.uid', 'travel_status', '-_id'),
            ('migrate_logs_status', 'migrate_process_uid'),
            'passengers.tickets.rzhd_status',
            'partner_data_history.im_order_id',
            'departure',
            'finished_at',
            'source.gclid',
            'partner_data_history.order_num',
        ],
        'index_background': True,
        'queryset_class': QuerySetWithCollation
    }

    def __str__(self):
        return 'TrainOrder(uid={})'.format(self.uid)

    def __repr__(self):
        return str(self)


class EmbeddedBackofficeUser(RepresentableEmbeddedDocument):
    user_id = IntField(required=True, min_value=1)
    username = StringField(required=True)


class BackofficeActionHistory(TrainPurchaseBaseDocument):
    uid = StringField(max_length=64, min_length=32, required=True)
    user = EmbeddedDocumentField(EmbeddedBackofficeUser, required=True)
    prev_state = DynamicField()
    action = StringField(required=True)
    details = DynamicField()
    completed = BooleanField(default=False)

    meta = {
        'indexes': [
            'uid',
        ],
        'index_background': True
    }


class ClientContract(RepresentableEmbeddedDocument):
    start_dt = DateTimeField()
    finish_dt = DateTimeField()
    is_active = BooleanField(default=False)
    is_cancelled = BooleanField(default=False)
    is_suspended = BooleanField(default=False)
    partner_commission_sum = DecimalField(precision=2)
    partner_commission_sum2 = DecimalField(precision=2)


class ClientContracts(TrainPurchaseBaseDocument):
    updated_at = DateTimeField()
    partner = StringEnumField(TrainPartner, required=True)
    contracts = ListField(EmbeddedDocumentField(ClientContract))

    meta = {
        'indexes': [
            'updated_at',
            'partner',
        ],
        'index_background': True
    }

    @classmethod
    def get_active_contract(cls, partner):
        last_contract = cls.objects.filter(partner=partner).order_by('-updated_at').first()
        if not last_contract:
            return None
        now = environment.now()
        for contract in last_contract.contracts:
            if contract.is_cancelled:
                continue
            if contract.is_suspended:
                continue
            if last_contract.updated_at.date() == now.date() and contract.is_active:
                return contract
            elif contract.start_dt <= now and (not contract.finish_dt or now <= contract.finish_dt):
                log.warning('Контракт c %s давно не обновлялся из биллинга, но он корректный', partner)
                return contract
        log.error('Нет активных договоров с %s!', partner)
        return None


class VerifyUserInfo(UserInfoBase):
    pass


class TrainPurchaseSmsVerification(TrainPurchaseBaseDocument):
    phone = StringField(required=True)
    sent_at = DateTimeField(required=True)
    used = BooleanField(default=False)
    code = StringField(required=True)
    message = StringField(required=True)
    action_name = StringField(help_text='Урезание частоты отправки и верификация идет с учетом action_name.')
    action_data = DictField(default={})
    yasms_id = StringField()

    meta = {
        'indexes': [
            ('phone', 'action_name', 'sent_at', 'used'),
        ],
        'index_background': True
    }

    @property
    def can_send_next_at(self):
        return self.sent_at + config.TRAIN_PURCHASE_VERIFICATION_SMS_THROTTLING_TIMEOUT

    @property
    def can_send_next_after(self):
        return (self.can_send_next_at - environment.now()).total_seconds()

    @property
    def can_send_next(self):
        return self.can_send_next_after <= 0

    @property
    def expired_at(self):
        return self.sent_at + config.TRAIN_PURCHASE_VERIFICATION_SMS_LIFE_TIME

    @property
    def expired_after(self):
        return (self.expired_at - environment.now()).total_seconds()

    @classmethod
    def get_latest_live_sms(cls, phone, action_name):
        expiration_dt = environment.now() - config.TRAIN_PURCHASE_VERIFICATION_SMS_LIFE_TIME
        return (cls.objects.filter(phone=phone, action_name=action_name, sent_at__gt=expiration_dt, used=False)
                .order_by('-sent_at').first())


class RefundPayment(TrainPurchaseBaseDocument):
    order_uid = StringField(max_length=64, min_length=32, required=True)
    purchase_token = StringField(verbose_name='Идентификатор корзины в TRUST')
    trust_refund_id = StringField(verbose_name='Идентификатор возврата в TRUST')
    refund_uuid = StringField(max_length=64, min_length=32, required=True)
    refund_payment_status = StringField(required=True, choices=RefundPaymentStatus.get_choices())
    refund_created_at = DateTimeField(required=True)
    is_email_sent = BooleanField(default=False)
    refund_blank_ids = ListField(StringField(min_length=1), default=[])
    refund_insurance_ids = ListField(StringField(min_length=1), default=[])
    is_retrying = BooleanField(default=False, verbose_name='Попытка возврата денег пользователю начата')
    refund_payment_finished_at = DateTimeField(help_text='Время завершения операции возврата денег пользователю')
    user_info = EmbeddedDocumentField(RefundUserInfo)
    trust_reversal_id = StringField(verbose_name='Идентификатор возврата в TRUST при частичном клиринге')
    payment_resized = BooleanField(default=False, verbose_name='Возврат через частичный клиринг')
    _refund_receipt_url = StringField(verbose_name='URL на чек возврата', db_field='refund_receipt_url')

    meta = {
        'indexes': [
            'is_email_sent',
            'refund_created_at',
            'refund_payment_status',
            'refund_uuid',
            'order_uid',
        ],
        'index_background': True,
        'collection': 'pending_refund_payment',
    }

    @property
    def payment(self):
        return Payment.objects.get(purchase_token=self.purchase_token)

    @property
    def train_refund(self):
        return TrainRefund.objects.get(uuid=self.refund_uuid)

    @property
    def refund_receipt_url(self):
        if not self._refund_receipt_url:
            try:
                trust = TrustClient()
                if self.payment_resized:
                    url = trust.get_receipt_clearing_url(self.purchase_token) if self.purchase_token else None
                else:
                    url = trust.get_refund_receipt_url(self.trust_refund_id) if self.trust_refund_id else None
                self._refund_receipt_url = url.replace('mode=mobile', 'mode=pdf') if url else None
            except Exception:
                log.exception('Error in getting refund_receipt_url')
            else:
                self.save()
        return self._refund_receipt_url


class RefundEmail(TrainPurchaseBaseDocument):
    """Коллекция для гарантированной отправки писем о возврате"""
    order_uid = StringField(max_length=64, min_length=32, required=True)
    refund_uuid = StringField(max_length=64, min_length=32, required=True)
    created_at = DateTimeField(required=True)
    is_sent = BooleanField(default=False)
    sent_at = DateTimeField()

    meta = {
        'indexes': [
            {'fields': ('order_uid', 'refund_uuid'), 'unique': True},
            ('is_sent', 'created_at'),
            'order_uid',
        ],
        'index_background': True
    }

    @classmethod
    def get_or_create_intent(cls, order_uid, refund_uuid):
        return cls.objects(order_uid=order_uid, refund_uuid=refund_uuid) \
            .modify(set_on_insert__created_at=environment.now_utc(), set_on_insert__is_sent=False,
                    upsert=True, new=True)

    @classmethod
    def mark_success(cls, order_uid, refund_uuid):
        intent = cls.get_or_create_intent(order_uid, refund_uuid)
        intent.modify(is_sent=True, sent_at=environment.now_utc())


class Payment(TrainPurchaseBaseDocument):
    uid = StringField(max_length=64, min_length=32, required=True, unique=True, default=gen_hex_uuid)
    order_uid = StringField(max_length=64, min_length=32, required=True)
    purchase_token = StringField(verbose_name='Идентификатор корзины в TRUST')
    payment_url = StringField(verbose_name='URL для iframe-а оплаты')
    _receipt_url = StringField(verbose_name='URL на чек оплаты', db_field='receipt_url')
    create_payment_counter = IntField(default=0)
    trust_created_at = DateTimeField(required=True, default=environment.now_utc)
    immediate_return = BooleanField(default=False, verbose_name='Используем продовый биллинг в тестинге')
    status = StringField()
    resp_code = StringField()
    resp_desc = StringField()
    process = DynamicField(default={})  # workflow data
    updated_from_billing_at = DateTimeField(help_text='Время обновления статусов и расшифровок платежей из биллинга')
    current_refund_payment_id = ObjectIdField(help_text='Последний возврат средств в обработке')
    use_deferred_clearing = BooleanField(default=True)
    hold_at = DateTimeField(verbose_name='Время холда платежа')
    clear_at = DateTimeField(verbose_name='Время когда был вызван клиринг платежа')

    meta = {
        'indexes': [
            'uid',
            'order_uid',
            'trust_created_at',
            'process.external_events_count',
            'process.lock_modified',
            'process.lock_uid',
            'process.state',
            'purchase_token',
        ],
        'index_background': True,
    }

    @property
    def order(self):
        return TrainOrder.objects.get(uid=self.order_uid)

    @property
    def current_refund_payment(self):
        return RefundPayment.objects.get(pk=self.current_refund_payment_id) if self.current_refund_payment_id else None

    @property
    def receipt_url(self):
        if not self._receipt_url and self.purchase_token:
            try:
                trust = TrustClient()
                url = trust.get_receipt_url(self.purchase_token)
                self._receipt_url = url.replace('mode=mobile', 'mode=pdf') if url else None
            except Exception:
                log.exception('Error in getting receipt_url')
            else:
                self.save()
        return self._receipt_url


class PartnerBalance(TrainPurchaseBaseDocument):
    partner = StringEnumField(TrainPartner, required=True)
    updated_at = DateTimeField(required=True, default=environment.now_utc)
    balance = DecimalField(required=True, precision=2, default=Decimal('0.00'))

    meta = {
        'indexes': [
            'partner',
        ],
        'index_background': True
    }

    @classmethod
    def get_partner_balance(cls, partner):
        try:
            return cls.objects.get(partner=partner)
        except cls.DoesNotExist:
            return cls.objects.create(partner=partner)
