# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import hashlib
import re

from django.conf import settings
from django.utils.encoding import force_bytes, force_text
from mongoengine.fields import StringField


def hash_string(value, salt):
    return hashlib.sha256(b''.join(map(force_bytes, (settings.SECRET_KEY, salt, value)))).hexdigest()


def hash_birth_date(birth_date):
    return hash_string(birth_date, 'passenger_birth_date')


def hash_doc_id(doc_id):
    return hash_string(doc_id, 'passenger_doc_id')


class RusEmailField(StringField):
    """
    Поле mongoengine для email с русскими буквами
    """

    EMAIL_REGEX = re.compile(
        # dot-atom
        r"(^[-!#$%&'*+/=?^_`{}|~0-9A-ZА-Яа-я]+(\.[-!#$%&'*+/=?^_`{}|~0-9A-ZА-Яа-я]+)*"
        # quoted-string
        r'|^"([\001-\010\013\014\016-\037!#-\[\]-\177]|\\[\001-011\013\014\016-\177])*"'
        # domain (max length of an ICAAN TLD is 22 characters)
        r')@(?:[A-Z0-9А-Яа-я](?:[A-Z0-9А-Яа-я-]{0,61}[A-ZА-Яа-я0-9])?\.)+'
        r'(?:[A-ZА-Яа-я]{2,6}|[A-ZА-Яа-я0-9-]{2,}(?<!-))$', re.IGNORECASE
    )

    def validate(self, value):
        if not RusEmailField.EMAIL_REGEX.match(value):
            self.error('Invalid email address: %s' % value)
        super(RusEmailField, self).validate(value)


def format_money(value):
    return force_text(value).replace('.00', '').replace('.', ',')


def format_places(place_list):
    return ', '.join(p.lstrip('0') for p in place_list)


def format_coach_number(coach_number):
    return coach_number.lstrip('0')
