# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import re
from collections import defaultdict
from decimal import Decimal

import six
from django.utils.encoding import force_text
from marshmallow import (
    Schema, ValidationError, fields, post_dump, post_load, pre_dump, validates_schema, validates,
)
from marshmallow_enum import EnumField
from pytz import utc

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.models.geo import Country, Station
from common.utils.date import UTC_TZ
from common.utils.railway import get_railway_tz_by_point
from common.utils.title_generator import DASH
from travel.rasp.train_api.serialization.experiment import ExperimentQuerySchema
from travel.rasp.train_api.serialization.fields import StationField, validate_station_express_code
from travel.rasp.train_api.serialization.schema_bases import MultiValueDictSchemaMixin
from travel.rasp.train_api.serialization.segment_station import SegmentStationSchema
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_partners.base.ticket_blank import BlankFormat
from travel.rasp.train_api.train_partners.base.train_details.serialization import TariffInfoSchema
from travel.rasp.train_api.train_purchase.core.enums import (
    AgeGroup, LoyaltyCardType, PlacesOption, PlacesType, OrderStatus, TravelOrderStatus,
    Gender, GenderChoice, DocumentType, TrainPartner, TrainPurchaseSource, Arrangement, TrainPartnerCredentialId,
    RoutePolicy
)
from travel.rasp.train_api.train_purchase.core.models import RefundStatus, TrainOrder
from travel.rasp.train_api.train_purchase.utils.electronic_registration import RegistrationStatus
from travel.rasp.train_api.train_purchase.workflow.payment.create_payment import (
    PAYMENT_MARGIN_BEFORE_TICKET_RESERVATION_ENDS,
)

log = logging.getLogger(__name__)
EMAIL_RE = re.compile(r'(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)')
PHONE_RE = re.compile(r'\+[0-9]{4,}')
IM_CREDENTIALS_BY_UTM_SOURCE = defaultdict(
    lambda: TrainPartnerCredentialId.IM,
    suburbans=TrainPartnerCredentialId.IM_SUBURBAN,
)


def validate_phone(value):
    if not PHONE_RE.match(value):
        raise ValidationError('Phone number {} does not match format +<code><number>')


def validate_email(value):
    # допускаются только латинские символы
    if not EMAIL_RE.match(value):
        raise ValidationError('Email {} is not valid'.format(value))


def validate_express_station_id(station_id):
    try:
        station = Station.objects.get(pk=station_id)
    except Station.DoesNotExist:
        raise ValidationError('Not existed station {}'.format(station_id))

    if not station.get_code('express'):
        raise ValidationError('Station {} has no express code'.format(station_id))


class BasePassengerSchema(Schema):
    first_name = fields.String(dump_to='firstName', load_from='firstName', required=True)
    last_name = fields.String(dump_to='lastName', load_from='lastName', required=True)
    patronymic = fields.String(missing=None)
    sex = EnumField(Gender, by_value=True, required=True)
    doc_type = EnumField(DocumentType, by_value=True, dump_to='docType', load_from='docType', required=True)
    doc_id = fields.String(dump_to='docId', load_from='docId', required=True)
    birth_date = fields.Date(dump_to='birthDate', load_from='birthDate', required=True)
    phone = fields.String(missing=None)
    email = fields.String(missing=None)


PLACES_TYPE_TO_DESCRIPTION = {  # TODO перенести на фронт
    PlacesType.LOWER_TIER: 'Нижний ярус',
    PlacesType.UPPER_TIER: 'Верхний ярус',
    PlacesType.MIDDLE_TIER: 'Средний ярус',
    PlacesType.FOR_PASSENGER_WITH_PET: 'Для пассажира с животным',
    PlacesType.FOR_MOTHER_WITH_CHILD: 'Для матери и ребенка',
    PlacesType.FOR_PASSENGER_WITH_CHILDREN: 'Для пассажира с детьми',
    PlacesType.FOR_IMPAIRED_PASSENGER: 'Для инвалидов',
    PlacesType.CONFERENCE_ROOM: 'Переговорная',
    PlacesType.NOT_NEAR_TABLE: 'Не у стола',
    PlacesType.NEAR_TABLE: 'У стола',
    PlacesType.NEAR_PLAYGROUND: 'У детской площадки',
    PlacesType.NEAR_PLAYGROUND_TABLE: 'У стола рядом с детской площадкой',
    PlacesType.NEAR_PASSENGERS_WITH_PETS: 'Рядом с местами для пассажиров с животными',
    PlacesType.FOLDING: 'Откидное',
    PlacesType.LASTOCHKA_COMPARTMENT: 'Отсек (купе) в поезде Ласточка'
}


class PlacesTypeSchema(Schema):
    code = fields.String(attribute='value')
    description = fields.Function(lambda places_type: PLACES_TYPE_TO_DESCRIPTION[places_type])


class PaymentSchema(Schema):
    amount = fields.Decimal()
    fee = fields.Decimal()
    bedding_amount = fields.Decimal(attribute='service_amount', dump_to='beddingAmount')
    bedding_fee = fields.Decimal(attribute='service_fee', dump_to='beddingFee')


class RefundSchema(Schema):
    amount = fields.Method('_calculate_amount_with_yandex_fee')

    @classmethod
    def _calculate_amount_with_yandex_fee(cls, refund):
        refund_amount = refund.amount
        if refund.refund_yandex_fee_amount:
            refund_amount += refund.refund_yandex_fee_amount
        return refund_amount


class UtmStringField(fields.String):
    def _deserialize(self, value, attr, data):
        if not isinstance(value, six.string_types):
            try:
                value = value[0]
            except (TypeError, IndexError):
                value = json.dumps(value, ensure_ascii=False)
        return super(UtmStringField, self)._deserialize(value, attr, data)


class SourceSchema(Schema):
    req_id = UtmStringField(load_from='reqId', dump_to='reqId')
    device = fields.Function(deserialize=lambda d: try_parse_enum(TrainPurchaseSource, d),
                             serialize=lambda s: s.device.value if s.device else None)
    utm_source = UtmStringField(load_from='utmSource', dump_to='utmSource')
    utm_medium = UtmStringField(load_from='utmMedium', dump_to='utmMedium')
    utm_campaign = UtmStringField(load_from='utmCampaign', dump_to='utmCampaign')
    utm_term = UtmStringField(load_from='utmTerm', dump_to='utmTerm')
    utm_content = UtmStringField(load_from='utmContent', dump_to='utmContent')
    from_ = UtmStringField(load_from='from', dump_to='from')
    gclid = UtmStringField()
    terminal = UtmStringField(missing=None)
    is_transfer = fields.Boolean(load_from='isTransfer', dump_to='isTransfer', missing=None)
    partner = UtmStringField(missing=None)
    subpartner = UtmStringField(missing=None)
    partner_uid = UtmStringField(load_from='partnerUid', dump_to='partnerUid', missing=None)
    test_id = UtmStringField(load_from='testId', dump_to='testId', missing=None)
    test_buckets = UtmStringField(load_from='testBuckets', dump_to='testBuckets', missing=None)
    icookie = UtmStringField(missing=None)
    serp_uuid = UtmStringField(load_from='serpUuid', dump_to='serpUuid', missing=None)

    class Meta:
        strict = True

    def load(self, *args, **kwargs):
        try:
            return super(SourceSchema, self).load(*args, **kwargs)
        except ValidationError as e:
            log.exception('Invalid Source. Input raw data: "%s".', e.data)
            return {}, {}


class TicketSchema(Schema):
    blank_id = fields.String(dump_to='blankId')
    amount = fields.Decimal(attribute='payment.total')  # TODO: удалить когда фронт начнёт использовать payment
    places = fields.List(fields.String())
    places_type = fields.Nested(PlacesTypeSchema, dump_to='placesType')
    rzhd_status = fields.Method('_rzhd_status', dump_to='rzhdStatus')
    pending = fields.Boolean()
    is_refundable = fields.Method('_is_refundable', dump_to='isRefundable')
    is_refunding = fields.Method('_is_refunding', dump_to='isRefunding')
    is_external_refund = fields.Method('_is_external_refund', dump_to='isExternalRefund')
    is_money_returned = fields.Method('_is_money_returned', dump_to='isMoneyReturned')
    payment = fields.Nested(PaymentSchema)
    refund = fields.Nested(RefundSchema)
    tariff_info = fields.Nested(TariffInfoSchema, dump_to='tariffInfo')
    raw_tariff_title = fields.String(dump_to='rawTariffTitle')

    def _rzhd_status(self, ticket):
        rzhd_status = ticket.rzhd_status
        if rzhd_status is None:
            return None
        return RzhdStatus(rzhd_status).name

    def _is_refunding(self, ticket):
        return ticket.blank_id in self.context.get('refunding_blank_ids', ())

    def _is_external_refund(self, ticket):
        return ticket.blank_id in self.context.get('external_refund_blank_ids', ())

    def _is_refundable(self, ticket):
        if self._is_refunding(ticket):
            return False

        rzhd_status = ticket.rzhd_status
        return rzhd_status is not None and RzhdStatus(rzhd_status).is_refundable()

    def _is_money_returned(self, ticket):
        return ticket.blank_id in self.context.get('refunded_payments_blank_ids', ())


class InsuranceSchema(Schema):
    amount = fields.Decimal()
    compensation = fields.Decimal()
    ordered = fields.Function(lambda insurance: True if insurance.trust_order_id else False)
    refunded = fields.Function(lambda insurance: True if insurance.refund_uuid else False)
    operation_id = fields.String(dump_to='operationId')


class InsuranceCheckoutSchema(Schema):
    enabled = fields.Boolean(missing=False)


class RunBookingSchema(Schema):
    with_insurance = fields.Boolean(missing=False)


class PassengerSchema(BasePassengerSchema):
    age = fields.Decimal()
    citizenship = fields.String(attribute='citizenship_country.code3')
    tickets = fields.Nested(TicketSchema, many=True)
    customer_id = fields.String(dump_to='customerId')
    insurance = fields.Nested(InsuranceSchema)


class UserInfoBaseSchema(Schema):
    ip = fields.String(required=True)
    uid = fields.String(allow_none=True)
    region_id = fields.Integer(required=True, dump_to='regionId', load_from='regionId')


class UserInfoSchema(UserInfoBaseSchema):
    email = fields.String(required=True, validate=validate_email)
    is_mobile = fields.Boolean(default=False, missing=False, dump_to='isMobile', load_from='isMobile')
    # FIXME: Телефон должен быть обязательным полем
    phone = fields.String(required=False, validate=validate_phone)
    yandex_uid = fields.String(missing=None)


class UserInfoRefundSchema(UserInfoBaseSchema):
    pass


class BaseOrderSchema(Schema):
    partner = EnumField(TrainPartner, by_value=True, required=True)
    train_number = fields.String(dump_to='trainNumber', load_from='trainNumber', required=True)
    train_ticket_number = fields.String(dump_to='trainTicketNumber', load_from='trainTicketNumber', required=True)
    train_name = fields.String(dump_to='trainName', load_from='trainName', missing=None)
    departure = fields.DateTime(required=True)
    arrival = fields.DateTime(required=True)
    coach_type = EnumField(CoachType, by_value=True, dump_to='carType', load_from='carType', required=True)
    gender = EnumField(GenderChoice, by_value=True, missing=None)
    user_info = fields.Nested(UserInfoSchema, required=True, load_from='userInfo', dump_to='userInfo')
    two_storey = fields.Boolean(default=False, load_from='twoStorey', dump_to='twoStorey')


class PlaceSchema(Schema):
    number = fields.Integer()
    is_upper = fields.Boolean(dump_to='isUpper', load_from='isUpper')


class LoyaltyCardSchema(Schema):
    type = EnumField(LoyaltyCardType, by_value=True, required=True)
    number = fields.String(required=True)


class ReserveTicketsPassengerSchema(BasePassengerSchema):
    age_group = EnumField(AgeGroup, by_value=True, load_from='ageGroup', missing=AgeGroup.ADULTS)
    tariff_info = fields.Function(
        deserialize=lambda tariff: TariffInfo.objects.get(code=tariff),
        load_from='tariff',
        required=True
    )
    citizenship_code = fields.String(load_from='citizenship')
    citizenship_geo_id = fields.Integer(load_from='citizenshipGeoId')

    loyalty_card = fields.Nested(LoyaltyCardSchema, load_from='loyaltyCard', missing=None)
    loyalty_cards = fields.Nested(LoyaltyCardSchema, load_from='loyaltyCards', missing=[], many=True)

    @post_load
    def _citizenship_country(self, data):
        if data.get('citizenship_geo_id'):
            data['citizenship_country'] = Country.objects.get(_geo_id=data['citizenship_geo_id'])
        elif data.get('citizenship_code'):
            data['citizenship_country'] = Country.objects.get(code3=data['citizenship_code'])

    @post_load
    def _ensure_loyalty_cards(self, data):
        # TODO: необходимо оторвать эту логику как только фронтенд поддержит loyalty_cards
        loyalty_cards = data['loyalty_cards']
        if not loyalty_cards and data['loyalty_card']:
            loyalty_cards = [data['loyalty_card']]

        data['loyalty_cards'] = loyalty_cards
        return data

    @validates_schema(skip_on_field_errors=False)
    def _validate_loyalty_cards(self, data):
        card_types = {card['type'] for card in data['loyalty_cards']}
        if len(card_types) != len(data['loyalty_cards']):
            raise ValidationError('loyalty_cards should has different types', ['loyalty_cards'])


class PlaceCountSchema(Schema):
    bottom = fields.Integer(missing=None)
    upper = fields.Integer(missing=None)
    near_window = fields.Integer(load_from='nearWindow', missing=None)
    near_passage = fields.Integer(load_from='nearPassage', missing=None)


class PlaceRequirementsSchema(Schema):
    arrangement = EnumField(Arrangement, by_value=True, missing=None)
    storey = fields.Integer(missing=0)
    count = fields.Nested(PlaceCountSchema, many=False, missing=None)


class ReserveTicketsSchema(BaseOrderSchema):
    passengers = fields.Nested(ReserveTicketsPassengerSchema, many=True, required=True)
    station_from = StationField(required=True, load_from='stationFromId', validate=validate_station_express_code)
    station_to = StationField(required=True, load_from='stationToId', validate=validate_station_express_code)
    electronic_registration = fields.Boolean(load_from='electronicRegistration', missing=False)
    coach_number = fields.String(load_from='carNumber', missing=None)
    coach_owner = fields.String(load_from='carOwner', missing=None)
    service_class = fields.String(load_from='serviceClass', required=True)
    international_service_class = fields.String(load_from='internationalServiceClass', missing=None)
    is_cppk = fields.Boolean(load_from='isCppk', missing=None)

    places = fields.Nested(PlaceSchema, many=True, required=True)
    place_demands = EnumField(PlacesOption, by_value=True, load_from='placeDemands', missing=None)
    requirements = fields.Nested(PlaceRequirementsSchema, many=False, missing=None)
    additional_place_requirements = fields.String(load_from='additionalPlaceRequirements', missing=None)

    bedding = fields.Boolean(missing=False)
    bedding_tariff = fields.Decimal(load_from='beddingTariff', missing=None)
    order_history = fields.List(fields.Dict(), load_from='orderHistory', missing=None)
    source = fields.Nested(SourceSchema, many=False)
    price_exp_id = fields.String(load_from='priceExpId', missing=None)

    give_child_without_place = fields.Boolean(load_from='giveChildWithoutPlace', missing=False)
    enable_rebooking = fields.Boolean(load_from='enableRebooking', missing=False)
    scheme_id = fields.Integer(load_from='schemeId', missing=None)
    route_policy = EnumField(RoutePolicy, by_value=True, load_from='routePolicy', missing=None)

    @validates_schema(skip_on_field_errors=False)
    def _validate_partner(self, data):
        if 'partner' in data and not data['partner'].enabled:
            raise ValidationError('partner is disabled', ['partner'])

    @post_load
    def utc_departure(self, data):
        data['departure'] = data['departure'].astimezone(UTC_TZ)
        return data

    @post_load
    def _get_partner_credential_id(self, data):
        if data['partner'] == TrainPartner.IM:
            utm_source = data.get('source', {}).get('utm_source')
            data['partner_credential_id'] = IM_CREDENTIALS_BY_UTM_SOURCE[utm_source]
        else:
            raise NotImplementedError('Non-Implemented partner: {}'.format(data['partner']))
        return data


class FindOrderSchema(Schema):
    email = fields.Email(missing=None)
    phone = fields.String(missing=None)
    order_num = fields.String(required=True)


class ErrorInfoSchema(Schema):
    type = fields.String(required=True)
    message = fields.String()
    data = fields.Dict()


class OrderBaseDumpSchema(BaseOrderSchema):
    status = EnumField(OrderStatus, by_value=True)
    uid = fields.String()
    error = fields.Nested(ErrorInfoSchema)
    coach_owner = fields.String(dump_to='coachOwner')
    displayed_coach_owner = fields.Function(lambda order: order.displayed_coach_owner or order.coach_owner,
                                            dump_to='displayedCoachOwner')
    passengers = fields.Nested(PassengerSchema, many=True)
    order_number = fields.String(attribute='current_partner_data.order_num', dump_to='orderNumber')
    tickets_status_freezes_at = fields.DateTime(attribute='current_partner_data.expire_set_er',
                                                dump_to='ticketsStatusFreezesAt')
    payment_url = fields.String(attribute='current_billing_payment.payment_url', dump_to='paymentUrl')
    reserved_to = fields.Function(
        serialize=lambda order: (order.reserved_to -
                                 PAYMENT_MARGIN_BEFORE_TICKET_RESERVATION_ENDS).replace(tzinfo=UTC_TZ).isoformat(),
        dump_to='reservedTo'
    )
    is_reservation_prolonged = fields.Boolean(attribute='current_partner_data.is_reservation_prolonged',
                                              dump_to='isReservationProlonged')
    refund_status = fields.String(attribute='last_refund.status', dump_to='refundStatus')
    special_notice = fields.String(attribute='current_partner_data.special_notice', dump_to='specialNotice')
    time_notice = fields.String(attribute='current_partner_data.time_notice', dump_to='timeNotice')
    compartment_gender = EnumField(GenderChoice, by_value=True,
                                   attribute='current_partner_data.compartment_gender', dump_to='compartmentGender')
    payment_code = fields.String(attribute='current_billing_payment.resp_code', dump_to='paymentCode')
    payment_status = fields.String(attribute='current_billing_payment.status', dump_to='paymentStatus')
    max_pending_till = fields.DateTime(dump_to='maxPendingTill')
    is_suburban = fields.Boolean(attribute='current_partner_data.is_suburban', dump_to='isSuburban')
    rebooking_available = fields.Function(lambda o: o.rebooking_info is not None, dump_to='rebookingAvailable')
    coach_number = fields.String(dump_to='carNumber', required=True)
    is_only_full_return_possible = fields.Boolean(attribute='current_partner_data.is_only_full_return_possible',
                                                  dump_to='isOnlyFullReturnPossible')


class RefundPaymentSchema(Schema):
    refund_payment_status = fields.String(dump_to='refundPaymentStatus')
    refund_created_at = fields.DateTime(dump_to='refundCreatedAt')
    refund_blank_ids = fields.List(fields.String, dump_to='blankIds')
    refund_insurance_ids = fields.List(fields.String, dump_to='insuranceIds')
    payment_refund_receipt_url = fields.Method('_dump_payment_refund_receipt_url', dump_to='paymentRefundReceiptUrl')

    @classmethod
    def _dump_payment_refund_receipt_url(cls, obj):
        return obj.refund_receipt_url


class OrderBulkSchema(OrderBaseDumpSchema):
    station_from = fields.Method('_dump_station_from', dump_to='stationFrom')
    station_to = fields.Method('_dump_station_to', dump_to='stationTo')
    start_station = fields.Method('_dump_start_station', dump_to='startStation')
    end_station = fields.Method('_dump_end_station', dump_to='endStation')
    insurance_enabled = fields.Boolean(default=False, dump_to='insuranceEnabled')
    insurance_auto_return = fields.Function(lambda o: bool(o.insurance_auto_return_uuid), dump_to='insuranceAutoReturn')
    insurance_status = fields.Function(
        lambda o: o.insurance.status.value if o.insurance and o.insurance.status else None,
        dump_to='insuranceStatus',
    )
    travel_status = EnumField(TravelOrderStatus, by_value=True, dump_to='travelStatus')
    brand_title = fields.String(attribute='train_name', dump_to='brandTitle')
    train_title = fields.Method('_dump_train_title', dump_to='trainTitle')
    warnings = fields.List(fields.Dict())

    def _dump_station_from(self, obj):
        """
        :type obj: common.apps.train_order.models.TrainOrder
        """
        ufs_title = obj.current_partner_data.station_from_title if obj.current_partner_data else None
        return self._dump_station(obj.station_from, ufs_title)

    def _dump_station_to(self, obj):
        """
        :type obj: common.apps.train_order.models.TrainOrder
        """
        ufs_title = obj.current_partner_data.station_to_title if obj.current_partner_data else None
        return self._dump_station(obj.station_to, ufs_title)

    def _dump_start_station(self, obj):
        """
        :type obj: common.apps.train_order.models.TrainOrder
        """
        station = getattr(obj, 'start_station', None)
        if not station:
            return None
        ufs_title = obj.current_partner_data.start_station_title if obj.current_partner_data else None
        return self._dump_station(station, ufs_title)

    def _dump_end_station(self, obj):
        """
        :type obj: common.apps.train_order.models.TrainOrder
        """
        station = getattr(obj, 'end_station', None)
        if not station:
            return None
        ufs_title = obj.current_partner_data.end_station_title if obj.current_partner_data else None
        return self._dump_station(station, ufs_title)

    def _dump_station(self, station, ufs_title):
        station_data = SegmentStationSchema(context=self.context).dump(station).data
        station_data['ufsTitle'] = ufs_title
        station_data['settlementGeoId'] = station.get_settlement_geo_id()
        station_data['settlementTitle'] = station.settlement.L_title() if station.settlement_id else None

        railway_timezone = get_railway_tz_by_point(station)
        if railway_timezone:
            station_data['railwayTimezone'] = railway_timezone.zone

        return station_data

    def _dump_train_title(self, obj):
        start_station = getattr(obj, 'start_station', None)
        end_station = getattr(obj, 'end_station', None)
        if (
            not start_station
            or not start_station.settlement
            or not end_station
            or not end_station.settlement
        ):
            return None
        return '{} {} {}'.format(start_station.settlement.L_title(), DASH, end_station.settlement.L_title())

    @pre_dump
    def _fetch_stations(self, obj):
        if (not getattr(obj, 'station_from', None) or not getattr(obj, 'station_to', None)
                or not getattr(obj, 'start_station', None) or not getattr(obj, 'end_station', None)):
            TrainOrder.fetch_stations([obj], fetch_start_and_end_stations=True, fetch_settlements=True)

    @pre_dump
    def _save_context(self, obj):
        external_refund_blank_ids = []
        last_refund = None
        for refund in obj.iter_refunds():
            if refund.is_external:
                external_refund_blank_ids.extend(refund.blank_ids)
            last_refund = refund
        if last_refund and last_refund.status not in (RefundStatus.DONE, RefundStatus.FAILED):
            self.context['refunding_blank_ids'] = last_refund.blank_ids
        self.context['external_refund_blank_ids'] = external_refund_blank_ids
        refunded_payments_blank_ids = set()
        for refund_payment in obj.refunded_payments:
            refunded_payments_blank_ids.update(refund_payment.refund_blank_ids)
        self.context['refunded_payments_blank_ids'] = refunded_payments_blank_ids

    @post_dump
    def _clean_context(self, _data):
        self.context.pop('refunding_blank_ids', None)
        self.context.pop('external_refund_blank_ids', None)
        self.context.pop('refunded_payments_blank_ids', None)


class OrderSchema(OrderBulkSchema):
    payment_receipt_url = fields.Method('_dump_payment_receipt_url', dump_to='paymentReceiptUrl')
    refunded_payments = fields.Nested(RefundPaymentSchema, many=True, dump_to='refundPayments')

    @classmethod
    def _dump_payment_receipt_url(cls, obj):
        payment = obj.current_billing_payment
        return payment.receipt_url if payment else None


class DownloadQuerySchema(Schema):
    ticket_format = EnumField(BlankFormat, missing=BlankFormat.PDF, by_value=True, load_from='ticketFormat')
    blank_id = fields.String(load_from='blankId')


class CalculateRefundAmountSchema(Schema):
    blank_ids = fields.List(fields.String, required=True, load_from='blankIds')

    @post_load
    def _ensure_unique_blank_ids(self, data):
        data['blank_ids'] = list(set(data['blank_ids']))
        return data

    @validates('blank_ids')
    def validate_length(self, blank_ids):
        if len(blank_ids) < 1:
            raise ValidationError('Must be minimum one blank.')


class RetrieveRequestSchema(Schema):
    first_actual_warning_only = fields.Boolean(load_from='firstActualWarningOnly', missing=False)


class UpdateTicketsStatusRequestSchema(Schema):
    first_actual_warning_only = fields.Boolean(load_from='firstActualWarningOnly', missing=False)


class RefundRequestSchema(Schema):
    request_sms_verification = fields.Boolean(default=False, load_from='requestSMSVerification')
    sms_verification_code = fields.String(missing=None, load_from='SMSVerificationCode')
    blank_ids = fields.List(fields.String, required=True, load_from='blankIds')
    user_info = fields.Nested(UserInfoRefundSchema, required=True, load_from='userInfo')


class ChangeRegistrationSchema(Schema, MultiValueDictSchemaMixin):
    request_sms_verification = fields.Boolean(default=False, load_from='requestSMSVerification')
    sms_verification_code = fields.String(missing=None, load_from='SMSVerificationCode')
    blank_ids = fields.List(fields.String, required=True, load_from='blankId')
    new_status = EnumField(RegistrationStatus, required=True, load_from='newStatus')


def try_parse_enum(enum, value):
    try:
        return enum(value)
    except ValueError:
        return None


class ActivePartnersResponseSchema(Schema):
    partner_codes = fields.List(fields.String(required=True), dump_to='partnerCodes')


COACH_TYPE_TO_RUS = {
    CoachType.PLATZKARTE: 'Плацкарт',
    CoachType.COMPARTMENT: 'Купе',
    CoachType.SUITE: 'Люкс',
    CoachType.SITTING: 'Сидячий',
    CoachType.COMMON: 'Общий',
    CoachType.SOFT: 'Мягкий'
}

TRAVEL_ORDER_STATUS_TO_RUS = {
    TravelOrderStatus.RESERVED: 'Зарезервирован',
    TravelOrderStatus.IN_PROGRESS: 'В обработке',
    TravelOrderStatus.DONE: 'Выполнен',
    TravelOrderStatus.CANCELLED: 'Отменен',
    TravelOrderStatus.UNKNOWN: 'Неизвестен',
}

RZHD_STATUS_TO_RUS = {
    0: 'Без электронной регистрации',
    1: 'Электронная регистрация',
    2: 'Оплата не подтверждена',
    3: 'Аннулирован',
    4: 'Возвращен',
    5: 'Возвращены места',
    6: 'Выдан посадочный купон на бланке строгой отчетности',
    7: 'Отложенная оплата',
    8: 'Выполнено прерывание поездки',
    9: 'Выполнено прерывание с возобновлением поездки',
}


class TakeoutPassengerDetailsSchema(Schema):
    fio = fields.String(attribute='fio')
    places = fields.Function(lambda p: ', '.join(pl for t in p.tickets for pl in t.places))
    tariffs = fields.Function(lambda p: ', '.join(t.raw_tariff_title for t in p.tickets if t.raw_tariff_title))
    amount = fields.Function(lambda p: force_text(p.total_amount))
    fee = fields.Function(lambda p: force_text(p.total_tickets_fee))
    insurance_amount = fields.Function(lambda p: force_text(p.total_insurance), dump_to='insuranceAmount')
    ticket_status = fields.Function(
        lambda p: ', '.join(RZHD_STATUS_TO_RUS[t.rzhd_status] for t in p.tickets if t.rzhd_status is not None),
        dump_to='ticketStatus',
    )


class TakeoutOrderDetailsSchema(Schema):
    order_num = fields.String(attribute='current_partner_data.order_num', dump_to='orderNum')
    order_status = fields.Function(lambda o: TRAVEL_ORDER_STATUS_TO_RUS[o.travel_status], dump_to='orderStatus')
    train_number = fields.String(dump_to='trainNumber')
    departure_station = fields.Function(lambda o: o.station_from.L_title(), dump_to='departureStation')
    arrival_station = fields.Function(lambda o: o.station_to.L_title(), dump_to='arrivalStation')
    departure_datetime = fields.Function(
        lambda o: TakeoutOrderDetailsSchema.local_dt_to_str(o.departure, o.station_from.pytz),
        dump_to='departureDatetime',
    )
    arrival_datetime = fields.Function(
        lambda o: TakeoutOrderDetailsSchema.local_dt_to_str(o.arrival, o.station_to.pytz),
        dump_to='arrivalDatetime',
    )
    travel_time = fields.Function(
        lambda o: TakeoutOrderDetailsSchema.timedelta_to_str(o.arrival - o.departure),
        dump_to='travelTime'
    )
    coach_type = fields.Function(lambda o: COACH_TYPE_TO_RUS[o.coach_type], dump_to='coachType')
    coach_number = fields.String(dump_to='coachNumber')
    total = fields.Function(lambda o: force_text(sum(p.total for p in o.passengers)))
    total_fee = fields.Function(lambda o: force_text(sum(p.total_tickets_fee for p in o.passengers)), dump_to='totalFee')
    total_refund = fields.Method('_calculate_refund', dump_to='totalRefund')
    passengers = fields.Nested(TakeoutPassengerDetailsSchema, many=True)

    @pre_dump(pass_many=True)
    def _fetch_stations(self, obj, many):
        TrainOrder.fetch_stations(obj if many else [obj])

    @classmethod
    def timedelta_to_str(cls, delta):
        hours = int(delta.seconds / 3600)
        minutes = int((delta.seconds % 3600) / 60)
        return '{}.{:02d}:{:02d}'.format(delta.days, hours, minutes) if delta.days\
            else '{:02d}:{:02d}'.format(hours, minutes)

    @classmethod
    def local_dt_to_str(cls, dt_in_utc, tz):
        return utc.localize(dt_in_utc).astimezone(tz).strftime('%Y-%m-%d %H:%M')

    @classmethod
    def _calculate_refund(cls, order):
        def _is_refunded(ticket):
            return ticket.refund and ticket.refund.amount and ticket.rzhd_status == RzhdStatus.REFUNDED

        refund = Decimal('0')
        for passenger in order.passengers:
            if passenger.insurance and passenger.insurance.refund_uuid:
                refund += passenger.insurance.amount
            refund += sum(
                t.refund.amount for t in passenger.tickets if _is_refunded(t)
            )
        return force_text(refund)


class UserOrdersRequestSchema(Schema):
    get_count_by_statuses = fields.Boolean(load_from='getCountByStatuses', missing=False)
    find_string = fields.String(load_from='findString', missing=None)
    limit = fields.Integer(missing=None)
    offset = fields.Integer(missing=None)
    travel_statuses = fields.Method(deserialize='_parse_statuses', load_from='travelStatus', missing='')

    @classmethod
    def _parse_statuses(cls, obj):
        statuses = [TravelOrderStatus(s) for s in obj.split(',')] if obj else [TravelOrderStatus.DONE]
        return statuses


class LogBanditSchema(ExperimentQuerySchema):
    departure = fields.DateTime()
    arrival = fields.DateTime()
    station_from = StationField(load_from='stationFromId')
    station_to = StationField(load_from='stationToId')
    train_type = fields.String(load_from='trainType', missing=None)
    car_type = fields.String(load_from='carType')
    service_class = fields.String(load_from='serviceClass')
    price = fields.Decimal()
    event_type = fields.String(load_from='eventType', missing='passenger-details')
    fee_calculation_token = fields.String(load_from='feeCalculationToken', missing=None)

    @post_load
    def _utc(self, data):
        data['departure'] = data['departure'].astimezone(UTC_TZ)
        data['arrival'] = data['arrival'].astimezone(UTC_TZ)
        return data
