# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from django.conf import settings
from django.utils.encoding import force_text
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ, UTC_TZ
from travel.rasp.train_api.train_partners.im.base import get_im_response, IM_DATETIME_FORMAT, parse_datetime
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, RefundStatus
from travel.rasp.train_api.train_purchase.utils.office_refund import (
    create_external_train_refund_and_run_it, ExternalRefundError, ExternalRefundCriticalError
)
from travel.rasp.train_api.train_purchase.utils.order import should_refund_yandex_fee

log = logging.getLogger(__name__)

MONTH_FORMAT = '%Y-%m'
ORDERS_ENDPOINT = 'Order/V1/Info/OrderList'
CHECK_DAYS_IN_PAST = 7

define_setting('OFFICE_REFUND_ERRORS_CAMPAIGN', {
    Configuration.PRODUCTION: 'OSXA9ZT2-4KY1',
    Configuration.TESTING: 'GUA03ZT2-WAB',
    Configuration.DEVELOPMENT: 'GUA03ZT2-WAB'
}, default=None)


class TooEarlyError(Exception):
    pass


@single_launch_task()
def check_im_office_refunds():
    if conf.TRAIN_PURCHASE_DISABLE_OFFICE_REFUND_PROCESSING:
        log.info('Обработка кассовых возвратов отключена')
        return

    today = environment.today()
    for day in (today - timedelta(i) for i in range(1, CHECK_DAYS_IN_PAST + 1)):
        log.info('Checking for IM office refunds %s', day)
        check_for_external_refunds(day)


def check_for_external_refunds(day):
    _check_day_is_save(day)

    orders_data = get_im_response(ORDERS_ENDPOINT, {
        'Date': day.strftime(IM_DATETIME_FORMAT),
        'OperationType': 'Return',
        'IsExternallyLoaded': True
    })
    for order_data in orders_data['Orders']:
        im_order_id = order_data['OrderId']
        try:
            order = TrainOrder.objects.get(partner_data_history__im_order_id=im_order_id)
        except TrainOrder.DoesNotExist:
            log.error('Не нашли заказа по im_order_id=%s', im_order_id)
            continue

        with log_context(order_uid=order.uid):
            log.info('Проверяем вернули ли деньги за возвраты в кассе')
            for order_item in order_data['OrderItems']:
                partner_refunded_blanks = [force_text(blank_data['PreviousOrderItemBlankId']) for blank_data
                                           in order_item['OrderItemBlanks']]
                if not _is_blanks_refunded(order, partner_refunded_blanks):
                    log.info('По заказу деньги не вернули по внешней операции %s', order_item['OrderItemId'])
                    refund_confirmed_at = parse_datetime(order_item['ConfirmDateTime'])
                    refund_confirmed_at_utc = refund_confirmed_at.astimezone(UTC_TZ).replace(tzinfo=None)
                    refund_yandex_fee = should_refund_yandex_fee(order, refund_confirmed_at_utc)
                    _create_external_train_refund_and_run_it(order, partner_refunded_blanks, refund_yandex_fee)
                    break


def _create_external_train_refund_and_run_it(order, blank_ids, refund_yandex_fee):
    try:
        create_external_train_refund_and_run_it(order, refund_yandex_fee)
    except ExternalRefundCriticalError as error:
        log.exception('Критическая ошибка при обработке кассового возврата')
        send_office_refund_error(uid=order.uid, error=error, blank_ids=blank_ids)
    except ExternalRefundError as error:
        log.exception('Ошибка при обработке кассового возврата')
        send_office_refund_error(uid=order.uid, error=error, blank_ids=blank_ids)


def send_office_refund_error(uid, error, blank_ids):
    email = conf.TRAIN_PURCHASE_ERRORS_EMAIL
    args = {
        'order_uid': uid,
        'error_message': str(error),
        'blank_ids': blank_ids,
    }
    guaranteed_send_email(
        key='office_refund_error_{}_blanks_{}'.format(uid, '_'.join(blank_ids)),
        to_email=email,
        args=args,
        campaign=settings.OFFICE_REFUND_ERRORS_CAMPAIGN,
        data={'order_uid': uid},
        log_context={'order_uid': uid},
    )


def _is_blanks_refunded(order, partner_refunded_blanks):
    refunded_blanks = get_refunded_or_inprocess_blank_ids(order)
    for blank_id in partner_refunded_blanks:
        if blank_id not in refunded_blanks:
            return False

    return True


def _check_day_is_save(day):
    now = environment.now_aware().astimezone(MSK_TZ)
    today = now.date()
    yesterday = today - timedelta(1)
    if day >= today:
        raise TooEarlyError
    # Нельзя запрашивать возвраты за вчера, если еще не 10 часов утра по Москве
    elif day == yesterday and now.hour < 10:
        raise TooEarlyError


def get_refunded_or_inprocess_blank_ids(order):
    refunded_blanks = []
    for refund in order.iter_refunds():
        if refund.status != RefundStatus.FAILED:
            refunded_blanks += refund.blank_ids

    return refunded_blanks
