# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta, datetime

from django.conf import settings
from six.moves.xmlrpc_client import ServerProxy

from common.data_api.tvm.instance import tvm_factory
from common.data_api.tvm.transport import TvmSafeTransport
from common.celery.task import single_launch_task
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_purchase.core.models import ClientContract, ClientContracts, TrainPartner
from travel.rasp.train_api.train_purchase.utils.billing import BILLING_PARTNERS

log = logging.getLogger(__name__)

define_setting("BALANCE_URL", env={
    Configuration.PRODUCTION: "https://balance-xmlrpc-tvm.paysys.yandex.net:8004/xmlrpctvm",
}, default="https://balance-xmlrpc-tvm-ts.paysys.yandex.net:8004/xmlrpctvm")

BALANCE_DELETE_CONTRACTS_AFTER = timedelta(days=1)


def get_client_contracts(now, client_id):
    transport = TvmSafeTransport(tvm_factory.get_provider(), settings.TVM_BALANCE)
    server = ServerProxy(settings.BALANCE_URL, transport)
    return server.Balance.GetClientContracts({'ClientID': client_id, 'Dt': now})


@single_launch_task()
def update_billing_client_contracts():
    now = environment.now()
    for partner in TrainPartner:
        update_contracts(now, partner)


def update_contracts(now, partner):
    client_id = BILLING_PARTNERS[partner].id
    client_contract_list = get_client_contracts(now, client_id)
    if not client_contract_list:
        log.error('Не удалось получить свежую информацию о договорах с %s из биллинга', partner.value)
        return

    contracts = []
    for client_contract in client_contract_list:
        contracts.append(ClientContract(
            start_dt=_dt_from_xmlrps_dt(client_contract['DT']),
            finish_dt=_dt_from_xmlrps_dt(client_contract.get('FINISH_DT')),
            is_active=client_contract['IS_ACTIVE'],
            is_cancelled=client_contract['IS_CANCELLED'],
            is_suspended=client_contract['IS_SUSPENDED'],
            partner_commission_sum=client_contract['PARTNER_COMMISSION_SUM'],
            partner_commission_sum2=client_contract['PARTNER_COMMISSION_SUM2'],
        ))
    ClientContracts.objects.create(updated_at=now, contracts=contracts, partner=partner)
    ClientContracts.objects(updated_at__lt=now - BALANCE_DELETE_CONTRACTS_AFTER).delete()
    log.info('Успешно обновили контракты из биллинга для %s', partner.value)


def _dt_from_xmlrps_dt(xmlrps_dt):
    if xmlrps_dt is None:
        return None
    return datetime(*xmlrps_dt.timetuple()[:6])
