# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from mongoengine import Q
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.data_api.billing.trust_client import TrustClientException, TrustClient, TrustPaymentStatuses
from travel.rasp.library.python.common23.date import environment
from common.utils.request_limiter import RequestLimiter
from travel.rasp.train_api.train_purchase.core.models import Payment

log = logging.getLogger(__name__)

request_limiter = RequestLimiter(max_rps=5)


@single_launch_task()
def fill_billing_status_and_response(trust_client=None):
    yesterday_utc = environment.now_utc() - timedelta(days=1)
    day_before_yesterday_utc = environment.now_utc() - timedelta(days=2)
    two_days_before_yesterday_utc = environment.now_utc() - timedelta(days=3)
    if trust_client is None:
        trust_client = TrustClient()
    # берем незавершенные платежи, которые уже точно не в обработке
    fill_status_and_response(trust_client, begin_utc_naive=two_days_before_yesterday_utc,
                             end_utc_naive=yesterday_utc, updated_at_boundary_utc_naive=day_before_yesterday_utc,
                             ignored_statuses=[TrustPaymentStatuses.CLEARED, TrustPaymentStatuses.AUTHORIZED])


@request_limiter.throttle
def _get_payment_info(purchase_token, trust_client):
    return trust_client.get_payment_info(purchase_token)


def fill_status_and_response(trust_client, begin_utc_naive=None, end_utc_naive=None,
                             updated_at_boundary_utc_naive=None, ignored_statuses=None):
    """
    Функция нужна для проставления статусов из консоли.

    :param trust_client: клиент для TRUST
    :param begin_utc_naive: время создания платежа, начиная с которого обновлять заказы (включительно)
    :param end_utc_naive: время создания платежа, после которого не надо обновлять заказы (исключительно)
    :param updated_at_boundary_utc_naive: не обновлять платежи, которые обновили позже этого времени
    """
    yesterday_utc = environment.now_utc() - timedelta(days=1)
    if end_utc_naive is None:
        end_utc_naive = yesterday_utc  # обрабатываем платежи, которые уже точно не в обработке
    payment_qs = Payment.objects.filter(trust_created_at__lt=end_utc_naive)
    if begin_utc_naive is not None:
        payment_qs = payment_qs.filter(trust_created_at__gte=begin_utc_naive)
    if updated_at_boundary_utc_naive:
        payment_qs = payment_qs.filter(
            Q(updated_from_billing_at__lt=updated_at_boundary_utc_naive)
            | Q(updated_from_billing_at__exists=False)  # если еще ни разу не обновляли, то надо обновить
        )
    if ignored_statuses:
        payment_qs = payment_qs.filter(status__nin=[s.value for s in ignored_statuses])
    for payment in payment_qs:
        with log_context(order_uid=payment.order_uid):
            try:
                update_spec = _build_payment_update_spec(payment, trust_client)
                update_spec['set__updated_from_billing_at'] = environment.now_utc()
                log.info('Обновляем информацию о статусе платежа %r', update_spec)
                update_result = Payment.objects.filter(uid=payment.uid).update_one(**update_spec)
                if not update_result:
                    log.error('Не смогли обновить информацию о статусе платежа.')
            except TrustClientException:
                log.exception('Ошибка при получении информации из биллинга.')
            except Exception:
                log.exception('Ошибка при обновлении платежных данных.')


def _build_payment_update_spec(billing_payment, trust_client):
    update_spec = {}
    if billing_payment.purchase_token:
        payment_info = _get_payment_info(billing_payment.purchase_token, trust_client)
        if (
            (payment_info.status.value != billing_payment.status)
            or (payment_info.resp_code != billing_payment.resp_code)
            or (payment_info.resp_desc != billing_payment.resp_desc)
        ):
            update_spec['set__status'] = payment_info.status.value
            update_spec['set__resp_code'] = payment_info.resp_code
            update_spec['set__resp_desc'] = payment_info.resp_desc
    return update_spec
