# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from datetime import timedelta

from django.conf import settings
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.data_api.billing.trust_client import TrustClient, TrustClientException
from common.data_api.sendr.api import Campaign, Attachment
from common.dynamic_settings.default import conf
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now_utc
from travel.rasp.train_api.train_purchase.core.models import RefundPayment, RefundPaymentStatus
from travel.rasp.train_api.train_purchase.workflow.payment.refund_payment import REFUND_FAILED_MESSAGE, REFUND_UNKNOWN_MESSAGE

log = logging.getLogger(__name__)

MAX_NUMBER_OF_FAILED_PAYMENT_REFUND_TO_PROCESS = 500

MAX_NUMBER_OF_TRULY_UNKNOWN_REFUND_PAYMENT_TO_PROCESS = 500
MAX_INTERVAL_FOR_RETRIEVE_UNKNOWN_PAYMENTS = timedelta(days=1)

MAX_NUMBER_OF_TRY_LATER_REFUND_PAYMENT_TO_PROCESS = 500
MAX_INTERVAL_FOR_RETRY_PAYMENTS = timedelta(days=1)

define_setting('REFUND_ERRORS_CAMPAIGN', {
    Configuration.PRODUCTION: '4VD6MWK2-UG3',
    Configuration.TESTING: 'O601ITK2-L5G1',
    Configuration.DEVELOPMENT: 'O601ITK2-L5G1'
}, default=None)
campaign = Campaign.create_rasp_campaign(settings.REFUND_ERRORS_CAMPAIGN)


@single_launch_task()
def send_failed_refund_payment_email():
    if not conf.TRAIN_PURCHASE_ERRORS_EMAIL:
        log.error('Не задана настройка TRAIN_PURCHASE_ERRORS_EMAIL.')
        return

    trust_client = TrustClient()
    send_bunch(trust_client, get_failed_refund_payments, REFUND_FAILED_MESSAGE)
    send_bunch(trust_client, get_expired_unknown_refund_payments, REFUND_UNKNOWN_MESSAGE)
    send_bunch(trust_client, get_expired_try_later_refund_payments, REFUND_FAILED_MESSAGE)


def send_bunch(trust_client, get_pending_refund_payments_function, error_message):
    for pending_refund_payment in get_pending_refund_payments_function():
        with log_context(order_uid=pending_refund_payment.order_uid):
            log.info('Отправляем сообщение о проваленом возврате денег пользователю для возврата %s',
                     pending_refund_payment.refund_uuid)
            try:
                send_email(trust_client, pending_refund_payment, error_message)
            except Exception:
                log.exception('Ошибка при отправке сообщения о проваленом возврате денег для возврата %s',
                              pending_refund_payment.refund_uuid)


def send_email(trust_client, pending_refund_payment, message):
    trust_payment_info = get_trust_payment_info(trust_client, pending_refund_payment.purchase_token)
    trust_refund_info = get_trust_refund_info(trust_client, pending_refund_payment.trust_refund_id)

    send_problem_refund(
        pending_refund_payment=pending_refund_payment,
        trust_refund_info=trust_refund_info,
        trust_payment_info=trust_payment_info,
        message=message
    )

    pending_refund_payment.is_email_sent = True
    pending_refund_payment.refund_payment_status = RefundPaymentStatus.FAILED
    pending_refund_payment.save()


def get_failed_refund_payments():
    return RefundPayment.objects.filter(
        refund_payment_status=RefundPaymentStatus.FAILED,
        is_email_sent=False
    ).order_by('-refund__created_at')[:MAX_NUMBER_OF_FAILED_PAYMENT_REFUND_TO_PROCESS]


def get_expired_unknown_refund_payments():
    return RefundPayment.objects.filter(
        refund_payment_status=RefundPaymentStatus.UNKNOWN,
        refund_created_at__lte=now_utc() - MAX_INTERVAL_FOR_RETRIEVE_UNKNOWN_PAYMENTS,
        is_email_sent=False
    ).order_by('-refund__created_at')[:MAX_NUMBER_OF_FAILED_PAYMENT_REFUND_TO_PROCESS]


def get_expired_try_later_refund_payments():
    return RefundPayment.objects.filter(
        refund_payment_status__in=[RefundPaymentStatus.TRY_LATER, RefundPaymentStatus.NEW],
        refund_created_at__lte=now_utc() - MAX_INTERVAL_FOR_RETRY_PAYMENTS,
        is_email_sent=False
    ).order_by('-refund__created_at')[:MAX_NUMBER_OF_TRY_LATER_REFUND_PAYMENT_TO_PROCESS]


def get_trust_payment_info(trust_client, purchase_token):
    try:
        trust_payment_info = trust_client.get_raw_payment_info(purchase_token)
    except TrustClientException:
        return None
    else:
        return json.dumps(trust_payment_info, indent=2, ensure_ascii=False, sort_keys=True).encode('utf-8')


def get_trust_refund_info(trust_client, trust_refund_id):
    if not trust_refund_id:
        return None

    try:
        trust_refund_info = trust_client.get_refund_info(trust_refund_id)
    except TrustClientException:
        return None
    else:
        return json.dumps(trust_refund_info, indent=2, ensure_ascii=False, sort_keys=True).encode('utf-8')


def send_problem_refund(pending_refund_payment, trust_refund_info, trust_payment_info, message):
    email = conf.TRAIN_PURCHASE_ERRORS_EMAIL
    args = _make_problem_refund_args(pending_refund_payment, message)
    attachments = _make_problem_refund_attachments(trust_refund_info, trust_payment_info)
    campaign.send(email, args=args, attachments=attachments)


def _make_problem_refund_args(pending_refund_payment, message):
    return {
        'order_uid': pending_refund_payment.order_uid,
        'refund_uuid': pending_refund_payment.refund_uuid,
        'blank_ids': pending_refund_payment.refund_blank_ids,
        'error_message': message,
        'trust_refund_id': pending_refund_payment.trust_refund_id,
        'purchase_token': pending_refund_payment.purchase_token,
        'refund_payment_status': pending_refund_payment.refund_payment_status,
    }


def _make_problem_refund_attachments(trust_refund_info, trust_payment_info):
    attachments = []
    if trust_refund_info:
        attachments.append(Attachment(filename='trust_refund_info.json', mime_type='application/json',
                                      content=trust_refund_info))
    if trust_payment_info:
        attachments.append(Attachment(filename='trust_payment_info.json', mime_type='application/json',
                                      content=trust_payment_info))
    return attachments
