# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from ylog.context import log_context

from common.celery.task import single_launch_task
from common.data_api.billing.trust_client import TrustClient, TrustClientException, TrustRefundStatuses
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_purchase.core.models import RefundPaymentStatus, RefundPayment

log = logging.getLogger(__name__)

MAX_NUMBER_OF_UNKNOWN_REFUND_TO_PROCESS = 500


@single_launch_task()
def update_refund_payment_status():
    trust_client = TrustClient()
    for pending_refund_payment in get_top_pending_refund_payments():
        with log_context(order_uid=pending_refund_payment.order_uid):
            try:
                if is_refund_payment_done(trust_client, pending_refund_payment.trust_refund_id):
                    pending_refund_payment.refund_payment_status = RefundPaymentStatus.DONE
                    pending_refund_payment.refund_payment_finished_at = environment.now_utc()
                    pending_refund_payment.save()
            except TrustClientException:
                log.error('Не смогли обновить статусы возврата денег для заказа %s и возврата %s',
                          pending_refund_payment.order_uid, pending_refund_payment.refund_uuid)
    for unknown_resize in get_top_unknown_payments_resized():
        with log_context(order_uid=unknown_resize.order_uid):
            try:
                pay_info = trust_client.get_payment_info(unknown_resize.purchase_token)
                if pay_info.reversal_id:
                    unknown_resize.refund_payment_status = RefundPaymentStatus.DONE
                    unknown_resize.trust_reversal_id = pay_info.reversal_id
                    unknown_resize.save()
            except TrustClientException:
                log.error('Не смогли обновить reversal_id для заказа %s и возврата %s',
                          unknown_resize.order_uid, unknown_resize.refund_uuid)


def is_refund_payment_done(trust_client, trust_refund_id):
    return trust_client.get_refund_status(trust_refund_id) == TrustRefundStatuses.SUCCESS


def get_top_pending_refund_payments():
    return RefundPayment.objects.filter(
        refund_payment_status=RefundPaymentStatus.UNKNOWN,
        trust_refund_id__exists=True,
        is_email_sent=False,
        payment_resized__ne=True,
    ).order_by('-refund_created_at')[:MAX_NUMBER_OF_UNKNOWN_REFUND_TO_PROCESS]


def get_top_unknown_payments_resized():
    return RefundPayment.objects.filter(
        refund_payment_status=RefundPaymentStatus.UNKNOWN,
        is_email_sent=False,
        payment_resized=True,
    ).order_by('-refund_created_at')[:MAX_NUMBER_OF_UNKNOWN_REFUND_TO_PROCESS]
