# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from django.conf import settings
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_purchase.core.models import Payment, TrainOrder

log = logging.getLogger(__name__)

LONG_EXTERNAL_EVENT_CHECK_RANGE = timedelta(days=7)
LONG_EXTERNAL_EVENT_TIMEOUT = timedelta(seconds=300)
MAX_ORDER_NUMBER_TO_PROCESS = 100


define_setting('LONG_EXTERNAL_EVENT_CAMPAIGN', {
    Configuration.PRODUCTION: 'GI31O6Z2-J022',
    Configuration.TESTING: '4FDMN6Z2-0YV',
    Configuration.DEVELOPMENT: '4FDMN6Z2-0YV'
}, default=None)


@single_launch_task()
def send_long_external_event_email():
    check_start_dt = environment.now_utc() - LONG_EXTERNAL_EVENT_CHECK_RANGE
    timeout_dt = environment.now_utc() - LONG_EXTERNAL_EVENT_TIMEOUT

    order_failures_qs = TrainOrder.objects.filter(
        __raw__={
            'reserved_to': {'$gt': check_start_dt},
            'process.received_event.is_external': True,
            'process.created_at': {'$lt': timeout_dt},
            'email_flags.problem_order_email_is_sent': {'$ne': True},
        },
    ).order_by('-reserved_to')[:MAX_ORDER_NUMBER_TO_PROCESS]
    log.info('Найдено %s заказов с устаревшими внешними событиями', order_failures_qs.count())

    order_failures_with_process_name = {o: 'booking' for o in order_failures_qs}

    payment_failures_qs = list(Payment.objects.aggregate(*[
        {'$match': {
            'trust_created_at': {'$gt': check_start_dt},
            'process.created_at': {'$lt': timeout_dt},
            'process.received_event.is_external': True,
        }},
        {'$lookup': {'from': 'train_order', 'localField': 'order_uid', 'foreignField': 'uid', 'as': 'orders'}},
        {'$match': {'orders.email_flags.problem_order_email_is_sent': {'$ne': True}}},
        {'$sort': {'trust_created_at': -1}},
        {'$limit': MAX_ORDER_NUMBER_TO_PROCESS},
        {'$project': {'order_uid': 1}},
    ]))
    log.info('Найдено %s платежей с устаревшими внешними событиями', len(payment_failures_qs))

    order_failures_in_payment_qs = TrainOrder.objects.filter(uid__in=[p['order_uid'] for p in payment_failures_qs])
    order_failures_with_process_name.update({o: 'payment' for o in order_failures_in_payment_qs})

    for order in order_failures_with_process_name:
        with log_context(order_uid=order.uid):
            try:
                guaranteed_send_email(
                    key='send_long_external_event_email_{}'.format(order.uid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={'order_uid': order.uid, 'process': order_failures_with_process_name[order]},
                    campaign=settings.LONG_EXTERNAL_EVENT_CAMPAIGN,
                    log_context={'order_uid': order.uid},
                )
                order.modify(set__email_flags__problem_order_email_is_sent=True)
                log.info('Отправлено письмо о заказе %s с устаревшим внешним событием', order.uid)
            except Exception:
                log.exception('Ошибка при отправке письма о заказе %s с устаревшим внешним событием', order.uid)
