# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from django.conf import settings
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_purchase.core.enums import TravelOrderStatus, OrderStatus
from travel.rasp.train_api.train_purchase.core.models import Payment, TrainOrder

log = logging.getLogger(__name__)

UNHANDLED_EXCEPTION_ORDER_CHECK_RANGE = timedelta(days=7)
MAX_ORDER_NUMBER_TO_PROCESS = 100


define_setting('UNHANDLED_EXCEPTION_ORDER_CAMPAIGN', {
    Configuration.PRODUCTION: 'CLMLR223-PPW1',
    Configuration.TESTING: 'WOO9O223-B221',
    Configuration.DEVELOPMENT: 'WOO9O223-B221'
}, default=None)


@single_launch_task()
def process_unhandled_exception_orders():
    check_start_dt = environment.now_utc() - UNHANDLED_EXCEPTION_ORDER_CHECK_RANGE

    order_failures_qs = TrainOrder.objects.filter(
        process__state='unhandled_exception_state',
        email_flags__problem_order_email_is_sent__ne=True,
        reserved_to__gt=check_start_dt,
    ).order_by('-reserved_to')[:MAX_ORDER_NUMBER_TO_PROCESS]
    log.info('Найдено %s заказов в состоянии unhandled_exception', order_failures_qs.count())

    order_failures_with_process_name = {o: 'booking' for o in order_failures_qs}

    payment_failures_qs = list(Payment.objects.aggregate(*[
        {'$match': {'trust_created_at': {'$gt': check_start_dt}, 'process.state': 'unhandled_exception_state'}},
        {'$lookup': {'from': 'train_order', 'localField': 'order_uid', 'foreignField': 'uid', 'as': 'orders'}},
        {'$match': {'orders.email_flags.problem_order_email_is_sent': {'$ne': True}}},
        {'$sort': {'trust_created_at': -1}},
        {'$limit': MAX_ORDER_NUMBER_TO_PROCESS},
        {'$project': {'order_uid': 1}}
    ]))
    log.info('Найдено %s платежей в состоянии unhandled_exception', len(payment_failures_qs))

    order_failures_in_payment_qs = TrainOrder.objects.filter(uid__in=[p['order_uid'] for p in payment_failures_qs])
    order_failures_with_process_name.update({o: 'payment' for o in order_failures_in_payment_qs})

    for order in order_failures_with_process_name:
        with log_context(order_uid=order.uid):
            try:
                guaranteed_send_email(
                    key='send_unhandled_exception_order_email_{}'.format(order.uid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={
                        'order_uid': order.uid,
                        'process': order_failures_with_process_name[order],
                        'order_status': order.status.value,
                        'payment_status': order.current_billing_payment.status,
                    },
                    campaign=settings.UNHANDLED_EXCEPTION_ORDER_CAMPAIGN,
                    log_context={'order_uid': order.uid},
                )
                modify_spec = dict(set__email_flags__problem_order_email_is_sent=True)
                if order.status != OrderStatus.DONE:
                    modify_spec['set__travel_status'] = TravelOrderStatus.CANCELLED
                order.modify(**modify_spec)
                log.info('Отправлено письмо о заказе %s в состоянии unhandled_exception', order.uid)
            except Exception:
                log.exception('Ошибка при отправке письма о заказе %s в состоянии unhandled_exception', order.uid)
