# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from django.conf import settings
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from travel.rasp.library.python.common23.date import environment
from travel.rasp.train_api.train_purchase.core.enums import TravelOrderStatus
from travel.rasp.train_api.train_purchase.core.models import TrainRefund
from travel.rasp.train_api.train_purchase.tasks.unhandled_exception_order import (
    UNHANDLED_EXCEPTION_ORDER_CHECK_RANGE, MAX_ORDER_NUMBER_TO_PROCESS
)

log = logging.getLogger(__name__)


@single_launch_task()
def process_unhandled_exception_refund():
    check_start_dt = environment.now_utc() - UNHANDLED_EXCEPTION_ORDER_CHECK_RANGE

    refund_failures_qs = TrainRefund.objects.filter(
        process__state='unhandled_exception_state',
        problem_refund_email_is_sent__ne=True,
        created_at__gt=check_start_dt,
    ).order_by('-created_at')[:MAX_ORDER_NUMBER_TO_PROCESS]
    log.info('Найдено %s возвратов в состоянии unhandled_exception', refund_failures_qs.count())

    for refund in refund_failures_qs:
        with log_context(order_uid=refund.order_uid):
            try:
                order = refund.order
                guaranteed_send_email(
                    key='send_unhandled_exception_refund_email_{}'.format(refund.uuid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={
                        'order_uid': order.uid,
                        'process': 'refund',
                        'order_status': order.status.value,
                        'payment_status': order.current_billing_payment.status,
                        'refund_status': refund.status,
                    },
                    campaign=settings.UNHANDLED_EXCEPTION_ORDER_CAMPAIGN,
                    log_context={'order_uid': order.uid},
                )
                refund.modify(set__problem_refund_email_is_sent=True)
                order.modify(set__travel_status=TravelOrderStatus.DONE)
                log.info('Отправлено письмо о возврате заказа %s в состоянии unhandled_exception', order.uid)
            except Exception:
                log.exception('Ошибка при отправке письма о возврате заказа %s в состоянии unhandled_exception',
                              refund.order_uid)
