# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from ylog.context import log_context

from common.celery.task import single_launch_task
from common.data_api.billing.trust_client import TrustClient, TrustPaymentStatuses
from travel.rasp.library.python.common23.date.environment import now_aware
from common.utils.try_hard import try_hard
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder
from travel.rasp.train_api.train_purchase.utils.order import make_ticket_statuses_update, make_order_status_update

log = logging.getLogger(__name__)

# максимальная задержка обновления статуса в биллинге (https://st.yandex-team.ru/RASPFRONT-4566#1516208203000)
BILLING_CHECK_DELAY = timedelta(hours=1)

# период обрабатываемых заказов
ORDER_FILTER_INTERVAL = timedelta(minutes=30)


@single_launch_task()
def unhold_payments(interval=ORDER_FILTER_INTERVAL):
    end_dt = now_aware() - BILLING_CHECK_DELAY
    trust_client = TrustClient()

    order_qs = TrainOrder.objects.filter(reserved_to__gt=end_dt - interval, reserved_to__lt=end_dt,
                                         invalid_payments_unholded__ne=True)
    for order in order_qs:
        try:
            update_spec = unhold_order_payments(trust_client, order)
            update_spec.update(set__invalid_payments_unholded=True)
            order.modify(**update_spec)
        except Exception:
            try:
                log.exception('error in unholding')
                order.modify(set__invalid_payments_unholded=False)
            except Exception:
                pass


def unhold_order_payments(trust_client, order):
    with log_context(order_uid=order.uid):
        update_spec = {}
        if not [o for o in order.payments if o.purchase_token]:
            return update_spec
        order_num = order.current_partner_data.order_num
        if not order_num:
            order_info = try_hard()(get_order_info)(order)
            if order_info.status == OperationStatus.OK:
                update_spec.update(make_ticket_statuses_update(order, order_info.tickets))
                update_spec.update(make_order_status_update(order, order_info))
                order_num = order_info.order_num
        log.info('order_num: %s', order_num)
        for i, payment in enumerate(order.payments):
            if i == 0 and order_num is not None:
                continue
            if not payment.purchase_token:
                continue
            log.info('checking payment status of %s...', payment.purchase_token)
            status = trust_client.get_payment_status(payment.purchase_token)

            if status == TrustPaymentStatuses.AUTHORIZED:
                log.info('unholding authorized payment')
                trust_client.unhold_payment(payment.purchase_token)
                payment.update(set__status=TrustPaymentStatuses.CANCELED.value)
            elif payment.status != status:
                payment.update(set__status=status.value)
                log.info('payment %s status updated to %s', payment.purchase_token, status)
            else:
                log.info('payment with status %s does not need unholding', status)
        return update_spec
