# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import wraps

from django.utils.decorators import method_decorator
from rest_framework import status
from rest_framework.decorators import detail_route
from rest_framework.response import Response
from ylog.context import log_context

from common.workflow.registry import get_process
from travel.rasp.train_api.helpers.error import ErrorType, render_error
from travel.rasp.train_api.train_purchase.core.models import ErrorInfo, TrainOrder
from travel.rasp.train_api.train_purchase.workflow.booking import TRAIN_BOOKING_PROCESS
from travel.rasp.train_api.train_purchase.workflow.payment import PAYMENT_PROCESS

log = logging.getLogger(__name__)


def order_view(log_call=True):
    def decorator(view_func):
        @wraps(view_func)
        def wrapper(request, uid):
            try:
                order = TrainOrder.objects.get(uid=uid)
            except TrainOrder.DoesNotExist:
                return Response({
                    'errors': {'uid': 'Order was not found'},
                }, status=status.HTTP_404_NOT_FOUND)

            booking_process = get_process(TRAIN_BOOKING_PROCESS, order)
            payment_process = get_process(PAYMENT_PROCESS, order.current_billing_payment)
            if (booking_process.process and booking_process.process.get('state') == booking_process.EXCEPTION_STATE) \
                    or (payment_process.process
                        and payment_process.process.get('state') == payment_process.EXCEPTION_STATE):
                order_number = None
                if order.current_partner_data and order.current_partner_data.order_num:
                    order_number = order.current_partner_data.order_num
                return render_error(
                    ErrorInfo(type=ErrorType.PROCESS_EXCEPTION_STATE, data={'orderNumber': order_number}),
                    status.HTTP_500_INTERNAL_SERVER_ERROR
                )

            with log_context(order_uid=order.uid):
                if log_call:
                    body = '\nDATA: {}'.format(request.data) if request.data else ''
                    log.info('%s: %s%s', request.method, request.get_full_path(), body)
                return view_func(request, order)

        return wrapper
    return decorator


def order_detail_route(*args, **kwargs):
    log_call = kwargs.pop('log_call', True)
    builtin = kwargs.pop('builtin', False)

    def wrapper(func):
        order_view_func = method_decorator(order_view(log_call))(func)

        if not builtin:
            order_view_func = detail_route(*args, **kwargs)(order_view_func)

        @wraps(order_view_func)
        def wrapped(self, request, pk):
            return order_view_func(self, request, uid=pk)
        return wrapped

    return wrapper
