# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import random
from decimal import Decimal, ROUND_HALF_UP
from hashlib import sha1

from common.dynamic_settings.default import conf
from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.train_api.train_partners.base.train_details.tariff_calculator.tariff_calculator import get_bedding_tariff

TOTAL_FEE = Decimal('0.11')  # наша комиссия от стоимости билета
MINIMAL_AMOUNT_FEE = Decimal('0.06')  # нижняя граница комиссии от стоимости билета без комиссии партнёра


class TicketCost(namedtuple_with_defaults(
    'TicketCost',
    (
        'amount_without_fee',           # Стоимость билета, без наценки
        'bedding_amount_without_fee',   # Стоимость постельного белья, без наценки, входит в amount_without_fee,
                                        # не ноль только для плацкарта
        'yandex_fee_percent',           # Номинальный процент сбора Яндекса
        'main_fee',                     # Сбор Яндекса. Если плацкарт, то сбор за белье сюда не входит
        'bedding_fee',                  # Сбор Яндекса за белье, если это плацкарт
        'is_bandit_fee_applied',        # Применена ли наценка "Бандита"
        'bandit_type',
        'bandit_version',
    ),
    defaults={
        'is_bandit_fee_applied': False,
        'bandit_type': None,
        'bandit_version': None,
    },
)):
    @property
    def full_fee(self):
        return self.main_fee + self.bedding_fee

    @property
    def full_bedding_amount(self):
        return self.bedding_amount_without_fee + self.bedding_fee

    @property
    def full_amount(self):
        return self.amount_without_fee + self.full_fee


def _get_experemental_fee(yandex_uid):
    try:
        delta = float(conf.TRAIN_PURCHASE_EXPERIMENTAL_DELTA_FEE)
        if not delta:
            return TOTAL_FEE
        random.seed(sha1(yandex_uid).digest())
        fee = TOTAL_FEE + Decimal(random.uniform(-delta, delta)).quantize(Decimal('0.0000'), rounding=ROUND_HALF_UP)
        return fee
    except Exception:
        pass
    return TOTAL_FEE


def _get_yandex_fee_percent(yandex_uid=None):
    if yandex_uid:
        return _get_experemental_fee(yandex_uid)
    else:
        return TOTAL_FEE


def _calculate_main_fee_and_bedding_fee(amount_without_bedding, bedding_amount,
                                        partner_commission_sum, yandex_fee_percent):
    """
    :type amount_without_bedding: Decimal
    :type bedding_amount: Decimal
    :type partner_commission_sum: Decimal
    :type yandex_fee_percent: Decimal
    :rtype: (Decimal, Decimal)

    https://st.yandex-team.ru/TRAINS-1021#5c3737a2ec0f81001bc3bda1
    bedding_amount будет не ноль только для плацкарта,
    в этом случае, даже если от белья нельзя отказаться, делаем отдельную наценку на белье
    """
    bedding_fee = bedding_amount * yandex_fee_percent
    main_fee = max(
        amount_without_bedding * yandex_fee_percent,
        amount_without_bedding * MINIMAL_AMOUNT_FEE + partner_commission_sum
    )
    return _quantize_all(main_fee, bedding_fee)


def calculate_ticket_cost(contract, coach_type, amount, service_amount, yandex_uid=None):
    """
    :param contract: common.apps.train_order.models.ClientContract
    :param coach_type: String
    :param amount: Decimal полная сумма
    :param service_amount: Decimal сумма сервиса, включена в amount
    :param yandex_uid: String
    :return: TicketCost
    """
    yandex_fee_percent = _get_yandex_fee_percent(yandex_uid=yandex_uid)
    bedding_amount = get_bedding_tariff(coach_type, service_amount) if amount > service_amount else Decimal(0)

    if not contract:
        return TicketCost(amount, bedding_amount, yandex_fee_percent, Decimal(0), Decimal(0))

    main_fee, bedding_fee = _calculate_main_fee_and_bedding_fee(amount - bedding_amount, bedding_amount,
                                                                contract.partner_commission_sum, yandex_fee_percent)
    return TicketCost(amount, bedding_amount, yandex_fee_percent, main_fee, bedding_fee)


def _quantize_all(*args):
    return [v.quantize(Decimal('0.00'), rounding=ROUND_HALF_UP) for v in args]
