from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from datetime import datetime

from django.conf import settings
from django.core.serializers.json import DjangoJSONEncoder
from django.utils.encoding import smart_text
from six import text_type

from travel.rasp.train_api.helpers.ydb import ydb_provider

YDB_ORDER_LOGS_TABLE_NAME = 'order_logs'


class YdbLogRecord(object):
    __slots__ = (
        'order_uid', 'created_utc', 'name', 'levelname', 'message', 'created', 'process', 'context', 'exception'
    )

    def __init__(self, order_uid, created_utc, name, levelname, message, created, process, context, exception):
        self.order_uid = order_uid
        self.created_utc = created_utc
        self.name = name
        self.levelname = levelname
        self.message = message
        self.created = created
        self.process = process
        self.context = context
        self.exception = exception


class YdbFormatter(logging.Formatter):
    def __init__(self, *args, **kwargs):
        super(YdbFormatter, self).__init__(*args, **kwargs)

    def log_record_to_ydb(self, record):
        message = record.getMessage()
        created = self.formatTime(record, self.datefmt)
        created_utc = datetime.utcfromtimestamp(record.created).isoformat() + 'Z'
        exception = {
            'message': smart_text(record.exc_info[1]),
            'traceback': smart_text(self.formatException(record.exc_info))
        } if record.exc_info is not None else None
        context = {
            k: text_type(v) for k, v in getattr(record, 'context', {}).items()
        }
        return YdbLogRecord(
            context['order_uid'],
            created_utc,
            record.name,
            record.levelname,
            message,
            created,
            record.process,
            json.dumps(context, ensure_ascii=False, cls=DjangoJSONEncoder) if context else None,
            json.dumps(exception, ensure_ascii=False, cls=DjangoJSONEncoder) if exception else None
        )

    def format(self, record):
        raise NotImplementedError


class YdbHandler(logging.Handler):
    def __init__(self, dbalias=settings.YDB_DATABASE, collection_name=YDB_ORDER_LOGS_TABLE_NAME, level=logging.NOTSET):
        super(YdbHandler, self).__init__(level)
        self.dbalias = dbalias
        self.collection_name = collection_name
        self.ydb_formatter = YdbFormatter()

    def insert_one(self, row):
        prepared_query = self.session.prepare("""
DECLARE $order_uid AS "Utf8";
DECLARE $created_utc AS "Utf8";
DECLARE $name AS "Utf8?";
DECLARE $levelname AS "Utf8?";
DECLARE $message AS "Utf8?";
DECLARE $created AS "Utf8?";
DECLARE $process AS "Int32?";
DECLARE $context AS "Json";
DECLARE $exception AS "Json?";

PRAGMA TablePathPrefix("{0}");
INSERT INTO {1} (
    order_uid,
    created_utc,
    name,
    levelname,
    message,
    created,
    process,
    context,
    exception)
VALUES (
    $order_uid,
    $created_utc,
    $name,
    $levelname,
    $message,
    $created,
    $process,
    $context,
    $exception);
""".format(self.dbalias, self.collection_name))

        self.session.transaction().execute(
            prepared_query, {
                '$order_uid': row.order_uid,
                '$created_utc': row.created_utc,
                '$name': row.name,
                '$levelname': row.levelname,
                '$message': row.message,
                '$created': row.created,
                '$process': row.process,
                '$context': row.context,
                '$exception': row.exception,
            },
            commit_tx=True,
        )

    def emit(self, record):
        try:
            self.insert_one(self.format(record))
        except Exception:
            self.handleError(record)

    @property
    def session(self):
        return ydb_provider.get_session()

    def format(self, record):
        ydb_record = self.ydb_formatter.log_record_to_ydb(record)
        return ydb_record
